"use client";

import { useCallback, useEffect, useMemo, useState, type FormEvent } from "react";
import { useRouter } from "next/navigation";
import { type User } from "@supabase/supabase-js";

import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "laem23.01@gmail.com",
  "ehs.mexico@meiwa.com.mx"
]);

const homePageOptions = [
  { value: "/meiwa-core", label: "/meiwa-core" },
];

type ClientPortal = {
  id: string;
  client_name: string;
  description: string | null;
  plan: string;
  client_code: string;
  home_page: string;
  is_active: boolean | null;
  created_at: string;
  updated_at: string;
};

type FormState = {
  client_name: string;
  description: string;
  plan: string;
  client_code: string;
  home_page: string;
};

const emptyForm: FormState = {
  client_name: "",
  description: "",
  plan: "",
  client_code: "",
  home_page: homePageOptions[0]?.value ?? "",
};

export default function ClientAuthLogAdminPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");
  const [portals, setPortals] = useState<ClientPortal[]>([]);
  const [formState, setFormState] = useState<FormState>(emptyForm);
  const [editingPortal, setEditingPortal] = useState<ClientPortal | null>(null);

  const canManage = allowedManagers.has(user?.email ?? "");

  const fetchPortals = useCallback(async () => {
    setLoading(true);
    const { data: { session } } = await supabase.auth.getSession();

    if (!session?.user) {
      router.push("/client-authlog");
      return;
    }

    setUser(session.user);

    if (!allowedManagers.has(session.user.email ?? "")) {
      setLoading(false);
      return;
    }

    const { data, error } = await supabase
      .from("client_portals")
      .select("*")
      .order("created_at", { ascending: false });

    if (error) {
      console.error(error);
      setErrorMessage("No se pudieron cargar los clientes.");
      setPortals([]);
      setLoading(false);
      return;
    }

    setPortals((data ?? []) as ClientPortal[]);
    setLoading(false);
  }, [router]);

  useEffect(() => {
    void fetchPortals();
  }, [fetchPortals]);

  const resetForm = useCallback(() => {
    setFormState(emptyForm);
    setEditingPortal(null);
  }, []);

  const validateForm = useCallback(() => {
    if (!formState.client_name.trim()) {
      return "El nombre del cliente es obligatorio.";
    }
    if (!formState.plan.trim()) {
      return "El plan es obligatorio.";
    }
    if (!formState.client_code.trim()) {
      return "El código de cliente es obligatorio.";
    }
    if (!formState.home_page.trim()) {
      return "La home page es obligatoria.";
    }
    return "";
  }, [formState]);

  const handleSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setErrorMessage("");

    if (!canManage) {
      setErrorMessage("No tienes permisos para administrar clientes.");
      return;
    }

    const validationError = validateForm();
    if (validationError) {
      setErrorMessage(validationError);
      return;
    }

    setSubmitting(true);

    let codeQuery = supabase
      .from("client_portals")
      .select("id")
      .eq("client_code", formState.client_code.trim());

    if (editingPortal?.id) {
      codeQuery = codeQuery.neq("id", editingPortal.id);
    }

    const { data: codeMatches, error: codeError } = await codeQuery;

    if (codeError) {
      console.error(codeError);
      setErrorMessage("No se pudo validar el código de cliente.");
      setSubmitting(false);
      return;
    }

    if ((codeMatches ?? []).length > 0) {
      setErrorMessage("El código de cliente ya está registrado.");
      setSubmitting(false);
      return;
    }

    const payload = {
      client_name: formState.client_name.trim(),
      description: formState.description.trim() || null,
      plan: formState.plan.trim(),
      client_code: formState.client_code.trim(),
      home_page: formState.home_page.trim(),
    };

    if (editingPortal) {
      const { error } = await supabase
        .from("client_portals")
        .update(payload)
        .eq("id", editingPortal.id);

      if (error) {
        console.error(error);
        setErrorMessage("No se pudo actualizar el cliente.");
        setSubmitting(false);
        return;
      }
    } else {
      const { error } = await supabase
        .from("client_portals")
        .insert(payload);

      if (error) {
        console.error(error);
        setErrorMessage("No se pudo registrar el cliente.");
        setSubmitting(false);
        return;
      }
    }

    resetForm();
    await fetchPortals();
    setSubmitting(false);
  };

  const handleEdit = (portal: ClientPortal) => {
    setEditingPortal(portal);
    setFormState({
      client_name: portal.client_name,
      description: portal.description ?? "",
      plan: portal.plan,
      client_code: portal.client_code,
      home_page: portal.home_page,
    });
  };

  const handleDelete = async (portal: ClientPortal) => {
    if (!canManage) {
      setErrorMessage("No tienes permisos para administrar clientes.");
      return;
    }

    const confirmed = window.confirm(
      `¿Seguro que deseas eliminar al cliente ${portal.client_name}? Esta acción no se puede deshacer.`
    );

    if (!confirmed) return;

    const { error } = await supabase
      .from("client_portals")
      .delete()
      .eq("id", portal.id);

    if (error) {
      console.error(error);
      setErrorMessage("No se pudo eliminar el cliente.");
      return;
    }

    if (editingPortal?.id === portal.id) {
      resetForm();
    }

    await fetchPortals();
  };

  const pageStateMessage = useMemo(() => {
    if (loading) return "Cargando clientes...";
    if (!canManage) return "Acceso restringido solo a administradores.";
    if (!portals.length) return "Aún no hay clientes registrados.";
    return "";
  }, [loading, canManage, portals.length]);

  return (
    <div className="min-h-screen bg-gradient-to-b from-yellow-50 via-gray-100 to-gray-200 px-6 py-12 text-gray-900">
      <div className="mx-auto flex w-full max-w-6xl flex-col gap-8">
        <header className="rounded-3xl border border-gray-200 bg-white/90 p-8 shadow-xl">
          <p className="text-sm font-semibold uppercase tracking-[0.3em] text-gray-500">SEKNUM</p>
          <h1 className="mt-3 text-3xl font-bold text-gray-900">Administración de clientes</h1>
          <p className="mt-2 text-base text-gray-600">
            Gestiona los portales y códigos únicos de acceso para cada cliente.
          </p>
        </header>

        <section className="rounded-3xl border border-gray-200 bg-white/90 p-8 shadow-xl">
          <h2 className="text-xl font-semibold text-gray-900">{editingPortal ? "Editar cliente" : "Registrar cliente"}</h2>
          {!canManage && !loading ? (
            <p className="mt-4 text-sm text-gray-500">Acceso restringido solo a administradores.</p>
          ) : (
            <form className="mt-6 grid gap-4 md:grid-cols-2" onSubmit={handleSubmit}>
              <div className="space-y-2">
                <label className="text-sm font-medium text-gray-700" htmlFor="clientName">
                  Nombre de cliente *
                </label>
                <input
                  id="clientName"
                  type="text"
                  value={formState.client_name}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, client_name: event.target.value }))
                  }
                  className="w-full rounded-2xl border border-gray-300 bg-white px-4 py-3 text-sm text-gray-900 shadow-sm focus:border-yellow-400 focus:outline-none focus:ring-2 focus:ring-yellow-200"
                />
              </div>
              <div className="space-y-2">
                <label className="text-sm font-medium text-gray-700" htmlFor="plan">
                  Plan *
                </label>
                <input
                  id="plan"
                  type="text"
                  value={formState.plan}
                  onChange={(event) => setFormState((prev) => ({ ...prev, plan: event.target.value }))}
                  className="w-full rounded-2xl border border-gray-300 bg-white px-4 py-3 text-sm text-gray-900 shadow-sm focus:border-yellow-400 focus:outline-none focus:ring-2 focus:ring-yellow-200"
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <label className="text-sm font-medium text-gray-700" htmlFor="description">
                  Descripción
                </label>
                <textarea
                  id="description"
                  rows={3}
                  value={formState.description}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, description: event.target.value }))
                  }
                  className="w-full resize-none rounded-2xl border border-gray-300 bg-white px-4 py-3 text-sm text-gray-900 shadow-sm focus:border-yellow-400 focus:outline-none focus:ring-2 focus:ring-yellow-200"
                />
              </div>
              <div className="space-y-2">
                <label className="text-sm font-medium text-gray-700" htmlFor="clientCode">
                  Código de cliente *
                </label>
                <input
                  id="clientCode"
                  type="text"
                  value={formState.client_code}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, client_code: event.target.value }))
                  }
                  className="w-full rounded-2xl border border-gray-300 bg-white px-4 py-3 text-sm text-gray-900 shadow-sm focus:border-yellow-400 focus:outline-none focus:ring-2 focus:ring-yellow-200"
                />
              </div>
              <div className="space-y-2">
                <label className="text-sm font-medium text-gray-700" htmlFor="homePage">
                  Home page *
                </label>
                <select
                  id="homePage"
                  value={formState.home_page}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, home_page: event.target.value }))
                  }
                  className="w-full rounded-2xl border border-gray-300 bg-white px-4 py-3 text-sm text-gray-900 shadow-sm focus:border-yellow-400 focus:outline-none focus:ring-2 focus:ring-yellow-200"
                >
                  {homePageOptions.map((option) => (
                    <option key={option.value} value={option.value}>
                      {option.label}
                    </option>
                  ))}
                </select>
              </div>
              {errorMessage ? (
                <div className="md:col-span-2 rounded-2xl border border-red-200 bg-red-50 px-4 py-3 text-sm text-red-600">
                  {errorMessage}
                </div>
              ) : null}
              <div className="flex flex-col gap-3 md:col-span-2 md:flex-row">
                <button
                  type="submit"
                  disabled={submitting}
                  className="inline-flex flex-1 items-center justify-center rounded-full bg-yellow-400 px-6 py-3 text-base font-semibold text-black shadow-lg transition hover:bg-yellow-300 disabled:cursor-not-allowed disabled:opacity-70"
                >
                  {submitting ? "Guardando..." : editingPortal ? "Actualizar cliente" : "Registrar cliente"}
                </button>
                {editingPortal ? (
                  <button
                    type="button"
                    onClick={resetForm}
                    className="inline-flex flex-1 items-center justify-center rounded-full border border-gray-300 bg-white px-6 py-3 text-base font-semibold text-gray-700 shadow-sm transition hover:border-gray-400"
                  >
                    Cancelar
                  </button>
                ) : null}
              </div>
            </form>
          )}
        </section>

        <section className="rounded-3xl border border-gray-200 bg-white/90 p-8 shadow-xl">
          <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
            <h2 className="text-xl font-semibold text-gray-900">Clientes registrados</h2>
            <span className="text-sm text-gray-500">Total: {portals.length}</span>
          </div>

          {pageStateMessage ? (
            <p className="mt-6 text-sm text-gray-500">{pageStateMessage}</p>
          ) : (
            <div className="mt-6 overflow-x-auto">
              <table className="min-w-full border-separate border-spacing-y-3 text-sm">
                <thead>
                  <tr className="text-left text-xs uppercase tracking-[0.2em] text-gray-400">
                    <th className="px-4">Nombre de cliente</th>
                    <th className="px-4">Descripción</th>
                    <th className="px-4">Plan</th>
                    <th className="px-4">Código de cliente</th>
                    <th className="px-4">Home page</th>
                    <th className="px-4">Acciones</th>
                  </tr>
                </thead>
                <tbody>
                  {portals.map((portal) => (
                    <tr
                      key={portal.id}
                      className="rounded-2xl bg-gray-50 text-gray-700 shadow-sm"
                    >
                      <td className="px-4 py-3 font-semibold text-gray-900">{portal.client_name}</td>
                      <td className="px-4 py-3 text-gray-600">
                        {portal.description || "Sin descripción"}
                      </td>
                      <td className="px-4 py-3">{portal.plan}</td>
                      <td className="px-4 py-3 font-medium text-gray-900">{portal.client_code}</td>
                      <td className="px-4 py-3">{portal.home_page}</td>
                      <td className="px-4 py-3">
                        <div className="flex flex-col gap-2 sm:flex-row">
                          <button
                            type="button"
                            onClick={() => handleEdit(portal)}
                            className="inline-flex items-center justify-center rounded-full border border-gray-300 bg-white px-4 py-2 text-xs font-semibold text-gray-700 shadow-sm transition hover:border-gray-400"
                          >
                            Editar
                          </button>
                          <button
                            type="button"
                            onClick={() => handleDelete(portal)}
                            className="inline-flex items-center justify-center rounded-full border border-red-200 bg-red-50 px-4 py-2 text-xs font-semibold text-red-600 shadow-sm transition hover:border-red-300"
                          >
                            Eliminar
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </section>
      </div>
    </div>
  );
}