"use client";

import { useState, type FormEvent } from "react";
import { useRouter } from "next/navigation";

import { resolvePortalHomePath, storeClientPortalAuth } from "@/app/client-portal-auth";
import { supabase } from "@/lib/supabase";

export default function ClientAuthLogPage() {
  const router = useRouter();
  const [clientCode, setClientCode] = useState("");
  const [loading, setLoading] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");

  const handleSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setErrorMessage("");

    const trimmedCode = clientCode.trim();
    if (!trimmedCode) {
      setErrorMessage("Ingresa un código de cliente.");
      return;
    }

    const normalizedCode = trimmedCode.replace(/\s+/g, "");
    const normalizedCodeLower = normalizedCode.toLowerCase();
    const escapedCode = normalizedCode.replace(/[%_]/g, "\\$&");
    setLoading(true);
    const { data, error } = await supabase
      .from("client_portals")
      .select("client_code, home_page")
      .ilike("client_code", `%${escapedCode}%`);

    setLoading(false);

    if (error) {
      console.error(error);
      setErrorMessage("Ocurrió un error al validar el código.");
      return;
    }

    const matchedPortal = (data ?? []).find((portal) =>
      portal.client_code.replace(/\s+/g, "").toLowerCase() === normalizedCodeLower
    );

    if (!matchedPortal) {
      setErrorMessage("Código no válido o no registrado.");
      return;
    }

    const homePage = matchedPortal.home_page;
    if (!homePage) {
      setErrorMessage("El portal aún no está configurado. Contacta al administrador.");
      return;
    }

    storeClientPortalAuth(matchedPortal.client_code, homePage);
    router.push(resolvePortalHomePath(homePage));
  };

  return (
    <div className="flex min-h-screen items-center justify-center bg-gradient-to-b from-yellow-50 via-gray-100 to-gray-200 px-6 py-12 text-gray-900">
      <div className="w-full max-w-lg rounded-3xl border border-gray-200 bg-white/90 p-8 shadow-xl">
        <div className="space-y-3 text-center">
          <p className="text-sm font-semibold uppercase tracking-[0.3em] text-gray-500">SEKNUM</p>
          <h1 className="text-3xl font-bold text-gray-900">Acceso</h1>
          <p className="text-lg text-gray-600">Ingresar a tu portal.</p>
          <p className="text-sm text-gray-500">
            Ingresa el código de cliente proporcionado para acceder a tu portal personalizado.
          </p>
        </div>

        <form className="mt-8 space-y-4" onSubmit={handleSubmit}>
          <div className="space-y-2">
            <label htmlFor="clientCode" className="text-sm font-medium text-gray-700">
              Código de cliente
            </label>
            <input
              id="clientCode"
              type="text"
              value={clientCode}
              onChange={(event) => setClientCode(event.target.value)}
              placeholder="Ej. SEKNUM-001"
              className="w-full rounded-2xl border border-gray-300 bg-white px-4 py-3 text-sm text-gray-900 shadow-sm focus:border-yellow-400 focus:outline-none focus:ring-2 focus:ring-yellow-200"
            />
          </div>

          {errorMessage ? (
            <div className="rounded-2xl border border-red-200 bg-red-50 px-4 py-3 text-sm text-red-600">
              {errorMessage}
            </div>
          ) : null}

          <button
            type="submit"
            disabled={loading}
            className="inline-flex w-full items-center justify-center rounded-full bg-yellow-400 px-6 py-3 text-base font-semibold text-black shadow-lg transition hover:bg-yellow-300 disabled:cursor-not-allowed disabled:opacity-70"
          >
            {loading ? "Validando..." : "Ingresar"}
          </button>
        </form>
      </div>
    </div>
  );
}