"use client";

import { useEffect, useMemo, useState } from 'react';
import { useRouter } from 'next/navigation';
import { User } from '@supabase/supabase-js';
import { format, isBefore, isSameDay, startOfDay, compareAsc } from 'date-fns';
import { es } from 'date-fns/locale';
import { supabase } from '@/lib/supabase';
import { DashboardLayout } from '@/components/dashboard/dashboard-layout';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Calendar as CalendarPicker } from '@/components/ui/calendar';
import { CalendarDays, Clock3, Flag } from 'lucide-react';
import { useUnifiedTasks } from '@/components/tasks/use-unified-tasks';

const priorityColors = {
  low: 'bg-green-100 text-green-800',
  medium: 'bg-yellow-100 text-yellow-800',
  high: 'bg-red-100 text-red-800',
};

const priorityLabels = {
  low: 'Baja',
  medium: 'Media',
  high: 'Alta',
};

export default function CalendarPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loadingAuth, setLoadingAuth] = useState(true);
  const [selectedDate, setSelectedDate] = useState<Date>(new Date());
  const { tasks, loading: tasksLoading, refetch } = useUnifiedTasks(undefined, { skip: !user });  

  useEffect(() => {
    let isMounted = true;

    const loadData = async () => {
      const {
        data: { session },
      } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push('/');
        return;
      }

      if (!isMounted) return;

      setUser(session.user);
      if (isMounted) {
        setLoadingAuth(false);
      }
    };

    loadData();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push('/');
      } else {
        setUser(session.user);
        refetch();
      }
    });

    return () => {
      isMounted = false;
      subscription.unsubscribe();
    };
  }, [refetch, router]);

  const tasksWithDates = useMemo(
    () =>
      tasks
        .filter((task) => task.due_date)
        .map((task) => ({
          ...task,
          dueDate: startOfDay(new Date(task.due_date as string)),
        })),
    [tasks]
  );

  const taskDates = useMemo(
    () => tasksWithDates.map((task) => task.dueDate),
    [tasksWithDates]
  );

  const tasksBySelectedDate = useMemo(
    () =>
      selectedDate
        ? tasksWithDates.filter((task) => isSameDay(task.dueDate, selectedDate))
        : [],
    [selectedDate, tasksWithDates]
  );

  const upcomingTasks = useMemo(() => {
    const today = startOfDay(new Date());

    return tasksWithDates
      .filter((task) => !isBefore(task.dueDate, today) && !task.completed)
      .sort((a, b) => compareAsc(a.dueDate, b.dueDate))
      .slice(0, 5);
  }, [tasksWithDates]);

  if (loadingAuth || tasksLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Calendario</h1>
            <p className="text-gray-600">
              Visualiza tus tareas en el calendario y organiza tu semana.
            </p>
          </div>
        </div>

        <div className="grid grid-cols-1 xl:grid-cols-3 gap-6">
          <Card className="xl:col-span-2">
            <CardHeader className="pb-4">
              <CardTitle className="text-xl">Calendario de tareas</CardTitle>
              <CardDescription>
                Selecciona un día para ver las tareas programadas.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div className="lg:col-span-2">
                  <CalendarPicker
                    mode="single"
                    locale={es}
                    selected={selectedDate}
                    onSelect={(date) => date && setSelectedDate(date)}
                    modifiers={{ scheduled: taskDates }}
                    modifiersClassNames={{
                      scheduled:
                        'border border-blue-600 bg-blue-50 text-blue-700 hover:bg-blue-100',
                    }}
                  />
                  <div className="flex items-center gap-3 text-sm text-gray-600 mt-4">
                    <div className="flex items-center gap-1">
                      <span className="inline-block w-3 h-3 rounded-full bg-blue-500"></span>
                      Días con tareas
                    </div>
                    <div className="flex items-center gap-1">
                      <Clock3 className="h-4 w-4" />
                      Próximas tareas resaltadas en la lista
                    </div>
                  </div>
                </div>
                <div className="space-y-3">
                  <div className="p-4 rounded-lg bg-blue-50 border border-blue-100">
                    <p className="text-sm text-blue-600">Resumen del día</p>
                    <p className="text-lg font-semibold text-gray-900 mt-1">
                      {format(selectedDate, "EEEE d 'de' MMMM", { locale: es })}
                    </p>
                    <p className="text-sm text-gray-600 mt-2">
                      {tasksBySelectedDate.length > 0
                        ? `Tienes ${tasksBySelectedDate.length} tarea${
                            tasksBySelectedDate.length > 1 ? 's' : ''
                          } programada${
                            tasksBySelectedDate.length > 1 ? 's' : ''
                          }`
                        : 'No hay tareas asignadas para este día'}
                    </p>
                  </div>
                  <div className="space-y-2">
                    <p className="text-sm font-medium text-gray-700 flex items-center gap-2">
                      <Flag className="h-4 w-4 text-gray-500" />
                      Prioridades
                    </p>
                    <div className="flex flex-wrap gap-2">
                      <Badge className={priorityColors.high}>Alta</Badge>
                      <Badge className={priorityColors.medium}>Media</Badge>
                      <Badge className={priorityColors.low}>Baja</Badge>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="text-xl">Próximas tareas</CardTitle>
              <CardDescription>
                Las siguientes tareas pendientes en tu calendario.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-3">
              {upcomingTasks.length === 0 ? (
                <div className="text-center text-gray-500 py-6">
                  No hay tareas próximas programadas.
                </div>
              ) : (
                upcomingTasks.map((task) => (
                  <div
                    key={task.id}
                    className="p-4 rounded-lg border bg-white flex items-start gap-3"
                  >
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2">
                        <span className={`text-xs px-2 py-1 rounded-full ${priorityColors[task.priority]}`}>
                          {priorityLabels[task.priority]}
                        </span>
                      </div>
                      <p className="font-semibold text-gray-900 mt-2 truncate">
                        {task.title}
                      </p>
                      <p className="text-sm text-gray-600 line-clamp-2">
                        {task.description || 'Sin descripción'}
                      </p>
                      <div className="flex items-center gap-2 text-sm text-gray-600 mt-2">
                        <CalendarDays className="h-4 w-4" />
                        {format(task.dueDate, "dd 'de' MMMM, yyyy", { locale: es })}
                      </div>
                    </div>
                  </div>
                ))
              )}
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="text-xl">Tareas para el día seleccionado</CardTitle>
            <CardDescription>
              Detalles de las tareas asignadas para la fecha elegida.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-3">
            {tasksBySelectedDate.length === 0 ? (
              <div className="text-center text-gray-500 py-8">
                No tienes tareas para esta fecha.
              </div>
            ) : (
              tasksBySelectedDate.map((task) => (
                <div
                  key={task.id}
                  className="p-4 border rounded-lg bg-white shadow-sm"
                >
                  <div className="flex flex-col md:flex-row md:items-start md:justify-between gap-3">
                    <div className="space-y-2 min-w-0">
                      <div className="flex items-center gap-2">
                        <Badge className={priorityColors[task.priority]}>
                          {priorityLabels[task.priority]}
                        </Badge>
                      </div>
                      <h3 className="text-lg font-semibold text-gray-900 break-words">
                        {task.title}
                      </h3>
                      {task.description && (
                        <p className="text-sm text-gray-600 whitespace-pre-line break-words">
                          {task.description}
                        </p>
                      )}
                    </div>
                    <div className="flex items-center gap-2 text-sm text-gray-600">
                      <CalendarDays className="h-4 w-4" />
                      {format(task.dueDate, "dd 'de' MMMM, yyyy", { locale: es })}
                    </div>
                  </div>
                </div>
              ))
            )}
          </CardContent>
        </Card>
      </div>
    </DashboardLayout>
  );
}
