"use client";

import { useEffect, useMemo, useState } from 'react';
import { useRouter } from 'next/navigation';
import { supabase, type Database } from '@/lib/supabase';
import { User } from '@supabase/supabase-js';
import { DashboardLayout } from '@/components/dashboard/dashboard-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { AlertCircle, BarChart3, CheckCircle, Clock, TrendingUp } from 'lucide-react';
import {
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
  Tooltip,
  Legend,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  AreaChart,
  Area
} from 'recharts';

type PersonalTask = Database['public']['Tables']['tasks']['Row'];
type TeamTask = Database['public']['Tables']['team_tasks']['Row'] & { updated_at?: string | null };

interface UnifiedTask {
  id: string;
  title: string;
  priority: PersonalTask['priority'];
  due_date: string | null;
  created_at: string;
  updated_at: string;
  completed: boolean;
}

interface StatusCard {
  title: string;
  value: number;
  icon: typeof CheckCircle;
  description: string;
  trend?: string;
  color: string;
  bg: string;
}

const PRIORITY_LABELS: Record<PersonalTask['priority'], string> = {
  high: 'Alta',
  medium: 'Media',
  low: 'Baja'
};

const STATUS_COLORS = ['#22c55e', '#fbbf24', '#ef4444'];

export default function StatsPage() {
  const [user, setUser] = useState<User | null>(null);
  const [tasks, setTasks] = useState<UnifiedTask[]>([]);
  const [loading, setLoading] = useState(true);
  const router = useRouter();

  useEffect(() => {
    const fetchData = async () => {
      const { data: { session } } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push('/');
        return;
      }

      setUser(session.user);

      const [personalTasks, assignedTasks] = await Promise.all([
        supabase
          .from('tasks')
          .select('*')
          .eq('user_id', session.user.id)
          .order('due_date', { ascending: true }),
        supabase
          .from('team_tasks')
          .select('*')
          .eq('member_id', session.user.id)
          .order('due_date', { ascending: true })
      ]);

      const mappedPersonal = (personalTasks.data || []).map<UnifiedTask>((task: PersonalTask) => ({
        id: task.id,
        title: task.title,
        priority: task.priority,
        due_date: task.due_date,
        created_at: task.created_at,
        updated_at: task.updated_at,
        completed: task.completed
      }));

      const mappedAssigned = (assignedTasks.data || []).map<UnifiedTask>((task: TeamTask) => ({
        id: task.id,
        title: task.title,
        priority: task.priority,
        due_date: task.due_date,
        created_at: task.created_at,
        updated_at: task.updated_at ?? task.created_at,
        completed: task.status === 'completed'
      }));

      if (personalTasks.error) {
        console.warn('Error fetching personal tasks', personalTasks.error);
      }

      if (assignedTasks.error) {
        console.warn('Error fetching assigned tasks', assignedTasks.error);
      }

      setTasks([...mappedPersonal, ...mappedAssigned]);

      setLoading(false);
    };

    fetchData();
  }, [router]);

  const now = useMemo(() => new Date(), []);

  const stats = useMemo(() => {
    const total = tasks.length;
    const completed = tasks.filter((task) => task.completed).length;
    const pending = tasks.filter((task) => !task.completed).length;
    const overdue = tasks.filter(
      (task) => !task.completed && task.due_date && new Date(task.due_date) < now
    ).length;
    const completionRate = total ? Math.round((completed / total) * 100) : 0;

    return { total, completed, pending, overdue, completionRate };
  }, [tasks, now]);

  const statusCards: StatusCard[] = [
    {
      title: 'Total de tareas',
      value: stats.total,
      icon: BarChart3,
      description: 'Todas las tareas registradas',
      trend: stats.total ? `${stats.completionRate}% completadas` : 'Sin tareas aún',
      color: 'text-blue-600',
      bg: 'bg-blue-50'
    },
    {
      title: 'Completadas',
      value: stats.completed,
      icon: CheckCircle,
      description: 'Tareas finalizadas',
      trend: stats.total ? `${stats.completionRate}% de avance` : 'Agrega tu primera tarea',
      color: 'text-green-600',
      bg: 'bg-green-50'
    },
    {
      title: 'Pendientes',
      value: stats.pending,
      icon: Clock,
      description: 'Tareas activas',
      trend: `${Math.max(stats.pending - stats.overdue, 0)} a tiempo`,
      color: 'text-yellow-600',
      bg: 'bg-yellow-50'
    },
    {
      title: 'Atrasadas',
      value: stats.overdue,
      icon: AlertCircle,
      description: 'Requieren atención',
      trend: stats.overdue ? 'Prioriza estas tareas' : 'Todo al día',
      color: 'text-red-600',
      bg: 'bg-red-50'
    }
  ];

  const statusData = useMemo(
    () => [
      { name: 'Completadas', value: stats.completed },
      { name: 'Pendientes', value: stats.pending },
      { name: 'Atrasadas', value: stats.overdue }
    ],
    [stats.completed, stats.pending, stats.overdue]
  );

  const priorityData = useMemo(() => {
    const priorityCounts = tasks.reduce(
      (acc, task) => {
        acc[task.priority] = (acc[task.priority] || 0) + 1;
        return acc;
      },
      { low: 0, medium: 0, high: 0 } as Record<PersonalTask['priority'], number>
    );

    return (Object.keys(priorityCounts) as PersonalTask['priority'][]).map((priority) => ({
      name: PRIORITY_LABELS[priority],
      value: priorityCounts[priority]
    }));
  }, [tasks]);

  const monthlyActivity = useMemo(() => {
    const months = [...Array(6)].map((_, index) => {
      const date = new Date();
      date.setMonth(date.getMonth() - (5 - index));
      return date;
    });

    return months.map((monthDate) => {
      const month = monthDate.getMonth();
      const year = monthDate.getFullYear();
      const label = monthDate.toLocaleDateString('es-ES', { month: 'short' });

      const created = tasks.filter((task) => {
        const createdAt = new Date(task.created_at);
        return createdAt.getMonth() === month && createdAt.getFullYear() === year;
      }).length;

      const completed = tasks.filter((task) => {
        if (!task.completed) return false;
        const updatedAt = new Date(task.updated_at);
        return updatedAt.getMonth() === month && updatedAt.getFullYear() === year;
      }).length;

      return { label, created, completed };
    });
  }, [tasks]);

  const upcomingTasks = useMemo(() => {
    return tasks
      .filter((task) => !task.completed && task.due_date && new Date(task.due_date) >= now)
      .sort((a, b) => new Date(a.due_date || '').getTime() - new Date(b.due_date || '').getTime())
      .slice(0, 5);
  }, [tasks, now]);

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600" />
      </div>
    );
  }

  if (!user) {
    return null;
  }

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col gap-2">
          <h1 className="text-2xl font-bold text-gray-900">Estadísticas</h1>
          <p className="text-gray-600">
            Revisa el rendimiento de tus tareas y obtén información accionable.
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          {statusCards.map((card) => (
            <Card key={card.title} className="border-0 shadow-sm">
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium text-gray-600">{card.title}</CardTitle>
                <div className={`p-2 rounded-full ${card.bg}`}>
                  <card.icon className={`h-4 w-4 ${card.color}`} />
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-gray-900">{card.value}</div>
                <p className="text-sm text-gray-500">{card.description}</p>
                {card.trend && (
                  <div className="flex items-center gap-2 text-xs text-gray-500 mt-2">
                    <TrendingUp className="h-3 w-3" />
                    <span>{card.trend}</span>
                  </div>
                )}
              </CardContent>
            </Card>
          ))}
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
          <Card className="border-0 shadow-sm lg:col-span-1">
            <CardHeader>
              <CardTitle>Estado de las tareas</CardTitle>
              <p className="text-sm text-gray-600">Distribución de pendientes, atrasadas y completadas</p>
            </CardHeader>
            <CardContent className="h-80">
              {tasks.length === 0 ? (
                <div className="h-full flex items-center justify-center text-gray-500">
                  Aún no hay datos para mostrar
                </div>
              ) : (
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie
                      data={statusData}
                      dataKey="value"
                      nameKey="name"
                      cx="50%"
                      cy="50%"
                      innerRadius={60}
                      outerRadius={90}
                      paddingAngle={4}
                      label
                    >
                      {statusData.map((entry, index) => (
                        <Cell key={`cell-${entry.name}`} fill={STATUS_COLORS[index % STATUS_COLORS.length]} />
                      ))}
                    </Pie>
                    <Tooltip />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              )}
            </CardContent>
          </Card>

          <Card className="border-0 shadow-sm lg:col-span-2">
            <CardHeader>
              <CardTitle>Prioridad de las tareas</CardTitle>
              <p className="text-sm text-gray-600">Balance de carga según la prioridad definida</p>
            </CardHeader>
            <CardContent className="h-80">
              {tasks.length === 0 ? (
                <div className="h-full flex items-center justify-center text-gray-500">
                  Aún no hay datos para mostrar
                </div>
              ) : (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={priorityData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" />
                    <YAxis allowDecimals={false} />
                    <Tooltip />
                    <Bar dataKey="value" fill="#6366f1" radius={[4, 4, 0, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              )}
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
          <Card className="border-0 shadow-sm lg:col-span-2">
            <CardHeader>
              <CardTitle>Tendencia mensual</CardTitle>
              <p className="text-sm text-gray-600">Evolución de tareas creadas y completadas en los últimos 6 meses</p>
            </CardHeader>
            <CardContent className="h-80">
              {tasks.length === 0 ? (
                <div className="h-full flex items-center justify-center text-gray-500">
                  Aún no hay datos para mostrar
                </div>
              ) : (
                <ResponsiveContainer width="100%" height="100%">
                  <AreaChart data={monthlyActivity}>
                    <defs>
                      <linearGradient id="colorCreated" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="5%" stopColor="#3b82f6" stopOpacity={0.35} />
                        <stop offset="95%" stopColor="#3b82f6" stopOpacity={0.05} />
                      </linearGradient>
                      <linearGradient id="colorCompleted" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="5%" stopColor="#22c55e" stopOpacity={0.35} />
                        <stop offset="95%" stopColor="#22c55e" stopOpacity={0.05} />
                      </linearGradient>
                    </defs>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="label" />
                    <YAxis allowDecimals={false} />
                    <Tooltip />
                    <Legend />
                    <Area
                      type="monotone"
                      dataKey="created"
                      stroke="#3b82f6"
                      fillOpacity={1}
                      fill="url(#colorCreated)"
                      name="Creadas"
                    />
                    <Area
                      type="monotone"
                      dataKey="completed"
                      stroke="#22c55e"
                      fillOpacity={1}
                      fill="url(#colorCompleted)"
                      name="Completadas"
                    />
                  </AreaChart>
                </ResponsiveContainer>
              )}
            </CardContent>
          </Card>

          <Card className="border-0 shadow-sm">
            <CardHeader>
              <CardTitle>Próximos vencimientos</CardTitle>
              <p className="text-sm text-gray-600">Prioriza las tareas con fecha cercana</p>
            </CardHeader>
            <CardContent className="space-y-4">
              {upcomingTasks.length === 0 ? (
                <div className="text-gray-500 text-sm">No tienes tareas próximas a vencer</div>
              ) : (
                <div className="space-y-3">
                  {upcomingTasks.map((task) => {
                    const dueDate = task.due_date ? new Date(task.due_date) : null;
                    const formattedDate = dueDate?.toLocaleDateString('es-ES', {
                      day: 'numeric',
                      month: 'short'
                    });

                    return (
                      <div key={task.id} className="flex items-start justify-between gap-3 p-3 rounded-lg bg-gray-50">
                        <div>
                          <p className="font-medium text-gray-900">{task.title}</p>
                        </div>
                        <div className="flex flex-col items-end gap-2">
                          <Badge variant="secondary" className="capitalize">
                            {PRIORITY_LABELS[task.priority]}
                          </Badge>
                          {formattedDate && (
                            <span className="text-xs text-gray-500">{formattedDate}</span>
                          )}
                        </div>
                      </div>
                    );
                  })}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}
