"use client";

import { type JSX, useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { supabase } from '@/lib/supabase';
import { User } from '@supabase/supabase-js';
import { DashboardLayout } from '@/components/dashboard/dashboard-layout';
import { TaskList } from '@/components/tasks/task-list';
import { TaskForm } from '@/components/tasks/task-form';
import { Button } from '@/components/ui/button';
import { Plus, SlidersHorizontal } from 'lucide-react';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { TaskGantt } from '@/components/tasks/task-gantt';
import { useUnifiedTasks } from '@/components/tasks/use-unified-tasks';

export default function TasksPage(): JSX.Element | null {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [showTaskForm, setShowTaskForm] = useState(false);
  const [refreshKey, setRefreshKey] = useState(0);
  const [viewMode, setViewMode] = useState<'gantt' | 'list'>('gantt');
  const { tasks, setTasks, loading: tasksLoading, refetch } = useUnifiedTasks(refreshKey);  
  const router = useRouter();

  useEffect(() => {
    const getSession = async () => {
      const {
        data: { session },
      } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push('/');
        return;
      }

      setUser(session.user);
      setLoading(false);
    };

    getSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push('/');
      } else {
        setUser(session.user);
      }
    });

    return () => subscription.unsubscribe();
  }, [router]);

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Tareas</h1>
            <p className="text-gray-600">
              Visualiza tu carga de trabajo en formato Gantt o lista y gestiona nuevas asignaciones
            </p>
          </div>
          <div className="flex gap-3">
            <Button variant="outline" className="flex items-center gap-2" onClick={() => setViewMode((prev) => (prev === 'gantt' ? 'list' : 'gantt'))}>
              <SlidersHorizontal className="h-4 w-4" />
              {viewMode === 'gantt' ? 'Ver lista' : 'Ver Gantt'}
            </Button>
            <Button onClick={() => setShowTaskForm(true)} className="flex items-center gap-2">
              <Plus className="h-4 w-4" />
              Nueva Tarea
            </Button>
          </div>
        </div>

        <Tabs value={viewMode} onValueChange={(value) => setViewMode(value as 'gantt' | 'list')}>
          <TabsList className="grid w-full max-w-md grid-cols-2">
            <TabsTrigger value="gantt">Cronograma (Gantt)</TabsTrigger>
            <TabsTrigger value="list">Lista</TabsTrigger>
          </TabsList>

          <TabsContent value="gantt" className="mt-4">
            <div className="bg-white rounded-lg shadow-sm border">
              <div className="p-6 border-b">
                <h2 className="text-lg font-semibold text-gray-900">Cronograma de tareas</h2>
                <p className="text-gray-600 text-sm mt-1">
                  Visualiza cómo se distribuyen tus tareas en el tiempo y detecta solapamientos.
                </p>
              </div>
              <div className="p-4">
                <TaskGantt tasks={tasks} loading={tasksLoading} />
              </div>
            </div>
          </TabsContent>

          <TabsContent value="list" className="mt-4">
            <div className="bg-white rounded-lg shadow-sm border">
              <div className="p-6 border-b">
                <h2 className="text-lg font-semibold text-gray-900">Listado de Tareas</h2>
                <p className="text-gray-600 text-sm mt-1">
                  Filtra, busca y actualiza el estado de tus tareas pendientes.
                </p>
              </div>
              <TaskList refreshKey={refreshKey} tasks={tasks} setTasks={setTasks} loading={tasksLoading} />
            </div>
          </TabsContent>
        </Tabs>

        {showTaskForm && (
          <TaskForm
            onClose={() => setShowTaskForm(false)}
            onSuccess={() => {
              setShowTaskForm(false);
              setRefreshKey((prev) => prev + 1);
              refetch();            
            }}
          />
        )}
      </div>
    </DashboardLayout>
  );
}

