"use client";

import { FormEvent, useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { supabase } from "@/lib/supabase";
import { User } from "@supabase/supabase-js";
import { DashboardLayout } from "@/components/dashboard/dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";
import {
  AlertCircle,
  BarChart3,
  Check,
  Edit3,
  Plus,
  Search,
  Trash2,
  UserPlus2,
} from "lucide-react";
import { format } from "date-fns";

type Team = {
  id: string;
  name: string;
  owner_id: string;
  created_at: string;
};

type ProfileSummary = {
  id: string;
  full_name: string | null;
  email: string | null;
  avatar_url: string | null;
};

type TeamMember = {
  id: string;
  member_id: string;
  owner_id: string;
  team_id: string;
  role: string | null;
  status: "pending" | "active" | "removed";
  profile: ProfileSummary | null;
};

type TeamStats = {
  totalMembers: number;
  totalTasks: number;
  completedTasks: number;
  inProgressTasks: number;
  pendingTasks: number;
};

const defaultStats: TeamStats = {
  totalMembers: 0,
  totalTasks: 0,
  completedTasks: 0,
  inProgressTasks: 0,
  pendingTasks: 0,
};

export default function TeamsPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [teams, setTeams] = useState<Team[]>([]);
  const [selectedTeamId, setSelectedTeamId] = useState("");
  const [members, setMembers] = useState<TeamMember[]>([]);
  const [profiles, setProfiles] = useState<ProfileSummary[]>([]);
  const [creatingTeam, setCreatingTeam] = useState(false);
  const [newTeamName, setNewTeamName] = useState("");
  const [editingTeamId, setEditingTeamId] = useState("");
  const [editingTeamName, setEditingTeamName] = useState("");
  const [memberRole, setMemberRole] = useState("Colaborador");
  const [selectedProfileId, setSelectedProfileId] = useState("");
  const [addLoading, setAddLoading] = useState(false);
  const [error, setError] = useState("");
  const [teamStats, setTeamStats] = useState<TeamStats>(defaultStats);
  const [statsLoading, setStatsLoading] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [membersLoading, setMembersLoading] = useState(false);

  const selectedTeam = useMemo(
    () => teams.find((team) => team.id === selectedTeamId) ?? null,
    [selectedTeamId, teams]
  );

  const getOwnerLabel = useCallback(
    (ownerId: string) => {
      if (ownerId === user?.id) {
        return "Tú";
      }

      const ownerProfile = profiles.find((profile) => profile.id === ownerId);

      return ownerProfile?.full_name || ownerProfile?.email || "Otro miembro";
    },
    [profiles, user]
  );

  const isTeamOwner = selectedTeam && user ? selectedTeam.owner_id === user.id : false;

  const filteredTeams = useMemo(() => {
    if (!searchTerm.trim()) return teams;
    return teams.filter((team) =>
      team.name.toLowerCase().includes(searchTerm.trim().toLowerCase())
    );
  }, [searchTerm, teams]);

  const getFriendlyErrorMessage = useCallback((supabaseError: { message?: string } | null, fallback: string) => {
    const message = supabaseError?.message?.toLowerCase() ?? "";

    if (message.includes("networkerror") || message.includes("fetch failed")) {
      return "No pudimos conectar con el servidor. Verifica tu conexión e inténtalo de nuevo.";
    }

    return fallback;
  }, []);

  const fetchTeams = useCallback(async () => {
    if (!user) return;

    const { data, error: teamsError } = await supabase
      .from("teams")
      .select("*")
      .order("created_at", { ascending: true });

    if (teamsError) {
      console.warn("Error fetching teams", teamsError);
      setError(getFriendlyErrorMessage(teamsError, "No pudimos cargar la lista de equipos disponibles."));
      return;
    }

    const sortedTeams = (data ?? []).sort(
      (first, second) => new Date(first.created_at).getTime() - new Date(second.created_at).getTime()
    );

    setTeams(sortedTeams);

    if (sortedTeams.length > 0 && (!selectedTeamId || !sortedTeams.some((team) => team.id === selectedTeamId))) {
      setSelectedTeamId(sortedTeams[0].id);
    }

    if (sortedTeams.length === 0) {
      setSelectedTeamId("");
      setMembers([]);
    }
  }, [getFriendlyErrorMessage, selectedTeamId, user]);

  const fetchProfiles = useCallback(async () => {
    setError("");
    const { data, error: profileError } = await supabase.rpc("list_profiles_for_team");

    if (profileError) {
      console.error("Error fetching profiles", profileError);
      setError("No pudimos cargar la lista de usuarios registrados.");
      return;
    }

    setProfiles(data || []);
  }, []);

  const fetchTeamMembers = useCallback(async () => {
    if (!user || !selectedTeamId) return;

    setMembersLoading(true);
    try {
      const { data, error: fetchError } = await supabase
        .from("team_members")
        .select("id, member_id, owner_id, team_id, role, status")
        .eq("team_id", selectedTeamId)
        .order("created_at", { ascending: false });

      if (fetchError) {
        console.warn("Error fetching team members", fetchError);
        setError(getFriendlyErrorMessage(fetchError, "No pudimos cargar los integrantes de tu equipo."));
        return;
      }

      const formatted: TeamMember[] = (data || []).map((member: any) => {
        const profile = profiles.find((profile) => profile.id === member.member_id) ?? null;

        return {
          id: member.id,
          member_id: member.member_id,
          owner_id: member.owner_id,
          team_id: member.team_id,
          role: member.role,
          status: member.status,
          profile,
        };
      });

      const ownerId = selectedTeam?.owner_id;
      const hasOwner = ownerId ? formatted.some((member) => member.member_id === ownerId) : false;

      if (ownerId && !hasOwner) {
        const ownerProfile = profiles.find((profile) => profile.id === ownerId) ?? null;

        formatted.unshift({
          id: `owner-${selectedTeamId}`,
          member_id: ownerId,
          owner_id: ownerId,
          team_id: selectedTeamId,
          role: "Propietario",
          status: "active",
          profile: ownerProfile,
        });
      }

      setMembers(formatted);
      setTeamStats(defaultStats);
    } catch (cause) {
      console.warn("Error fetching team members", cause);
      setError(
        getFriendlyErrorMessage(
          { message: cause instanceof Error ? cause.message : String(cause) },
          "No pudimos cargar los integrantes de tu equipo."
        )
      );
    } finally {
      setMembersLoading(false);
    }
  }, [getFriendlyErrorMessage, profiles, selectedTeam, selectedTeamId, user]);

  const loadTeamStats = useCallback(async () => {
    if (!selectedTeamId) return;

    const memberIds = members.map((member) => member.member_id);

    setStatsLoading(true);
    try {
      if (memberIds.length === 0) {
        setTeamStats(defaultStats);
        return;
      }

      const { data, error: statsError } = await supabase
        .from("team_tasks")
        .select("status, member_id")
        .eq("team_id", selectedTeamId)        
        .in("member_id", memberIds);

      if (statsError) {
        console.warn("Error fetching stats", statsError);
        setError(getFriendlyErrorMessage(statsError, "No pudimos obtener las estadísticas del equipo."));
        return;
      }

      const totals = (data || []).reduce(
        (acc, task) => {
          acc.totalTasks += 1;
          if (task.status === "completed") acc.completedTasks += 1;
          if (task.status === "in_progress") acc.inProgressTasks += 1;
          if (task.status === "pending") acc.pendingTasks += 1;
          return acc;
        },
        { ...defaultStats, totalMembers: memberIds.length }
      );

      setTeamStats(totals);
    } finally {
      setStatsLoading(false);
    }
  }, [getFriendlyErrorMessage, members, selectedTeamId]);

  useEffect(() => {
    const getSession = async () => {
      const {
        data: { session },
      } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/");
        return;
      }

      setUser(session.user);
      setLoading(false);
    };

    getSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push("/");
      } else {
        setUser(session.user);
      }
    });

    return () => subscription.unsubscribe();
  }, [router]);

  useEffect(() => {
    if (!user) return;

    void fetchTeams();
    void fetchProfiles();
  }, [user, fetchProfiles, fetchTeams]);

  useEffect(() => {
    if (!user || !selectedTeamId) return;

    void fetchTeamMembers();
  }, [fetchTeamMembers, selectedTeamId, user]);

  useEffect(() => {
    setSelectedProfileId("");
    setMemberRole("Colaborador");
    setTeamStats(defaultStats);
  }, [selectedTeamId]);

  useEffect(() => {
    setTeamStats((previous) => ({
      ...previous,
      totalMembers: members.length,
    }));
  }, [members.length]);

  const handleCreateTeam = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setError("");
    setCreatingTeam(true);

    if (!newTeamName.trim()) {
      setError("Ingresa un nombre para el equipo");
      setCreatingTeam(false);
      return;
    }

    const {
      data: { user: currentUser },
    } = await supabase.auth.getUser();

    if (!currentUser) {
      setError("No se pudo validar tu sesión");
      setCreatingTeam(false);
      return;
    }

    const { data: newTeam, error: insertError } = await supabase
      .from("teams")
      .insert({ name: newTeamName.trim(), owner_id: currentUser.id })
      .select()
      .single();

    if (insertError || !newTeam) {
      console.error("Error creating team", insertError);
      setError("No pudimos crear el equipo. Inténtalo nuevamente.");
      setCreatingTeam(false);
      return;
    }

    await supabase.from("team_members").insert({
      owner_id: currentUser.id,
      member_id: currentUser.id,
      role: "Propietario",
      status: "active",
      team_id: newTeam.id,
    });

    setNewTeamName("");
    await fetchTeams();
    setSelectedTeamId(newTeam.id);
    await fetchTeamMembers();
    setCreatingTeam(false);
  };

  const handleUpdateTeam = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setError("");

    if (!editingTeamId) {
      setError("Selecciona un equipo para editar.");
      return;
    }

    if (!editingTeamName.trim()) {
      setError("Ingresa un nombre válido para el equipo.");
      return;
    }

    const { error: updateError } = await supabase
      .from("teams")
      .update({ name: editingTeamName.trim() })
      .eq("id", editingTeamId);

    if (updateError) {
      console.error("Error updating team", updateError);
      setError("No pudimos actualizar el equipo. Inténtalo nuevamente.");
      return;
    }

    setEditingTeamId("");
    setEditingTeamName("");
    await fetchTeams();
  };

  const handleDeleteTeam = async (teamId: string) => {
    setError("");

    const confirmed = window.confirm("¿Deseas eliminar este equipo? Esta acción no se puede deshacer.");
    if (!confirmed) return;

    const { error: membersError } = await supabase.from("team_members").delete().eq("team_id", teamId);
    if (membersError) {
      console.error("Error deleting team members", membersError);
      setError("No pudimos eliminar a los integrantes del equipo.");
      return;
    }

    const { error: deleteError } = await supabase.from("teams").delete().eq("id", teamId);
    if (deleteError) {
      console.error("Error deleting team", deleteError);
      setError("No pudimos eliminar el equipo.");
      return;
    }

    if (selectedTeamId === teamId) {
      setSelectedTeamId("");
      setMembers([]);
    }

    await fetchTeams();
  };

  const handleAddMember = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setError("");
    setAddLoading(true);

    if (!selectedTeamId) {
      setError("Primero crea o selecciona un equipo");
      setAddLoading(false);
      return;
    }

    const isValidTeam = teams.some((team) => team.id === selectedTeamId);

    if (!isValidTeam) {
      setError("Selecciona un equipo válido para agregar integrantes.");
      setAddLoading(false);
      return;
    }

    if (!selectedProfileId) {
      setError("Selecciona un usuario registrado para agregar al equipo");
      setAddLoading(false);
      return;
    }

    const selectedProfile = profiles.find((profile) => profile.id === selectedProfileId);

    if (!selectedProfile) {
      setError("No pudimos encontrar al usuario seleccionado.");
      setAddLoading(false);
      return;
    }

    if (!user) {
      setError("No pudimos validar tu sesión. Inicia sesión nuevamente.");
      setAddLoading(false);
      return;
    }

    const selectedTeamData = teams.find((team) => team.id === selectedTeamId);

    if (selectedTeamData?.owner_id !== user.id) {
      setError("Solo el propietario del equipo puede agregar integrantes.");
      setAddLoading(false);
      return;
    }

    if (!selectedProfile.email) {
      setError("El usuario seleccionado no tiene un correo registrado.");
      setAddLoading(false);
      return;
    }

    const { error: upsertError } = await supabase
      .from("team_members")
      .upsert(
        [
          {
            owner_id: user.id,
            member_id: selectedProfile.id,
            role: memberRole || "Colaborador",
            status: "active",
            team_id: selectedTeamId,
          },
        ],
        { onConflict: "team_id,member_id" }
      );

    if (upsertError) {
      console.error("Error adding member", upsertError);

      const message = upsertError.message?.toLowerCase() ?? "";

      if (message.includes("row-level security")) {
        setError("Tu sesión no tiene permisos para agregar miembros a este equipo.");
      } else if (message.includes("profile_not_found") || upsertError.code === "23503" || message.includes("foreign key")) {
        setError("No encontramos un perfil con ese correo electrónico.");
      } else if (message.includes("not_team_owner")) {
        setError("Solo el propietario del equipo puede agregar integrantes.");
      } else {
        setError("No pudimos agregar al miembro. Inténtalo nuevamente.");
      }

      setAddLoading(false);
      return;
    }

    setSelectedProfileId("");
    setMemberRole("Colaborador");
    await fetchTeamMembers();
    setAddLoading(false);
  };

  const handleRemoveMember = async (memberId: string) => {
    if (!selectedTeamId || !isTeamOwner) return;

    const confirmed = window.confirm("¿Deseas remover este integrante del equipo?");
    if (!confirmed) return;

    const { error: deleteError } = await supabase
      .from("team_members")
      .delete()
      .eq("team_id", selectedTeamId)
      .eq("member_id", memberId);

    if (deleteError) {
      console.error("Error removing member", deleteError);
      setError("No pudimos remover al integrante. Inténtalo nuevamente.");
      return;
    }

    await fetchTeamMembers();
  };

  if (loading) {
    return (
      <DashboardLayout>
        <div className="space-y-6">
          <div className="flex flex-col gap-2">
            <h1 className="text-2xl font-bold text-gray-900">Equipos</h1>
            <p className="text-gray-600">Cargando información del equipo...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  if (!user) return null;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col gap-2">
          <h1 className="text-2xl font-bold text-gray-900">Gestión de equipos</h1>
          <p className="text-gray-600">
            Crea y administra tus equipos sin cargar tareas ni datos adicionales hasta que los necesites.
          </p>
        </div>

        {error && (
          <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg flex items-center gap-2">
            <AlertCircle className="h-5 w-5" />
            <span>{error}</span>
          </div>
        )}

        <div className="grid grid-cols-1 xl:grid-cols-3 gap-6 items-stretch">
          <Card className="h-full min-h-[150px] flex flex-col">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Plus className="h-5 w-5 text-blue-600" />
                Crear nuevo equipo
              </CardTitle>
            </CardHeader>
            <CardContent className="flex-1 flex flex-col">
              <form className="space-y-4 flex-1 flex flex-col" onSubmit={handleCreateTeam}>
                <div className="space-y-2 flex-1">
                  <Label htmlFor="team-name">Nombre del equipo</Label>
                  <Input
                    id="team-name-input"
                    placeholder="Equipo de ventas, Proyecto A..."
                    value={newTeamName}
                    onChange={(event) => setNewTeamName(event.target.value)}
                    required
                  />
                  <p className="text-sm text-gray-500">
                    Cada nuevo equipo se crea vacío y solo carga sus datos cuando lo seleccionas.
                  </p>
                </div>
                <div className="flex justify-end">
                  <Button type="submit" disabled={creatingTeam} className="flex items-center gap-2">
                    <Check className="h-4 w-4" />
                    {creatingTeam ? "Creando..." : "Crear equipo"}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>

          <Card className="xl:col-span-2 h-full min-h-[150px] flex flex-col">
            <CardHeader>
              <div className="flex items-center justify-between gap-2 flex-wrap">
                <div>
                  <CardTitle>Equipos disponibles</CardTitle>
                  <p className="text-sm text-gray-600">
                    Incluye tus equipos creados y los equipos donde fuiste agregado.
                  </p>
                </div>
                <div className="flex items-center gap-2">
                  <Search className="h-4 w-4 text-gray-500" />
                  <Input
                    className="w-52"
                    placeholder="Buscar equipo"
                    value={searchTerm}
                    onChange={(event) => setSearchTerm(event.target.value)}
                  />
                </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-4 flex-1 min-h-0 overflow-hidden">
              {filteredTeams.length === 0 ? (
                <p className="text-gray-600">Crea tu primer equipo para comenzar.</p>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 gap-3 overflow-y-auto pr-2 pb-1 max-h-[150px]">
                  {filteredTeams.map((team) => (
                    <div
                      key={team.id}
                      className={`border rounded-lg p-4 bg-white space-y-3 ${
                        selectedTeamId === team.id ? "border-blue-200 shadow-sm" : ""
                      }`}
                    >
                      <div className="flex items-center justify-between gap-2">
                        <div>
                          <p className="font-semibold text-gray-900">{team.name}</p>
                          <p className="text-sm text-gray-600">
                            Creado el {format(new Date(team.created_at), "dd/MM/yyyy")}
                          </p>
                          <p className="text-xs text-gray-500">
                            Propietario: {getOwnerLabel(team.owner_id)}
                          </p>
                        </div>
                        {selectedTeamId === team.id && <Badge>Seleccionado</Badge>}
                      </div>
                      {editingTeamId === team.id ? (
                        <form className="space-y-3" onSubmit={handleUpdateTeam}>
                          <Input
                            value={editingTeamName}
                            onChange={(event) => setEditingTeamName(event.target.value)}
                            placeholder="Nuevo nombre del equipo"
                          />
                          <div className="flex gap-2 justify-end">
                            <Button type="button" variant="outline" onClick={() => setEditingTeamId("")}>
                              Cancelar
                            </Button>
                            <Button type="submit">Guardar</Button>
                          </div>
                        </form>
                      ) : (
                        <div className="flex items-center justify-end gap-2 flex-wrap">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => setSelectedTeamId(team.id)}
                          >
                            Seleccionar
                          </Button>
                          {team.owner_id === user?.id && (
                            <>
                              <Button variant="secondary" size="sm" onClick={() => {
                                setEditingTeamId(team.id);
                                setEditingTeamName(team.name);
                              }}>
                                <Edit3 className="h-4 w-4 mr-1" /> Editar
                              </Button>
                              <Button variant="destructive" size="sm" onClick={() => handleDeleteTeam(team.id)}>
                                <Trash2 className="h-4 w-4 mr-1" /> Eliminar
                              </Button>
                            </>
                          )}
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 xl:grid-cols-3 gap-6">
          <Card className="xl:col-span-2">
            <CardHeader>
              <CardTitle>Detalle del equipo</CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              {!selectedTeam ? (
                <p className="text-gray-600">Selecciona un equipo para ver sus integrantes.</p>
              ) : (
                <>
                  <div className="flex items-center justify-between gap-2 flex-wrap">
                    <div>
                      <p className="text-sm text-gray-600">Equipo seleccionado</p>
                      <p className="text-xl font-semibold text-gray-900">{selectedTeam.name}</p>
                      <p className="text-xs text-gray-500">
                        Propietario: {getOwnerLabel(selectedTeam.owner_id)}
                      </p>
                    </div>
                    <div className="text-sm text-gray-600">
                      <p>Integrantes cargados: {members.length}</p>
                      {membersLoading && <p className="text-blue-600">Actualizando integrantes...</p>}
                    </div>
                  </div>

                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <Card>
                      <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                          <UserPlus2 className="h-5 w-5 text-blue-600" />
                          Agregar integrante
                        </CardTitle>
                      </CardHeader>
                      <CardContent>
                        <form className="space-y-4" onSubmit={handleAddMember}>
                          <div className="space-y-2">
                            <Label>Selecciona un usuario registrado</Label>
                            <div className="relative">
                              <SelectMember
                                profiles={profiles}
                                selectedProfileId={selectedProfileId}
                                onChange={setSelectedProfileId}
                              />
                            </div>
                          </div>

                          <div className="space-y-2">
                            <Label>Rol dentro del equipo</Label>
                            <Input
                              value={memberRole}
                              onChange={(event) => setMemberRole(event.target.value)}
                              placeholder="Ej. Diseñador"
                            />
                          </div>

                          <div className="flex justify-end">
                            <Button type="submit" disabled={addLoading || !isTeamOwner} className="flex items-center gap-2">
                              <Plus className="h-4 w-4" />
                              {addLoading ? "Agregando..." : "Agregar integrante"}
                            </Button>
                          </div>
                          {!isTeamOwner && (
                            <p className="text-xs text-amber-600">
                              Solo el propietario puede agregar integrantes en este equipo.
                            </p>
                          )}
                        </form>
                      </CardContent>
                    </Card>

                    <Card>
                      <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                          <BarChart3 className="h-5 w-5 text-blue-600" />
                          Estadísticas del equipo
                        </CardTitle>
                      </CardHeader>
                      <CardContent className="space-y-3">
                        <div className="flex items-center justify-between">
                          <p className="text-gray-700">Integrantes</p>
                          <span className="font-semibold text-gray-900">{teamStats.totalMembers}</span>
                        </div>
                        <div className="flex items-center justify-between">
                          <p className="text-gray-700">Tareas totales</p>
                          <span className="font-semibold text-gray-900">{teamStats.totalTasks}</span>
                        </div>
                        <div className="flex items-center justify-between">
                          <p className="text-gray-700">Completadas</p>
                          <span className="font-semibold text-emerald-700">{teamStats.completedTasks}</span>
                        </div>
                        <div className="flex items-center justify-between">
                          <p className="text-gray-700">En progreso</p>
                          <span className="font-semibold text-blue-700">{teamStats.inProgressTasks}</span>
                        </div>
                        <div className="flex items-center justify-between">
                          <p className="text-gray-700">Pendientes</p>
                          <span className="font-semibold text-amber-700">{teamStats.pendingTasks}</span>
                        </div>
                        <div className="flex justify-end">
                          <Button onClick={loadTeamStats} disabled={statsLoading || members.length === 0}>
                            {statsLoading ? "Actualizando..." : "Actualizar estadísticas"}
                          </Button>
                        </div>
                        {members.length === 0 && (
                          <p className="text-xs text-gray-500">
                            Agrega integrantes para activar las métricas. Los conteos se consultan bajo demanda.
                          </p>
                        )}
                      </CardContent>
                    </Card>
                  </div>

                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <h3 className="font-semibold text-gray-900">Integrantes del equipo</h3>
                      <Button size="sm" variant="outline" onClick={fetchTeamMembers}>
                        Actualizar lista
                      </Button>
                    </div>

                    {members.length === 0 ? (
                      <p className="text-gray-600">Todavía no has agregado integrantes a este equipo.</p>
                    ) : (
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        {members.map((member) => (
                          <div key={member.id} className="border rounded-lg p-4 bg-white space-y-3">
                            <div className="flex items-center gap-3">
                              <Avatar>
                                <AvatarImage src={member.profile?.avatar_url || undefined} />
                                <AvatarFallback>
                                  {member.profile?.full_name?.slice(0, 2).toUpperCase() || "EQ"}
                                </AvatarFallback>
                              </Avatar>
                              <div>
                                <p className="font-semibold text-gray-900">
                                  {member.profile?.full_name || member.profile?.email || "Miembro"}
                                </p>
                                <p className="text-sm text-gray-600">{member.role || "Colaborador"}</p>
                              </div>
                            </div>
                            <div className="flex items-center gap-2">
                              <Badge variant="secondary" className="capitalize">
                                {member.status === "active"
                                  ? "Activo"
                                  : member.status === "pending"
                                  ? "Pendiente"
                                  : "Removido"}
                              </Badge>
                              <Badge variant="outline">{member.profile?.email}</Badge>
                            </div>
                            {isTeamOwner && member.member_id !== user.id && (
                              <div className="flex justify-end">
                                <Button
                                  variant="destructive"
                                  size="sm"
                                  onClick={() => handleRemoveMember(member.member_id)}
                                >
                                  <Trash2 className="h-4 w-4 mr-1" /> Remover
                                </Button>
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                </>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}

function SelectMember({
  profiles,
  selectedProfileId,
  onChange,
}: {
  profiles: ProfileSummary[];
  selectedProfileId: string;
  onChange: (value: string) => void;
}) {
  return (
    <select
      className="w-full h-10 rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-blue-500 focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50"
      value={selectedProfileId}
      onChange={(event) => onChange(event.target.value)}
    >
      <option value="" disabled>
        Elige a un integrante
      </option>
      {profiles.map((profile) => (
        <option key={profile.id} value={profile.id}>
          {profile.full_name || profile.email || "Usuario sin nombre"}
        </option>
      ))}
    </select>
  );
}