"use client";

import { useEffect, useState } from "react";
import { PointsDashboardLayout } from "@/components/dashboard/points-dashboard-layout";
import { PointsAdminGuard } from "@/components/points-admin-guard/points-admin-guard";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { toast } from "sonner";
import { fetchActivities } from "@/lib/meiwa-points-data";
import { supabase } from "@/lib/supabase";

const emptyForm = {
  date: "",
  name: "",
  description: "",
  points_value: 0,
};

type Activity = {
  id: string;
  date: string | null;
  name: string;
  description: string | null;
  points_value: number;
};

export default function ActivityMatrixPage() {
  const [activities, setActivities] = useState<Activity[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingActivity, setEditingActivity] = useState<Activity | null>(null);
  const [deleteTarget, setDeleteTarget] = useState<Activity | null>(null);
  const [formData, setFormData] = useState({ ...emptyForm });

  const loadActivities = async () => {
    setLoading(true);
    const { data, error } = await fetchActivities();
    if (error) {
      toast.error("No se pudieron cargar las actividades");
    } else {
      setActivities((data as Activity[]) ?? []);
    }
    setLoading(false);
  };

  useEffect(() => {
    void loadActivities();
  }, []);

  const openCreate = () => {
    setEditingActivity(null);
    setFormData({ ...emptyForm });
    setDialogOpen(true);
  };

  const openEdit = (activity: Activity) => {
    setEditingActivity(activity);
    setFormData({
      date: activity.date ? activity.date.slice(0, 10) : "",
      name: activity.name ?? "",
      description: activity.description ?? "",
      points_value: activity.points_value ?? 0,
    });
    setDialogOpen(true);
  };

  const handleSave = async () => {
    const payload = {
      date: formData.date || null,
      name: formData.name.trim(),
      description: formData.description.trim(),
      points_value: Number(formData.points_value),
    };

    if (!payload.name || payload.points_value <= 0) {
      toast.error("Ingresa un nombre y puntos válidos");
      return;
    }

    const query = editingActivity
      ? supabase.from("mp_activities").update(payload).eq("id", editingActivity.id)
      : supabase.from("mp_activities").insert(payload);

    const { error } = await query;
    if (error) {
      toast.error("No se pudo guardar la actividad");
      return;
    }

    toast.success(editingActivity ? "Actividad actualizada" : "Actividad agregada");
    setDialogOpen(false);
    await loadActivities();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase
      .from("mp_activities")
      .delete()
      .eq("id", deleteTarget.id);
    if (error) {
      toast.error("No se pudo eliminar la actividad");
      return;
    }
    toast.success("Actividad eliminada");
    setDeleteTarget(null);
    await loadActivities();
  };

  return (
    <PointsDashboardLayout>
      <PointsAdminGuard>
        <div className="space-y-6">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Matriz de actividades</h1>
              <p className="text-gray-600">Gestiona el catálogo de actividades.</p>
            </div>
            <Button onClick={openCreate}>Agregar actividad</Button>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Actividades</CardTitle>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-10">
                  <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-600" />
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Fecha</TableHead>
                      <TableHead>Nombre de actividad</TableHead>
                      <TableHead>Descripción</TableHead>
                      <TableHead>Valor de puntos</TableHead>
                      <TableHead className="text-right">Acciones</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {activities.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={5} className="text-center text-gray-500">
                          No hay actividades registradas.
                        </TableCell>
                      </TableRow>
                    )}
                    {activities.map((activity) => (
                      <TableRow key={activity.id}>
                        <TableCell>{activity.date ? activity.date.slice(0, 10) : "-"}</TableCell>
                        <TableCell className="font-medium">{activity.name}</TableCell>
                        <TableCell className="max-w-xs text-sm text-gray-600">
                          {activity.description || "-"}
                        </TableCell>
                        <TableCell>{activity.points_value}</TableCell>
                        <TableCell className="text-right">
                          <div className="flex justify-end gap-2">
                            <Button variant="outline" size="sm" onClick={() => openEdit(activity)}>
                              Editar
                            </Button>
                            <Button
                              variant="destructive"
                              size="sm"
                              onClick={() => setDeleteTarget(activity)}
                            >
                              Eliminar
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </div>

        <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
          <DialogContent className="max-w-xl">
            <DialogHeader>
              <DialogTitle>
                {editingActivity ? "Editar actividad" : "Agregar actividad"}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="activity-date">Fecha</Label>
                <Input
                  id="activity-date"
                  type="date"
                  value={formData.date}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, date: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="activity-name">Nombre</Label>
                <Input
                  id="activity-name"
                  value={formData.name}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, name: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="activity-description">Descripción</Label>
                <Textarea
                  id="activity-description"
                  value={formData.description}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, description: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="activity-points">Valor de puntos</Label>
                <Input
                  id="activity-points"
                  type="number"
                  min={0}
                  value={formData.points_value}
                  onChange={(event) =>
                    setFormData((prev) => ({
                      ...prev,
                      points_value: Number(event.target.value),
                    }))
                  }
                />
              </div>
            </div>
            <DialogFooter className="mt-6">
              <Button variant="outline" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button onClick={handleSave}>
                {editingActivity ? "Guardar cambios" : "Agregar"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <AlertDialog
          open={Boolean(deleteTarget)}
          onOpenChange={() => setDeleteTarget(null)}
        >
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>¿Eliminar actividad?</AlertDialogTitle>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancelar</AlertDialogCancel>
              <AlertDialogAction onClick={handleDelete}>Eliminar</AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </PointsAdminGuard>
    </PointsDashboardLayout>
  );
}