"use client";

import { useEffect, useState } from "react";
import { PointsDashboardLayout } from "@/components/dashboard/points-dashboard-layout";
import { PointsAdminGuard } from "@/components/points-admin-guard/points-admin-guard";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { toast } from "sonner";
import { fetchAllocations, fetchActivities, fetchUsers, getCurrentUser } from "@/lib/meiwa-points-data";
import { supabase } from "@/lib/supabase";

const emptyForm = {
  user_id: "",
  activity_id: "",
  points: 0,
  assigned_at: "",
  comment: "",
};

type Allocation = {
  id: string;
  user_id: string;
  activity_id: string;
  points: number;
  assigned_at: string | null;
  assigned_by: string | null;
  comment: string | null;
  profiles: { full_name: string | null; email: string | null } | null;
  mp_activities: { name: string | null; points_value: number | null } | null;
};

type AllocationRow = Omit<Allocation, "profiles" | "mp_activities"> & {
  profiles:
    | { full_name: string | null; email: string | null }
    | { full_name: string | null; email: string | null }[]
    | null;
  mp_activities:
    | { name: string | null; points_value: number | null }
    | { name: string | null; points_value: number | null }[]
    | null;
};

type Activity = {
  id: string;
  name: string;
  points_value: number;
};

type UserProfile = {
  id: string;
  full_name: string | null;
  email: string | null;
};

export default function AllocationMatrixPage() {
  const [allocations, setAllocations] = useState<Allocation[]>([]);
  const [activities, setActivities] = useState<Activity[]>([]);
  const [users, setUsers] = useState<UserProfile[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingAllocation, setEditingAllocation] = useState<Allocation | null>(null);
  const [deleteTarget, setDeleteTarget] = useState<Allocation | null>(null);
  const [formData, setFormData] = useState({ ...emptyForm });
  const [assignedBy, setAssignedBy] = useState<string | null>(null);

  const loadData = async () => {
    setLoading(true);
    const [{ data: allocationsData }, { data: activitiesData }, { data: usersData }] =
      await Promise.all([fetchAllocations(), fetchActivities(), fetchUsers()]);

    const normalizedAllocations = ((allocationsData ?? []) as AllocationRow[]).map((row) => ({
      ...row,
      profiles: Array.isArray(row.profiles) ? row.profiles[0] ?? null : row.profiles ?? null,
      mp_activities: Array.isArray(row.mp_activities)
        ? row.mp_activities[0] ?? null
        : row.mp_activities ?? null,
    }));
    setAllocations(normalizedAllocations);
    setActivities((activitiesData as Activity[]) ?? []);
    setUsers((usersData as UserProfile[]) ?? []);
    setLoading(false);
  };

  useEffect(() => {
    void loadData();
    const loadUser = async () => {
      const user = await getCurrentUser();
      setAssignedBy(user?.email ?? null);
    };
    void loadUser();
  }, []);

  const openCreate = () => {
    setEditingAllocation(null);
    setFormData({ ...emptyForm });
    setDialogOpen(true);
  };

  const openEdit = (allocation: Allocation) => {
    setEditingAllocation(allocation);
    setFormData({
      user_id: allocation.user_id,
      activity_id: allocation.activity_id,
      points: allocation.points ?? 0,
      assigned_at: allocation.assigned_at ? allocation.assigned_at.slice(0, 10) : "",
      comment: allocation.comment ?? "",
    });
    setDialogOpen(true);
  };

  const handleActivityChange = (activityId: string) => {
    const activity = activities.find((item) => item.id === activityId);
    setFormData((prev) => ({
      ...prev,
      activity_id: activityId,
      points: activity?.points_value ?? 0,
    }));
  };

  const handleSave = async () => {
    if (!formData.user_id || !formData.activity_id) {
      toast.error("Selecciona usuario y actividad");
      return;
    }

    const payload = {
      user_id: formData.user_id,
      activity_id: formData.activity_id,
      points: Number(formData.points),
      assigned_at: formData.assigned_at || new Date().toISOString(),
      assigned_by: assignedBy,
      comment: formData.comment.trim() || null,
    };

    const query = editingAllocation
      ? supabase.from("mp_allocations").update(payload).eq("id", editingAllocation.id)
      : supabase.from("mp_allocations").insert(payload);

    const { error } = await query;
    if (error) {
      toast.error("No se pudo guardar la asignación");
      return;
    }

    toast.success(editingAllocation ? "Asignación actualizada" : "Asignación creada");
    setDialogOpen(false);
    await loadData();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase
      .from("mp_allocations")
      .delete()
      .eq("id", deleteTarget.id);
    if (error) {
      toast.error("No se pudo eliminar la asignación");
      return;
    }
    toast.success("Asignación eliminada");
    setDeleteTarget(null);
    await loadData();
  };

  return (
    <PointsDashboardLayout>
      <PointsAdminGuard>
        <div className="space-y-6">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Matriz de asignación</h1>
              <p className="text-gray-600">
                Asigna puntos a usuarios basados en actividades.
              </p>
            </div>
            <Button onClick={openCreate}>Agregar asignación</Button>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Asignaciones</CardTitle>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-10">
                  <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-600" />
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Usuario</TableHead>
                      <TableHead>Actividad</TableHead>
                      <TableHead>Puntos</TableHead>
                      <TableHead>Fecha de asignación</TableHead>
                      <TableHead>Asignado por</TableHead>
                      <TableHead>Comentario</TableHead>
                      <TableHead className="text-right">Acciones</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {allocations.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center text-gray-500">
                          No hay asignaciones registradas.
                        </TableCell>
                      </TableRow>
                    )}
                    {allocations.map((allocation) => (
                      <TableRow key={allocation.id}>
                        <TableCell>
                          {allocation.profiles?.full_name || allocation.profiles?.email || "-"}
                        </TableCell>
                        <TableCell>{allocation.mp_activities?.name || "-"}</TableCell>
                        <TableCell>{allocation.points}</TableCell>
                        <TableCell>
                          {allocation.assigned_at ? allocation.assigned_at.slice(0, 10) : "-"}
                        </TableCell>
                        <TableCell>{allocation.assigned_by || "-"}</TableCell>
                        <TableCell className="max-w-xs text-sm text-gray-600">
                          {allocation.comment || "-"}
                        </TableCell>
                        <TableCell className="text-right">
                          <div className="flex justify-end gap-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => openEdit(allocation)}
                            >
                              Editar
                            </Button>
                            <Button
                              variant="destructive"
                              size="sm"
                              onClick={() => setDeleteTarget(allocation)}
                            >
                              Eliminar
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </div>

        <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
          <DialogContent className="max-w-xl">
            <DialogHeader>
              <DialogTitle>
                {editingAllocation ? "Editar asignación" : "Agregar asignación"}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label>Usuario</Label>
                <Select
                  value={formData.user_id}
                  onValueChange={(value) =>
                    setFormData((prev) => ({ ...prev, user_id: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona un usuario" />
                  </SelectTrigger>
                  <SelectContent>
                    {users.map((user) => (
                      <SelectItem key={user.id} value={user.id}>
                        {user.full_name || user.email}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Actividad</Label>
                <Select value={formData.activity_id} onValueChange={handleActivityChange}>
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona una actividad" />
                  </SelectTrigger>
                  <SelectContent>
                    {activities.map((activity) => (
                      <SelectItem key={activity.id} value={activity.id}>
                        {activity.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="grid gap-4 sm:grid-cols-2">
                <div className="space-y-2">
                  <Label>Puntos</Label>
                  <Input value={formData.points} disabled />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="assigned-at">Fecha de asignación</Label>
                  <Input
                    id="assigned-at"
                    type="date"
                    value={formData.assigned_at}
                    onChange={(event) =>
                      setFormData((prev) => ({ ...prev, assigned_at: event.target.value }))
                    }
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="comment">Comentario</Label>
                <Textarea
                  id="comment"
                  value={formData.comment}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, comment: event.target.value }))
                  }
                />
              </div>
            </div>
            <DialogFooter className="mt-6">
              <Button variant="outline" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button onClick={handleSave}>
                {editingAllocation ? "Guardar cambios" : "Agregar"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <AlertDialog
          open={Boolean(deleteTarget)}
          onOpenChange={() => setDeleteTarget(null)}
        >
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>¿Eliminar asignación?</AlertDialogTitle>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancelar</AlertDialogCancel>
              <AlertDialogAction onClick={handleDelete}>Eliminar</AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </PointsAdminGuard>
    </PointsDashboardLayout>
  );
}