"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { Gift, TrendingUp, Users, Wallet } from "lucide-react";
import { PointsDashboardLayout } from "@/components/dashboard/points-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  fetchUserAllocations,
  fetchUserPointsSummaryForUser,
  fetchUserRedemptions,
  getCurrentUser,
} from "@/lib/meiwa-points-data";
import type { User } from "@supabase/supabase-js";

const initialListLimit = 6;

type Allocation = {
  id: string;
  user_id: string;
  activity_id: string;
  points: number;
  assigned_at: string | null;
  assigned_by: string | null;
  comment: string | null;
  mp_activities: { name: string | null; description: string | null; date: string | null } | null;
};

type AllocationRow = Omit<Allocation, "mp_activities"> & {
  mp_activities:
    | { name: string | null; description: string | null; date: string | null }
    | { name: string | null; description: string | null; date: string | null }[]
    | null;
};

type Redemption = {
  id: string;
  reward_id: string;
  user_id: string;
  points: number;
  requested_at: string | null;
  status: string | null;
  mp_rewards: { name: string | null; description: string | null } | null;
};

type RedemptionRow = Omit<Redemption, "mp_rewards"> & {
  mp_rewards:
    | { name: string | null; description: string | null }
    | { name: string | null; description: string | null }[]
    | null;
};

const formatDate = (value: string | null) => {
  if (!value) return "-";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "-";
  return new Intl.DateTimeFormat("es-MX", {
    dateStyle: "medium",
  }).format(date);
};

const formatPoints = (value: number) => value.toLocaleString("es-MX");

export default function PointsDashboard() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [pointsEarned, setPointsEarned] = useState(0);
  const [pointsSpent, setPointsSpent] = useState(0);
  const [pointsAvailable, setPointsAvailable] = useState(0);
  const [allocations, setAllocations] = useState<Allocation[]>([]);
  const [redemptions, setRedemptions] = useState<Redemption[]>([]);
  const [showAllAllocations, setShowAllAllocations] = useState(false);
  const [showAllRedemptions, setShowAllRedemptions] = useState(false);

  useEffect(() => {
    const loadDashboard = async () => {
      const sessionUser = await getCurrentUser();
      if (!sessionUser) {
        router.push("/");
        return;
      }

      setUser(sessionUser);

      const [summaryResult, allocationsResult, redemptionsResult] = await Promise.all([
        fetchUserPointsSummaryForUser(sessionUser.id),
        fetchUserAllocations(sessionUser.id),
        fetchUserRedemptions(sessionUser.id),
      ]);

      setPointsEarned(summaryResult.data?.points_earned ?? 0);
      setPointsSpent(summaryResult.data?.points_spent ?? 0);
      setPointsAvailable(summaryResult.data?.points_available ?? 0);

      const normalizedAllocations = ((allocationsResult.data ?? []) as AllocationRow[]).map(
        (row) => ({
          ...row,
          mp_activities: Array.isArray(row.mp_activities)
            ? row.mp_activities[0] ?? null
            : row.mp_activities ?? null,
        })
      );

      const normalizedRedemptions = ((redemptionsResult.data ?? []) as RedemptionRow[]).map(
        (row) => ({
          ...row,
          mp_rewards: Array.isArray(row.mp_rewards) ? row.mp_rewards[0] ?? null : row.mp_rewards ?? null,
        })
      );

      setAllocations(normalizedAllocations);
      setRedemptions(normalizedRedemptions);
      setLoading(false);
    };

    void loadDashboard();
  }, [router]);

  const allocationCount = allocations.length;
  const visibleAllocations = useMemo(() => {
    if (showAllAllocations) return allocations;
    return allocations.slice(0, initialListLimit);
  }, [allocations, showAllAllocations]);

  const visibleRedemptions = useMemo(() => {
    if (showAllRedemptions) return redemptions;
    return redemptions.slice(0, initialListLimit);
  }, [redemptions, showAllRedemptions]);

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600" />
      </div>
    );
  }

  if (!user) {
    return null;
  }

  const stats = [
    {
      title: "Puntos ganados",
      value: formatPoints(pointsEarned),
      helper: (
        <a className="text-sm text-blue-600 hover:underline" href="#activities">
          Ver detalle
        </a>
      ),
      accent: "from-emerald-500 via-emerald-400 to-teal-400",
      icon: TrendingUp,
    },
    {
      title: "Puntos canjeados",
      value: formatPoints(pointsSpent),
      helper: (
        <a className="text-sm text-blue-600 hover:underline" href="#redemptions">
          Ver detalle
        </a>
      ),
      accent: "from-orange-500 via-amber-400 to-yellow-400",
      icon: Gift,
    },
    {
      title: "Puntos disponibles",
      value: formatPoints(pointsAvailable),
      helper: <p className="text-xs text-gray-500">Ganados menos canjeados.</p>,
      accent: "from-indigo-500 via-purple-500 to-pink-500",
      icon: Wallet,
    },
    {
      title: "Participaciones",
      value: formatPoints(allocationCount),
      helper: <p className="text-xs text-gray-500">Asignaciones registradas.</p>,
      accent: "from-sky-500 via-cyan-400 to-blue-500",
      icon: Users,
    },
  ];

  return (
    <PointsDashboardLayout>
      <div className="space-y-10">
        <header className="space-y-6">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">MEIWA POINTS</h1>
              <p className="text-gray-600">
                Bienvenido de vuelta, {user.user_metadata?.full_name || user.email}
              </p>
            </div>
          </div>
          <section className="rounded-3xl bg-gradient-to-br from-indigo-500 via-purple-500 to-pink-400 px-8 py-10 text-white shadow-lg">
            <div className="max-w-2xl space-y-4">
              <p className="text-sm font-semibold uppercase tracking-[0.5em] text-white/90">
                Programa de Gamificación
              </p>
              <p className="text-lg text-white/90">
                Consulta actividades, premios y puntos desde un
                solo lugar.
              </p>
            </div>
          </section>
        </header>

        <section className="grid gap-4 sm:grid-cols-2 xl:grid-cols-4">
          {stats.map((stat) => (
            <Card
              key={stat.title}
              className="relative overflow-hidden border border-slate-200/70 bg-white/85 shadow-sm backdrop-blur"
            >
              <div className={`absolute inset-x-0 top-0 h-1 bg-gradient-to-r ${stat.accent}`} />
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium text-gray-500">{stat.title}</CardTitle>
                <span className={`rounded-full bg-gradient-to-br ${stat.accent} p-2 text-white shadow`}>
                  <stat.icon className="h-4 w-4" />
                </span>
              </CardHeader>
              <CardContent className="space-y-1">
                <div className="text-3xl font-semibold text-gray-900">{stat.value}</div>
                {stat.helper}
              </CardContent>
            </Card>
          ))}
        </section>

        <section id="activities" className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Actividades</h2>
              <p className="text-sm text-gray-600">Historial de asignaciones recibidas.</p>
            </div>
          </div>

          <Card>
            <CardContent className="pt-6">
              {allocations.length === 0 ? (
                <p className="text-sm text-gray-600">Aún no tienes actividades asignadas.</p>
              ) : (
                <div className="space-y-4">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Fecha</TableHead>
                        <TableHead>Actividad</TableHead>
                        <TableHead>Descripción</TableHead>
                        <TableHead className="text-right">Puntos</TableHead>
                        <TableHead>Asignado por</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {visibleAllocations.map((allocation) => (
                        <TableRow key={allocation.id}>
                          <TableCell>{formatDate(allocation.assigned_at)}</TableCell>
                          <TableCell>{allocation.mp_activities?.name ?? "Actividad"}</TableCell>
                          <TableCell className="text-gray-500">
                            {allocation.mp_activities?.description ?? allocation.comment ?? "Sin descripción"}
                          </TableCell>
                          <TableCell className="text-right font-medium">
                            {formatPoints(allocation.points ?? 0)}
                          </TableCell>
                          <TableCell>{allocation.assigned_by ?? "-"}</TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>

                  {allocations.length > initialListLimit && !showAllAllocations && (
                    <div className="flex justify-center">
                      <Button variant="outline" onClick={() => setShowAllAllocations(true)}>
                        Ver más
                      </Button>
                    </div>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </section>

        <section id="redemptions" className="space-y-4">
          <div>
            <h2 className="text-xl font-semibold text-gray-900">Premios canjeados</h2>
            <p className="text-sm text-gray-600">Tus solicitudes de canje registradas.</p>
          </div>

          <Card>
            <CardContent className="pt-6">
              {redemptions.length === 0 ? (
                <p className="text-sm text-gray-600">Aún no has canjeado premios.</p>
              ) : (
                <div className="space-y-4">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Fecha</TableHead>
                        <TableHead>Premio</TableHead>
                        <TableHead>Descripción</TableHead>
                        <TableHead className="text-right">Puntos usados</TableHead>
                        <TableHead>Estado</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {visibleRedemptions.map((redemption) => (
                        <TableRow key={redemption.id}>
                          <TableCell>{formatDate(redemption.requested_at)}</TableCell>
                          <TableCell>{redemption.mp_rewards?.name ?? "Premio"}</TableCell>
                          <TableCell className="text-gray-500">
                            {redemption.mp_rewards?.description ?? "Sin descripción"}
                          </TableCell>
                          <TableCell className="text-right font-medium">
                            {formatPoints(redemption.points ?? 0)}
                          </TableCell>
                          <TableCell className="capitalize">
                            {redemption.status ?? "solicitada"}
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>

                  {redemptions.length > initialListLimit && !showAllRedemptions && (
                    <div className="flex justify-center">
                      <Button variant="outline" onClick={() => setShowAllRedemptions(true)}>
                        Ver más
                      </Button>
                    </div>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </section>
      </div>
    </PointsDashboardLayout>
  );
}