"use client";

import { useEffect, useState } from "react";
import { PointsDashboardLayout } from "@/components/dashboard/points-dashboard-layout";
import { PointsAdminGuard } from "@/components/points-admin-guard/points-admin-guard";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { toast } from "sonner";
import { fetchRewardsMatrix } from "@/lib/meiwa-points-data";
import { supabase } from "@/lib/supabase";

const emptyForm = {
  name: "",
  description: "",
  points_value: 0,
  stock_available: 0,
  image_url: "",
};

type Reward = {
  id: string;
  name: string;
  description: string | null;
  points_value: number;
  stock_available: number;
  image_url: string | null;
  storage_path: string | null;
};

export default function RewardsMatrixPage() {
  const [rewards, setRewards] = useState<Reward[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [deleteTarget, setDeleteTarget] = useState<Reward | null>(null);
  const [editingReward, setEditingReward] = useState<Reward | null>(null);
  const [formData, setFormData] = useState({ ...emptyForm });

  const loadRewards = async () => {
    setLoading(true);
    const { data, error } = await fetchRewardsMatrix();
    if (error) {
      toast.error("No se pudieron cargar los premios");
    } else {
      setRewards((data as Reward[]) ?? []);
    }
    setLoading(false);
  };

  useEffect(() => {
    void loadRewards();
  }, []);

  const openCreate = () => {
    setEditingReward(null);
    setFormData({ ...emptyForm });
    setDialogOpen(true);
  };

  const openEdit = (reward: Reward) => {
    setEditingReward(reward);
    setFormData({
      name: reward.name ?? "",
      description: reward.description ?? "",
      points_value: reward.points_value ?? 0,
      stock_available: reward.stock_available ?? 0,
      image_url: reward.image_url ?? "",
    });
    setDialogOpen(true);
  };

  const handleSave = async () => {
    const payload = {
      name: formData.name.trim(),
      description: formData.description.trim(),
      points_value: Number(formData.points_value),
      stock_available: Number(formData.stock_available),
      image_url: formData.image_url.trim() || null,
    };

    if (!payload.name || payload.points_value <= 0) {
      toast.error("Ingresa un nombre y puntos válidos");
      return;
    }

    const query = editingReward
      ? supabase.from("mp_rewards").update(payload).eq("id", editingReward.id)
      : supabase.from("mp_rewards").insert(payload);

    const { error } = await query;
    if (error) {
      toast.error("No se pudo guardar el premio");
      return;
    }

    toast.success(editingReward ? "Premio actualizado" : "Premio agregado");
    setDialogOpen(false);
    await loadRewards();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase.from("mp_rewards").delete().eq("id", deleteTarget.id);
    if (error) {
      toast.error("No se pudo eliminar el premio");
      return;
    }
    toast.success("Premio eliminado");
    setDeleteTarget(null);
    await loadRewards();
  };

  return (
    <PointsDashboardLayout>
      <PointsAdminGuard>
        <div className="space-y-6">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Matriz de premios</h1>
              <p className="text-gray-600">Administra el catálogo de premios disponibles.</p>
            </div>
            <Button onClick={openCreate}>Agregar premio</Button>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Premios disponibles</CardTitle>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-10">
                  <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-600" />
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Imagen</TableHead>
                      <TableHead>Nombre</TableHead>
                      <TableHead>Descripción</TableHead>
                      <TableHead>Valor de puntos</TableHead>
                      <TableHead>Stock disponible</TableHead>
                      <TableHead>Estado</TableHead>
                      <TableHead className="text-right">Acciones</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {rewards.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center text-gray-500">
                          No hay premios registrados.
                        </TableCell>
                      </TableRow>
                    )}
                    {rewards.map((reward) => (
                      <TableRow key={reward.id}>
                        <TableCell>
                          <div className="h-12 w-12 overflow-hidden rounded-lg bg-gray-100">
                            <img
                              src={reward.image_url ?? "/placeholder.svg"}
                              alt={reward.name}
                              className="h-full w-full object-cover"
                            />
                          </div>
                        </TableCell>
                        <TableCell className="font-medium">{reward.name}</TableCell>
                        <TableCell className="max-w-xs text-sm text-gray-600">
                          {reward.description || "-"}
                        </TableCell>
                        <TableCell>{reward.points_value}</TableCell>
                        <TableCell>{reward.stock_available}</TableCell>
                        <TableCell>
                          {reward.stock_available > 0 ? "Disponible" : "Agotado"}
                        </TableCell>
                        <TableCell className="text-right">
                          <div className="flex justify-end gap-2">
                            <Button variant="outline" size="sm" onClick={() => openEdit(reward)}>
                              Editar
                            </Button>
                            <Button
                              variant="destructive"
                              size="sm"
                              onClick={() => setDeleteTarget(reward)}
                            >
                              Eliminar
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </div>

        <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
          <DialogContent className="max-w-xl">
            <DialogHeader>
              <DialogTitle>
                {editingReward ? "Editar premio" : "Agregar premio"}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="reward-name">Nombre</Label>
                <Input
                  id="reward-name"
                  value={formData.name}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, name: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="reward-description">Descripción</Label>
                <Textarea
                  id="reward-description"
                  value={formData.description}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, description: event.target.value }))
                  }
                />
              </div>
              <div className="grid gap-4 sm:grid-cols-2">
                <div className="space-y-2">
                  <Label htmlFor="reward-points">Valor de puntos</Label>
                  <Input
                    id="reward-points"
                    type="number"
                    min={0}
                    value={formData.points_value}
                    onChange={(event) =>
                      setFormData((prev) => ({
                        ...prev,
                        points_value: Number(event.target.value),
                      }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="reward-stock">Stock disponible</Label>
                  <Input
                    id="reward-stock"
                    type="number"
                    min={0}
                    value={formData.stock_available}
                    onChange={(event) =>
                      setFormData((prev) => ({
                        ...prev,
                        stock_available: Number(event.target.value),
                      }))
                    }
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="reward-image">Imagen (URL)</Label>
                <Input
                  id="reward-image"
                  value={formData.image_url}
                  onChange={(event) =>
                    setFormData((prev) => ({ ...prev, image_url: event.target.value }))
                  }
                />
              </div>
            </div>
            <DialogFooter className="mt-6">
              <Button variant="outline" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button onClick={handleSave}>
                {editingReward ? "Guardar cambios" : "Agregar"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <AlertDialog
          open={Boolean(deleteTarget)}
          onOpenChange={() => setDeleteTarget(null)}
        >
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>¿Eliminar premio?</AlertDialogTitle>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancelar</AlertDialogCancel>
              <AlertDialogAction onClick={handleDelete}>Eliminar</AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </PointsAdminGuard>
    </PointsDashboardLayout>
  );
}