"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { PointsDashboardLayout } from "@/components/dashboard/points-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import { supabase } from "@/lib/supabase";
import {
  fetchRedemptions,
  fetchRewards,
  fetchUserPointsSummaryForUser,
  getCurrentUser,
  requestRewardRedemption,
} from "@/lib/meiwa-points-data";
import { isPointsAdminEmail } from "@/lib/points-auth";
import type { User } from "@supabase/supabase-js";

const redemptionStatuses = ["solicitada", "aprobada", "entregada", "cancelada"];

type Reward = {
  id: string;
  name: string;
  description: string | null;
  points_value: number;
  stock_available: number;
  image_url: string | null;
  storage_path: string | null;
};

type Redemption = {
  id: string;
  reward_id: string;
  user_id: string;
  points: number;
  stock_before: number;
  stock_after: number;
  requested_at: string | null;
  status: string | null;
  mp_rewards: { name: string | null; description: string | null; points_value: number | null } | null;
  profiles: { full_name: string | null; email: string | null } | null;
};

type RedemptionRow = Omit<Redemption, "mp_rewards" | "profiles"> & {
  mp_rewards:
    | { name: string | null; description: string | null; points_value: number | null }
    | { name: string | null; description: string | null; points_value: number | null }[]
    | null;
  profiles:
    | { full_name: string | null; email: string | null }
    | { full_name: string | null; email: string | null }[]
    | null;
};

type UserStatus = {
  status: string;
};

export default function RewardsPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [isAdmin, setIsAdmin] = useState(false);
  const [loading, setLoading] = useState(true);
  const [rewards, setRewards] = useState<Reward[]>([]);
  const [redemptions, setRedemptions] = useState<Redemption[]>([]);
  const [userStatus, setUserStatus] = useState<UserStatus | null>(null);
  const [pointsAvailable, setPointsAvailable] = useState(0);
  const [pointsEarned, setPointsEarned] = useState(0);
  const [pointsSpent, setPointsSpent] = useState(0);
  const [editingRedemption, setEditingRedemption] = useState<Redemption | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [deleteTarget, setDeleteTarget] = useState<Redemption | null>(null);
  const [statusValue, setStatusValue] = useState("solicitada");

  const loadRewards = async () => {
    const { data, error } = await fetchRewards();
    if (error) {
      toast.error("No se pudieron cargar los premios");
      return;
    }
    setRewards((data as Reward[]) ?? []);
  };

  const loadRedemptions = async () => {
    const { data, error } = await fetchRedemptions();
    if (error) {
      toast.error("No se pudieron cargar los canjes");
      return;
    }
    const normalizedRedemptions = ((data ?? []) as RedemptionRow[]).map((row) => ({
      ...row,
      mp_rewards: Array.isArray(row.mp_rewards) ? row.mp_rewards[0] ?? null : row.mp_rewards ?? null,
      profiles: Array.isArray(row.profiles) ? row.profiles[0] ?? null : row.profiles ?? null,
    }));
    setRedemptions(normalizedRedemptions);
  };

  const loadUserStatus = async (currentUser: User) => {
    const { data } = await supabase
      .from("mp_user_status")
      .select("status")
      .eq("user_id", currentUser.id)
      .maybeSingle();
    setUserStatus((data as UserStatus | null) ?? null);
  };

  const loadPointsSummary = async (currentUser: User) => {
    const { data } = await fetchUserPointsSummaryForUser(currentUser.id);
    setPointsAvailable(data?.points_available ?? 0);
    setPointsEarned(data?.points_earned ?? 0);
    setPointsSpent(data?.points_spent ?? 0);
  };

  useEffect(() => {
    const loadSession = async () => {
      const sessionUser = await getCurrentUser();
      if (!sessionUser) {
        router.push("/");
        return;
      }
      setUser(sessionUser);
      setIsAdmin(isPointsAdminEmail(sessionUser.email));
      await Promise.all([
        loadRewards(),
        loadUserStatus(sessionUser),
        loadPointsSummary(sessionUser),
      ]);
      if (isPointsAdminEmail(sessionUser.email)) {
        await loadRedemptions();
      }
      setLoading(false);
    };

    void loadSession();
  }, [router]);

  const availableStatus = useMemo(() => {
    return userStatus?.status === "activo";
  }, [userStatus]);

  const handleRedeem = async (reward: Reward) => {
    if (!user) return;

    if (!availableStatus) {
      toast.error("Usuario inactivo. No puedes solicitar canjes.");
      return;
    }

    if (reward.stock_available <= 0) {
      toast.error("El premio está agotado.");
      return;
    }

    if (pointsAvailable < reward.points_value) {
      toast.error("No tienes puntos suficientes para este canje.");
      return;
    }

    const { error } = await requestRewardRedemption(reward.id, user);
    if (error) {
      toast.error(error.message || "No se pudo registrar el canje");
      return;
    }

    toast.success("Canje solicitado correctamente");
    await Promise.all([
      loadRewards(),
      loadPointsSummary(user),
      isAdmin ? loadRedemptions() : Promise.resolve(),
    ]);
  };

  const openEditRedemption = (redemption: Redemption) => {
    setEditingRedemption(redemption);
    setStatusValue(redemption.status ?? "solicitada");
    setDialogOpen(true);
  };

  const handleSaveRedemption = async () => {
    if (!editingRedemption) return;
    const { error } = await supabase
      .from("mp_redemptions")
      .update({ status: statusValue })
      .eq("id", editingRedemption.id);
    if (error) {
      toast.error("No se pudo actualizar el canje");
      return;
    }
    toast.success("Canje actualizado");
    setDialogOpen(false);
    await loadRedemptions();
  };

  const handleDeleteRedemption = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase
      .from("mp_redemptions")
      .delete()
      .eq("id", deleteTarget.id);
    if (error) {
      toast.error("No se pudo eliminar el canje");
      return;
    }
    toast.success("Canje eliminado");
    setDeleteTarget(null);
    await loadRedemptions();
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-blue-600" />
      </div>
    );
  }

  return (
    <PointsDashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Catálogo de premios</h1>
            <p className="text-gray-600">
              Canjea tus puntos por premios disponibles en Meiwa Points.
            </p>
          </div>
          <Card className="sm:w-80">
            <CardHeader>
              <CardTitle className="text-base">Tus puntos</CardTitle>
            </CardHeader>
            <CardContent className="space-y-1 text-sm text-gray-600">
              <p>Puntos ganados: {pointsEarned}</p>
              <p>Puntos canjeados: {pointsSpent}</p>
              <p className="font-semibold text-gray-900">Disponibles: {pointsAvailable}</p>
            </CardContent>
          </Card>
        </div>

        <Tabs defaultValue="rewards">
          <TabsList>
            <TabsTrigger value="rewards">Premios</TabsTrigger>
            {isAdmin && <TabsTrigger value="redemptions">Matriz de canje</TabsTrigger>}
          </TabsList>

          <TabsContent value="rewards">
            <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3">
              {rewards.map((reward) => (
                <Card key={reward.id} className="flex flex-col">
                  <div className="h-40 w-full overflow-hidden rounded-t-xl bg-gray-100">
                    <img
                      src={reward.image_url ?? "/placeholder.svg"}
                      alt={reward.name}
                      className="h-full w-full object-cover"
                    />
                  </div>
                  <CardContent className="flex flex-1 flex-col gap-3 p-5">
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900">{reward.name}</h3>
                      <p className="text-sm text-gray-600">{reward.description}</p>
                    </div>
                    <div className="text-sm text-gray-600 space-y-1">
                      <p>Valor: {reward.points_value} puntos</p>
                      <p>Stock disponible: {reward.stock_available}</p>
                      <p>
                        Estado: {reward.stock_available > 0 ? "Disponible" : "Agotado"}
                      </p>
                    </div>
                    <Button
                      className="mt-auto"
                      disabled={reward.stock_available <= 0}
                      onClick={() => handleRedeem(reward)}
                    >
                      Solicitar canje
                    </Button>
                  </CardContent>
                </Card>
              ))}
              {rewards.length === 0 && (
                <div className="col-span-full text-center text-gray-500">
                  No hay premios disponibles.
                </div>
              )}
            </div>
          </TabsContent>

          {isAdmin && (
            <TabsContent value="redemptions">
              <Card>
                <CardHeader>
                  <CardTitle>Matriz de canje</CardTitle>
                </CardHeader>
                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Premio</TableHead>
                        <TableHead>Descripción</TableHead>
                        <TableHead>Valor de puntos</TableHead>
                        <TableHead>Usuario</TableHead>
                        <TableHead>Stock antes</TableHead>
                        <TableHead>Stock después</TableHead>
                        <TableHead>Estado</TableHead>
                        <TableHead>Solicitud</TableHead>
                        <TableHead className="text-right">Acciones</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {redemptions.length === 0 && (
                        <TableRow>
                          <TableCell colSpan={9} className="text-center text-gray-500">
                            No hay solicitudes registradas.
                          </TableCell>
                        </TableRow>
                      )}
                      {redemptions.map((redemption) => (
                        <TableRow key={redemption.id}>
                          <TableCell>{redemption.mp_rewards?.name || "-"}</TableCell>
                          <TableCell className="max-w-xs text-sm text-gray-600">
                            {redemption.mp_rewards?.description || "-"}
                          </TableCell>
                          <TableCell>{redemption.points}</TableCell>
                          <TableCell>
                            {redemption.profiles?.full_name ||
                              redemption.profiles?.email ||
                              "-"}
                          </TableCell>
                          <TableCell>{redemption.stock_before}</TableCell>
                          <TableCell>{redemption.stock_after}</TableCell>
                          <TableCell>
                            {redemption.stock_after > 0 ? "Disponible" : "Agotado"}
                          </TableCell>
                          <TableCell className="capitalize">
                            {redemption.status ?? "solicitada"}
                          </TableCell>
                          <TableCell className="text-right">
                            <div className="flex justify-end gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => openEditRedemption(redemption)}
                              >
                                Editar
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => setDeleteTarget(redemption)}
                              >
                                Eliminar
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            </TabsContent>
          )}
        </Tabs>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Editar solicitud de canje</DialogTitle>
          </DialogHeader>
          <div className="space-y-2">
            <Select value={statusValue} onValueChange={setStatusValue}>
              <SelectTrigger>
                <SelectValue placeholder="Selecciona estatus" />
              </SelectTrigger>
              <SelectContent>
                {redemptionStatuses.map((status) => (
                  <SelectItem key={status} value={status}>
                    {status}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <DialogFooter className="mt-6">
            <Button variant="outline" onClick={() => setDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSaveRedemption}>Guardar</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <AlertDialog
        open={Boolean(deleteTarget)}
        onOpenChange={() => setDeleteTarget(null)}
      >
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar canje?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={handleDeleteRedemption}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </PointsDashboardLayout>
  );
}