"use client";

import { useEffect, useMemo, useState } from "react";
import { PointsDashboardLayout } from "@/components/dashboard/points-dashboard-layout";
import { PointsAdminGuard } from "@/components/points-admin-guard/points-admin-guard";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { toast } from "sonner";
import {
  fetchUserPointsSummary,
  fetchUserStatuses,
  fetchUsers,
  PointsSummary,
} from "@/lib/meiwa-points-data";
import { supabase } from "@/lib/supabase";

const emptyForm = {
  user_id: "",
  status: "activo",
};

type UserStatus = {
  id: string;
  user_id: string;
  status: string;
  profiles: { full_name: string | null; email: string | null } | null;
};

type UserStatusRow = Omit<UserStatus, "profiles"> & {
  profiles:
    | { full_name: string | null; email: string | null }
    | { full_name: string | null; email: string | null }[]
    | null;
};

type UserProfile = {
  id: string;
  full_name: string | null;
  email: string | null;
};

export default function UsersMatrixPage() {
  const [userStatuses, setUserStatuses] = useState<UserStatus[]>([]);
  const [users, setUsers] = useState<UserProfile[]>([]);
  const [summaries, setSummaries] = useState<PointsSummary[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingStatus, setEditingStatus] = useState<UserStatus | null>(null);
  const [deleteTarget, setDeleteTarget] = useState<UserStatus | null>(null);
  const [formData, setFormData] = useState({ ...emptyForm });

  const loadData = async () => {
    setLoading(true);
    const [{ data: statusData }, { data: usersData }, { data: summaryData }] =
      await Promise.all([fetchUserStatuses(), fetchUsers(), fetchUserPointsSummary()]);
    const normalizedStatuses = ((statusData ?? []) as UserStatusRow[]).map((row) => ({
      ...row,
      profiles: Array.isArray(row.profiles) ? row.profiles[0] ?? null : row.profiles ?? null,
    }));
    setUserStatuses(normalizedStatuses);
    setUsers((usersData as UserProfile[]) ?? []);
    setSummaries((summaryData as PointsSummary[]) ?? []);
    setLoading(false);
  };

  useEffect(() => {
    void loadData();
  }, []);

  const summaryMap = useMemo(() => {
    return new Map(
      summaries.map((summary) => [summary.user_id, summary])
    );
  }, [summaries]);

  const openCreate = () => {
    setEditingStatus(null);
    setFormData({ ...emptyForm });
    setDialogOpen(true);
  };

  const openEdit = (status: UserStatus) => {
    setEditingStatus(status);
    setFormData({ user_id: status.user_id, status: status.status });
    setDialogOpen(true);
  };

  const handleSave = async () => {
    if (!formData.user_id) {
      toast.error("Selecciona un usuario");
      return;
    }

    const payload = {
      user_id: formData.user_id,
      status: formData.status,
    };

    const query = editingStatus
      ? supabase.from("mp_user_status").update(payload).eq("id", editingStatus.id)
      : supabase.from("mp_user_status").insert(payload);

    const { error } = await query;
    if (error) {
      toast.error("No se pudo guardar el estado");
      return;
    }

    toast.success(editingStatus ? "Estado actualizado" : "Estado agregado");
    setDialogOpen(false);
    await loadData();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase
      .from("mp_user_status")
      .delete()
      .eq("id", deleteTarget.id);
    if (error) {
      toast.error("No se pudo eliminar el registro");
      return;
    }
    toast.success("Registro eliminado");
    setDeleteTarget(null);
    await loadData();
  };

  return (
    <PointsDashboardLayout>
      <PointsAdminGuard>
        <div className="space-y-6">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Matriz de usuarios</h1>
              <p className="text-gray-600">
                Administra el estado y balance de puntos de los usuarios.
              </p>
            </div>
            <Button onClick={openCreate}>Agregar usuario</Button>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Usuarios</CardTitle>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-10">
                  <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-600" />
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Usuario</TableHead>
                      <TableHead>Puntos ganados</TableHead>
                      <TableHead>Puntos canjeados</TableHead>
                      <TableHead>Puntos disponibles</TableHead>
                      <TableHead>Estado</TableHead>
                      <TableHead className="text-right">Acciones</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {userStatuses.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={6} className="text-center text-gray-500">
                          No hay usuarios registrados.
                        </TableCell>
                      </TableRow>
                    )}
                    {userStatuses.map((status) => {
                      const summary = summaryMap.get(status.user_id) ?? {
                        points_earned: 0,
                        points_spent: 0,
                        points_available: 0,
                      };

                      return (
                        <TableRow key={status.id}>
                          <TableCell>
                            {status.profiles?.full_name || status.profiles?.email || "-"}
                          </TableCell>
                          <TableCell>{summary.points_earned}</TableCell>
                          <TableCell>{summary.points_spent}</TableCell>
                          <TableCell>{summary.points_available}</TableCell>
                          <TableCell className="capitalize">{status.status}</TableCell>
                          <TableCell className="text-right">
                            <div className="flex justify-end gap-2">
                              <Button variant="outline" size="sm" onClick={() => openEdit(status)}>
                                Editar
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => setDeleteTarget(status)}
                              >
                                Eliminar
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </div>

        <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
          <DialogContent className="max-w-lg">
            <DialogHeader>
              <DialogTitle>
                {editingStatus ? "Editar usuario" : "Agregar usuario"}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label>Usuario</Label>
                <Select
                  value={formData.user_id}
                  onValueChange={(value) =>
                    setFormData((prev) => ({ ...prev, user_id: value }))
                  }
                  disabled={Boolean(editingStatus)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona un usuario" />
                  </SelectTrigger>
                  <SelectContent>
                    {users.map((user) => (
                      <SelectItem key={user.id} value={user.id}>
                        {user.full_name || user.email}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Estado</Label>
                <Select
                  value={formData.status}
                  onValueChange={(value) =>
                    setFormData((prev) => ({ ...prev, status: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona estado" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="activo">Activo</SelectItem>
                    <SelectItem value="baja">Baja</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
            <DialogFooter className="mt-6">
              <Button variant="outline" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button onClick={handleSave}>
                {editingStatus ? "Guardar cambios" : "Agregar"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <AlertDialog
          open={Boolean(deleteTarget)}
          onOpenChange={() => setDeleteTarget(null)}
        >
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>¿Eliminar usuario?</AlertDialogTitle>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancelar</AlertDialogCancel>
              <AlertDialogAction onClick={handleDelete}>Eliminar</AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </PointsAdminGuard>
    </PointsDashboardLayout>
  );
}