"use client";

import { useMemo, useState } from "react";
import Image from "next/image";
import { Award, BadgeCheck, Leaf, ShieldCheck } from "lucide-react";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";

const categories = [
  {
    id: "medio-ambiente",
    label: "Medio Ambiente",
    description: "Gestión ambiental, impacto y sostenibilidad.",
    accent: "from-emerald-500 via-emerald-400 to-lime-400",
    icon: Leaf,
  },
  {
    id: "calidad",
    label: "Calidad",
    description: "Estandarización y mejora continua.",
    accent: "from-sky-500 via-indigo-500 to-violet-500",
    icon: Award,
  },
  {
    id: "seguridad",
    label: "Seguridad",
    description: "Protección del equipo y cultura preventiva.",
    accent: "from-rose-500 via-orange-500 to-amber-400",
    icon: ShieldCheck,
  },
];

const certificates = [
  {
    id: "ema-iso-14001",
    category: "medio-ambiente",
    name: "Certificación ISO 14001",
    issuer: "Organismo Internacional de Certificación",
    code: "MEIWA-EMA-2024",
    validity: "2024 - 2026",
    status: "Vigente",
    image: "/meiwa-core/meiwa-sgi/certificate/CERTIFICADO1.jpg",
    highlights: [
      "Gestión ambiental alineada a ISO 14001",
      "Reducción de huella ambiental y residuos",
      "Auditoría anual y planes de mejora",
    ],
  },
  {
    id: "cal-iso-9001",
    category: "calidad",
    name: "Certificación ISO 9001",
    issuer: "Instituto de Calidad Industrial",
    code: "MEIWA-CAL-2024",
    validity: "2024 - 2026",
    status: "Vigente",
    image: null,
    highlights: [
      "Procesos documentados y trazables",
      "Control de calidad por área",
      "Cultura de mejora continua",
    ],
  },
  {
    id: "seg-iso-45001",
    category: "seguridad",
    name: "Certificación ISO 45001",
    issuer: "Consejo de Seguridad Laboral",
    code: "MEIWA-SEG-2024",
    validity: "2024 - 2026",
    status: "Vigente",
    image: null,
    highlights: [
      "Gestión integral de riesgos",
      "Entrenamientos preventivos periódicos",
      "Indicadores de seguridad monitoreados",
    ],
  },
];

export default function CertificadoPage() {
  const [activeCategory, setActiveCategory] = useState(categories[0].id);
  const activeCategoryData = categories.find((category) => category.id === activeCategory);

  const filteredCertificates = useMemo(
    () => certificates.filter((certificate) => certificate.category === activeCategory),
    [activeCategory]
  );

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-emerald-50">
        <div className="relative overflow-hidden">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pt-16 pb-10">
            <div className="max-w-3xl space-y-4">
              <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
                Certificaciones SGI
              </p>
              <h1 className="text-4xl sm:text-5xl font-bold text-gray-900">
                Certificados por categoría
              </h1>
              <p className="text-lg text-gray-600">
                Consulta los certificados obtenidos en Medio Ambiente, Calidad y Seguridad.
                Selecciona una categoría para ver el detalle de cada acreditación.
              </p>
            </div>
          </div>
        </div>

        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pb-20">
          <div className="flex flex-wrap gap-3">
            {categories.map((category) => {
              const Icon = category.icon;
              const isActive = activeCategory === category.id;

              return (
                <button
                  key={category.id}
                  type="button"
                  onClick={() => setActiveCategory(category.id)}
                  className={`flex items-center gap-2 rounded-full border px-4 py-2 text-sm font-semibold transition-all duration-200 ${
                    isActive
                      ? "border-transparent bg-gradient-to-r from-emerald-500 to-lime-500 text-white shadow-lg"
                      : "border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
                  }`}
                >
                  <Icon className={`h-4 w-4 ${isActive ? "text-white" : "text-emerald-500"}`} />
                  {category.label}
                </button>
              );
            })}
          </div>

          <div className="mt-8 grid gap-6 lg:grid-cols-[280px_1fr]">
            <aside className="space-y-6">
              <div className="rounded-3xl border border-emerald-100 bg-white p-6 shadow-lg">
                <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-600">
                  Resumen
                </p>
                <p className="mt-3 text-2xl font-semibold text-gray-900">
                  {activeCategoryData?.label}
                </p>
                <p className="mt-2 text-sm text-gray-500">
                  {activeCategoryData?.description}
                </p>
                <div className="mt-6 space-y-3">
                  <div className="rounded-2xl bg-emerald-50 px-4 py-3">
                    <p className="text-xs font-semibold uppercase text-emerald-600">Certificados</p>
                    <p className="text-xl font-semibold text-emerald-900">
                      {filteredCertificates.length}
                    </p>
                  </div>
                  <div className="rounded-2xl bg-gray-50 px-4 py-3">
                    <p className="text-xs font-semibold uppercase text-gray-500">Estado general</p>
                    <p className="mt-1 inline-flex items-center gap-2 text-sm font-semibold text-gray-800">
                      <BadgeCheck className="h-4 w-4 text-emerald-500" /> Vigente
                    </p>
                  </div>
                </div>
              </div>

              <div className={`rounded-3xl bg-gradient-to-br ${activeCategoryData?.accent} p-6 text-white shadow-lg`}>
                <p className="text-xs font-semibold uppercase tracking-[0.3em] text-white/80">
                  Estado
                </p>
                <p className="mt-2 text-2xl font-semibold">Actualizado 2024</p>
                <p className="mt-2 text-sm text-white/80">
                  Auditorías y seguimiento anual programados.
                </p>
              </div>
            </aside>

            <section className="space-y-6">
              {filteredCertificates.map((certificate) => (
                <article
                  key={certificate.id}
                  className="overflow-hidden rounded-3xl border border-gray-200 bg-white shadow-sm transition-transform duration-200 hover:-translate-y-1 hover:shadow-lg"
                >
                  <div className="grid gap-6 lg:grid-cols-[280px_1fr]">
                    <div className="relative flex h-full items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-lime-50 p-6">
                      {certificate.image ? (
                        <Image
                          src={certificate.image}
                          alt={`Imagen del ${certificate.name}`}
                          width={460}
                          height={320}
                          className="h-52 w-full rounded-2xl border border-emerald-100 object-cover"
                          unoptimized
                        />
                      ) : (
                        <div className="flex h-52 w-full flex-col items-center justify-center gap-3 rounded-2xl border border-dashed border-emerald-200 bg-white">
                          <div className="flex h-12 w-12 items-center justify-center rounded-full bg-emerald-50">
                            <Award className="h-6 w-6 text-emerald-600" />
                          </div>
                          <p className="text-sm font-semibold text-emerald-700">Imagen disponible próximamente</p>
                          <p className="text-xs text-gray-500">Certificado digital en preparación</p>
                        </div>
                      )}
                      <span className="absolute left-6 top-6 rounded-full bg-emerald-500/90 px-3 py-1 text-xs font-semibold uppercase tracking-wide text-white">
                        {certificate.status}
                      </span>
                    </div>

                    <div className="space-y-5 p-6">
                      <div>
                        <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-500">
                          {activeCategoryData?.label}
                        </p>
                        <h2 className="mt-2 text-2xl font-semibold text-gray-900">
                          {certificate.name}
                        </h2>
                        <p className="mt-1 text-sm text-gray-500">{certificate.issuer}</p>
                      </div>

                      <div className="grid gap-4 sm:grid-cols-2 text-sm">
                        <div className="rounded-2xl bg-gray-50 px-4 py-3">
                          <p className="text-xs uppercase text-gray-500">Código</p>
                          <p className="mt-1 font-semibold text-gray-900">{certificate.code}</p>
                        </div>
                        <div className="rounded-2xl bg-gray-50 px-4 py-3">
                          <p className="text-xs uppercase text-gray-500">Vigencia</p>
                          <p className="mt-1 font-semibold text-gray-900">{certificate.validity}</p>
                        </div>
                      </div>

                      <ul className="space-y-2 text-sm text-gray-600">
                        {certificate.highlights.map((item) => (
                          <li key={item} className="flex items-start gap-2">
                            <span className="mt-2 h-2 w-2 rounded-full bg-emerald-500" />
                            <span>{item}</span>
                          </li>
                        ))}
                      </ul>
                    </div>
                  </div>
                </article>
              ))}
            </section>
          </div>
        </div>
      </div>
    </SGIDashboardLayout>
  );
}