"use client";

import Link from "next/link";
import { useEffect, useMemo, useState } from "react";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

const resourceTypeOptions = [
  { value: "human", label: "Humanos" },
  { value: "financial", label: "Financieros" },
  { value: "technical", label: "Materiales / Técnicos" },
  { value: "organizational", label: "Organizacionales" },
];

const resourceStatusOptions = [
  { value: "current", label: "Vigente" },
  { value: "improving", label: "En mejora" },
  { value: "implementing", label: "En implementación" },
  { value: "to_update", label: "Por actualizar" },
];

const competenceDimensions = [
  { key: "env_score", label: "Medio Ambiente" },
  { key: "quality_score", label: "Calidad" },
  { key: "safety_score", label: "Seguridad" },
  { key: "s5_score", label: "5S" },
  { key: "incidents_score", label: "Incidencias" },
  { key: "productivity_score", label: "Productividad" },
  { key: "general_eval_score", label: "Evaluación General" },
  { key: "kpi_training_score", label: "KPI/Capacitación" },
] as const;

const documentClassificationLabels: Record<string, string> = {
  SGA: "SG Ambiental",
  SGC: "SG Calidad",
  SGS: "SG Seguridad",
};

const monthKeys = [
  "jan",
  "feb",
  "mar",
  "apr",
  "may",
  "jun",
  "jul",
  "aug",
  "sep",
  "oct",
  "nov",
  "dec",
] as const;

type RiskOpportunity = {
  p_residual: number | null;
  i_residual: number | null;
  linked_to_legal: boolean | null;
};

type EnvironmentalAspect = {
  id: string;
  magnitude_m: number | null;
  control_influence_fci: number | null;
  legal_requirement_rl: number | null;
  frequency_f: number | null;
  extension_e: number | null;
};

type LegalRequirement = {
  compliance_status: string | null;
  action_status: string | null;
};

type ActionPlanItem = {
  source_type: string | null;
  source_id: string | null;
  owner: string | null;
  due_date: string | null;
  start_date: string | null;
  end_date: string | null;
};

type Objective = {
  status: string | null;
  countermeasures: string | null;
};

type IndicatorRecord = {
  annual_sum_target: number | null;
  annual_average_target: number | null;
  last_updated_at: string | null;
  [key: string]: number | string | null;
};

type ResourceRecord = {
  resource_type: string;
  status: string;
  process_activity: string | null;
  is_critical: boolean | null;
  linked_to_legal: boolean | null;
};

type ProcessRecord = {
  is_active: boolean | null;
};

type CompetenceResult = {
  overall_score: number | null;
  env_score: number | null;
  quality_score: number | null;
  safety_score: number | null;
  s5_score: number | null;
  incidents_score: number | null;
  productivity_score: number | null;
  general_eval_score: number | null;
  kpi_training_score: number | null;
};

type AwarenessModule = {
  id: string;
  points_available: number | null;
  is_active?: boolean | null;
};

type AwarenessLesson = {
  id: string;
  module_id: string;
  is_active?: boolean | null;
};

type AwarenessProgress = {
  module_id: string;
  status: string | null;
  final_score: number | null;
  points_obtained: number | null;
};

type AwarenessLessonView = {
  lesson_id: string;
};

type CommunicationMatrix = {
  category: string;
};

type CommunicationPost = {
  category: string | null;
  status: string | null;
  publish_date: string | null;
};

type DocumentControlRecord = {
  classification: string;
  target_date: string | null;
  start_date: string | null;
  issue_date: string | null;
};

type DocumentRevision = {
  revision_date: string | null;
};

type Supplier = {
  id: string;
};

type SupplierEvaluation = {
  supplier_id: string | null;
  status: string | null;
};

type SalidaEntry = {
  cantidad: string | null;
  fecha: string | null;
};

type QirRecord = {
  lostCost: number | null;
};

type StockRecord = {
  approxWeight: number | null;
};

type ScrapRecord = {
  approxWeight: number | null;
};

type EnergyRecord = {
  generationKwh: number;
  consumptionKwh: number;
  energyAcquiredCfeKwh: number;
  energySoldCfeKwh: number;
  cfeBill: number;
};

type WaterRecord = {
  date: string;
  month: string;
  year: number;
  quantityM3: number;
  totalCost: number;
  type?: "Red" | "RO" | null;
};

type FuelGasolineRecord = {
  budget: number;
  equivalentLiters: number;
  averageCostPerLiter: number;
};

type FuelGasRecord = {
  totalLiters: number;
  totalCost: number;
  costUnit: string | null;
  quantity: number;
  provider: string | null;
};

type AuditSession = {
  id: string;
};

type AuditResponse = {
  response: string;
};

type NonconformityRecord = {
  status: string | null;
  classification: string | null;
};

type ContinuousImprovementRecord = {
  status: string | null;
  source_type: string | null;
};

type StatsState = {
  riskTotal: string;
  riskResidualNotAcceptable: string;
  riskCritical: string;
  riskLegal: string;
  aspectsTotal: string;
  aspectsSignificant: string;
  aspectsActionPercent: string;
  legalComplies: string;
  legalNonComplies: string;
  legalNotEvaluated: string;
  legalOpenActions: string;
  actionPlanTotal: string;
  actionPlanCompletion: string;
  actionPlanDelayed: string;
  actionPlanWithoutOwner: string;
  objectivesTotal: string;
  objectivesCompliance: string;
  objectivesRisk: string;
  objectivesNotMet: string;
  objectivesCountermeasures: string;
  indicatorsUpdatedPercent: string;
  indicatorsOutOfTarget: string;
  resourcesDistribution: string;
  resourcesStatus: string;
  resourcesCoverage: string;
  resourcesCriticalWithoutBackup: string;
  competenceScoreGlobal: string;
  competenceMainGap: string;
  awarenessModulesCompleted: string;
  awarenessProgress: string;
  awarenessScore: string;
  awarenessPoints: string;
  communicationMatrixItems: string;
  communicationPostsPeriod: string;
  communicationTopCategory: string;
  documentControlClassification: string;
  documentControlDelayed: string;
  documentControlRevisionsPeriod: string;
  externalEvaluated: string;
  externalCurrent: string;
  externalLinkedNc: string;
  rsuTotal: string;
  rsuIncidents: string;
  rmeTotal: string;
  rmeIncidents: string;
  rpTotal: string;
  rpIncidents: string;
  qualityQir: string;
  qualityStockPercent: string;
  qualityScrapPercent: string;
  qualityStockScrap: string;
  qualityLostCost: string;
  qualityScrapVolume: string;
  energySelfSufficiency: string;  
  energyDependency: string;
  energyBill: string;
  energyGeneratedConsumed: string;
  waterRedConsumption: string;
  waterRoConsumption: string;
  waterTrend: string;
  waterRedTrend: string;
  waterRoTrend: string;
  waterRedCost: string;  
  waterRoCost: string;
  fuelsGasolineLiters: string;
  fuelsGasolineSpend: string;
  fuelsGasolineAvgCost: string;
  fuelsGasLiters: string;
  fuelsGasSpend: string;
  fuelsSuppliers: string;
  auditTotal: string;
  auditFindings: string;
  auditNcMajorMinor: string;
  auditClosedPercent: string;
  ncRecords: string;
  ncStatus: string;
  ncClassification: string;
  improvementTotal: string;
  improvementClosedPercent: string;
  improvementTopSource: string;
};

const defaultStats: StatsState = {
  riskTotal: "0",
  riskResidualNotAcceptable: "0",
  riskCritical: "0",
  riskLegal: "0",
  aspectsTotal: "0",
  aspectsSignificant: "0",
  aspectsActionPercent: "0.0%",
  legalComplies: "0.0%",
  legalNonComplies: "0.0%",
  legalNotEvaluated: "0.0%",
  legalOpenActions: "0",
  actionPlanTotal: "0",
  actionPlanCompletion: "0.0%",
  actionPlanDelayed: "0",
  actionPlanWithoutOwner: "0",
  objectivesTotal: "0",
  objectivesCompliance: "0.0%",
  objectivesRisk: "0.0%",
  objectivesNotMet: "0.0%",
  objectivesCountermeasures: "0",
  indicatorsUpdatedPercent: "0.0%",
  indicatorsOutOfTarget: "0",
  resourcesDistribution: "-",
  resourcesStatus: "-",
  resourcesCoverage: "0/0",
  resourcesCriticalWithoutBackup: "0",
  competenceScoreGlobal: "0",
  competenceMainGap: "-",
  awarenessModulesCompleted: "0 / 0",
  awarenessProgress: "0.0%",
  awarenessScore: "0",
  awarenessPoints: "0 / 0",
  communicationMatrixItems: "0",
  communicationPostsPeriod: "0",
  communicationTopCategory: "-",
  documentControlClassification: "-",
  documentControlDelayed: "0",
  documentControlRevisionsPeriod: "0",
  externalEvaluated: "0 / 0",
  externalCurrent: "0",
  externalLinkedNc: "0",
  rsuTotal: "0",
  rsuIncidents: "0",
  rmeTotal: "0",
  rmeIncidents: "0",
  rpTotal: "0",
  rpIncidents: "0",
  qualityQir: "0",
  qualityStockPercent: "0.0%",
  qualityScrapPercent: "0.0%",
  qualityStockScrap: "0 / 0",
  qualityLostCost: "$0.00",
  qualityScrapVolume: "0 kg",  
  energySelfSufficiency: "0.0%",
  energyDependency: "0.0% / 0.0%",
  energyBill: "$0.00",
  energyGeneratedConsumed: "0 kWh / 0 kWh",
  waterRedConsumption: "0 m³",
  waterRoConsumption: "0 m³",
  waterTrend: "0 m³",
  waterRedTrend: "0 m³",
  waterRoTrend: "0 m³",
  waterRedCost: "$0.00",
  waterRoCost: "$0.00",
  fuelsGasolineLiters: "0",
  fuelsGasolineSpend: "$0.00",
  fuelsGasolineAvgCost: "$0.00",
  fuelsGasLiters: "0",
  fuelsGasSpend: "$0.00",
  fuelsSuppliers: "0",
  auditTotal: "0",
  auditFindings: "0 / 0",
  auditNcMajorMinor: "0 / 0",
  auditClosedPercent: "0.0%",
  ncRecords: "0",
  ncStatus: "-",
  ncClassification: "-",
  improvementTotal: "0",
  improvementClosedPercent: "0.0%",
  improvementTopSource: "-",
};

const toNumber = (value: number | string | null | undefined) => {
  if (typeof value === "number") return value;
  if (value === null || value === undefined) return 0;
  const parsed = Number(value);
  return Number.isNaN(parsed) ? 0 : parsed;
};

const formatPercent = (value: number, decimals = 1) => `${value.toFixed(decimals)}%`;

const computeExpectedValue = (aspect: EnvironmentalAspect) =>
  toNumber(aspect.magnitude_m) +
  toNumber(aspect.control_influence_fci) +
  toNumber(aspect.legal_requirement_rl) +
  toNumber(aspect.frequency_f) +
  toNumber(aspect.extension_e);

const getLevel = (p?: number | null, i?: number | null) => {
  if (!p || !i) return null;
  return p * i;
};

const getClassification = (level?: number | null, impact?: number | null) => {
  if (!level) return null;
  let classification = "low";
  if (level >= 20) classification = "critical";
  else if (level >= 13) classification = "high";
  else if (level >= 6) classification = "medium";

  if (impact === 5 && (classification === "low" || classification === "medium")) {
    classification = "high";
  }

  return classification;
};

const getAcceptable = (level?: number | null) => {
  if (!level) return null;
  return level <= 12;
};

const isDateBeforeToday = (value?: string | null) => {
  if (!value) return false;
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return false;
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  date.setHours(0, 0, 0, 0);
  return date < today;
};

const deriveStatus = (item: Pick<ActionPlanItem, "owner" | "due_date" | "start_date" | "end_date">) => {
  if (item.end_date) return "completed";
  if (item.due_date && isDateBeforeToday(item.due_date)) return "delayed";
  if (item.start_date) return "in_progress";
  if (item.owner && item.due_date) return "planned";
  if (item.owner) return "planned";
  return "unassigned";
};

const computeAnnualSummary = (record: IndicatorRecord) => {
  const values = monthKeys
    .map((key) => record[key])
    .filter((value): value is number => typeof value === "number");
  if (!values.length) {
    return { sum: null, average: null };
  }
  const sum = values.reduce((total, current) => total + current, 0);
  const average = values.length ? sum / values.length : null;
  return { sum, average };
};

const formatAnnualTargetMatch = (
  summary: { sum: number | null; average: number | null },
  record: IndicatorRecord,
) => {
  if (
    summary.sum === null ||
    summary.average === null ||
    record.annual_sum_target == null ||
    record.annual_average_target == null
  ) {
    return null;
  }
  const averageRounded = Number(summary.average.toFixed(2));
  const sumMatches = summary.sum === record.annual_sum_target;
  const averageMatches = averageRounded === record.annual_average_target;
  return sumMatches && averageMatches;
};

const computeDocumentStatus = (targetDate?: string | null, startDate?: string | null, issueDate?: string | null) => {
  if (!targetDate) return "Sin objetivo";
  const today = new Date();
  const target = new Date(`${targetDate}T23:59:59`);

  if (!issueDate && today > target) return "Retrasado";
  if (targetDate && !startDate) return "Planeado";
  if (targetDate && startDate && !issueDate) return "En Proceso";
  if (targetDate && startDate && issueDate) return "Completo";
  return "Sin objetivo";
};

const safeDivide = (numerator: number, denominator: number) =>
  denominator ? numerator / denominator : 0;

export default function SGIStatisticsPage() {
  const [stats, setStats] = useState<StatsState>(defaultStats);
  const [isLoading, setIsLoading] = useState(true);
  const [viewMode, setViewMode] = useState<"panorama" | "modules">("panorama");
  const [moduleFilter, setModuleFilter] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");

  const numberFormatter = useMemo(
    () =>
      new Intl.NumberFormat("es-MX", {
        maximumFractionDigits: 2,
      }),
    [],
  );

  const currencyFormatter = useMemo(
    () =>
      new Intl.NumberFormat("es-MX", {
        style: "currency",
        currency: "MXN",
        maximumFractionDigits: 2,
      }),
    [],
  );

  useEffect(() => {
    let isMounted = true;

    const loadStats = async () => {
      setIsLoading(true);

      const { data: sessionData } = await supabase.auth.getSession();
      const userId = sessionData.session?.user.id ?? null;

      const [
        riskResponse,
        aspectsResponse,
        legalResponse,
        actionResponse,
        objectivesResponse,
        indicatorsResponse,
        resourcesResponse,
        processesResponse,
        competenceResponse,
        awarenessModulesResponse,
        awarenessLessonsResponse,
        awarenessProgressResponse,
        matrixResponse,
        postsResponse,
        documentsResponse,
        revisionsResponse,
        suppliersResponse,
        evaluationsResponse,
        rsuResponse,
        rmeResponse,
        rpResponse,
        qualityQirResponse,
        qualityStockResponse,
        qualityScrapResponse,
        energyResponse,
        waterResponse,
        gasolineResponse,
        gasResponse,
        auditSessionsResponse,
        auditResponsesResponse,
        ncResponse,
        improvementResponse,
      ] = await Promise.all([
        supabase.from("env_risk_opportunities").select("p_residual,i_residual,linked_to_legal"),
        supabase
          .from("environmental_aspects")
          .select("id,magnitude_m,control_influence_fci,legal_requirement_rl,frequency_f,extension_e"),
        supabase.from("legal_requirements").select("compliance_status,action_status"),
        supabase.from("sga_action_plan").select("source_type,source_id,owner,due_date,start_date,end_date"),
        supabase.from("objectives").select("status,countermeasures"),
        supabase
          .from("indicators_yearly")
          .select(
            `annual_sum_target,annual_average_target,last_updated_at,${monthKeys.join(",")}`,
          ) as unknown,
        supabase
          .from("sgm_resources")
          .select("resource_type,status,process_activity,is_critical,linked_to_legal"),
        supabase.from("sgm_processes").select("is_active"),
        supabase
          .from("competence_results")
          .select(
            "overall_score,env_score,quality_score,safety_score,s5_score,incidents_score,productivity_score,general_eval_score,kpi_training_score",
          ),
        supabase.from("awareness_modules").select("id,points_available,is_active"),
        supabase.from("awareness_lessons").select("id,module_id,is_active"),
        userId
          ? supabase.from("awareness_user_module_progress").select("module_id,status,final_score,points_obtained").eq("user_id", userId)
          : Promise.resolve({ data: [] }),
        supabase.from("communication_matrix").select("category"),
        supabase.from("communications_posts").select("category,status,publish_date"),
        supabase.from("document_control").select("classification,target_date,start_date,issue_date"),
        supabase.from("document_control_revisions").select("revision_date"),
        supabase.from("suppliers").select("id"),
        supabase.from("supplier_evaluations").select("supplier_id,status"),
        supabase.from("rsu_salidas").select("cantidad,fecha"),
        supabase.from("rme_salidas").select("cantidad,fecha"),
        supabase.from("rp_salidas").select("cantidad,fecha"),
        supabase.from("quality_qir").select("lost_cost"),
        supabase.from("quality_stock").select("approx_weight"),
        supabase.from("quality_scrap").select("approx_weight"),
        supabase
          .from("energy_records")
          .select("generation_kwh,consumption_kwh,energy_acquired_cfe_kwh,energy_sold_cfe_kwh,cfe_bill"),
        supabase
          .from("water_records")
          .select("date,month,year,quantity_m3,total_cost,type"),
        supabase.from("fuel_gasoline_records").select("budget,equivalent_liters,average_cost_per_liter"),
        supabase.from("fuel_gas_records").select("total_liters,total_cost,cost_unit,quantity,provider"),
        supabase.from("internal_audit_sessions").select("id"),
        supabase.from("internal_audit_responses").select("response"),
        supabase.from("nc_corrective_actions").select("status,classification"),
        supabase.from("continuous_improvement_register").select("status,source_type"),
      ]);

      const lessonIds = (awarenessLessonsResponse.data ?? [])
        .filter((lesson) => lesson.is_active !== false)
        .map((lesson) => lesson.id);

      const lessonViewsResponse = userId && lessonIds.length
        ? await supabase
            .from("awareness_user_lesson_views")
            .select("lesson_id")
            .eq("user_id", userId)
            .in("lesson_id", lessonIds)
        : { data: [] };

      if (!isMounted) return;

      const riskRecords = (riskResponse.data ?? []) as RiskOpportunity[];
      const aspects = (aspectsResponse.data ?? []) as EnvironmentalAspect[];
      const legalRequirements = (legalResponse.data ?? []) as LegalRequirement[];
      const actions = (actionResponse.data ?? []) as ActionPlanItem[];
      const objectives = (objectivesResponse.data ?? []) as Objective[];
      const indicators = (
        (indicatorsResponse as { data: IndicatorRecord[] | null }).data ?? []
      ) as IndicatorRecord[];
      const resources = (resourcesResponse.data ?? []) as ResourceRecord[];
      const processes = (processesResponse.data ?? []) as ProcessRecord[];
      const competenceResults = (competenceResponse.data ?? []) as CompetenceResult[];
      const awarenessModules = (awarenessModulesResponse.data ?? [])
        .filter((module) => module.is_active !== false) as AwarenessModule[];
      const awarenessLessons = (awarenessLessonsResponse.data ?? [])
        .filter((lesson) => lesson.is_active !== false) as AwarenessLesson[];
      const awarenessProgress = (awarenessProgressResponse.data ?? []) as AwarenessProgress[];
      const awarenessViews = (lessonViewsResponse.data ?? []) as AwarenessLessonView[];
      const matrixRecords = (matrixResponse.data ?? []) as CommunicationMatrix[];
      const posts = (postsResponse.data ?? []) as CommunicationPost[];
      const documents = (documentsResponse.data ?? []) as DocumentControlRecord[];
      const revisions = (revisionsResponse.data ?? []) as DocumentRevision[];
      const suppliers = (suppliersResponse.data ?? []) as Supplier[];
      const evaluations = (evaluationsResponse.data ?? []) as SupplierEvaluation[];
      const rsuSalidas = (rsuResponse.data ?? []) as SalidaEntry[];
      const rmeSalidas = (rmeResponse.data ?? []) as SalidaEntry[];
      const rpSalidas = (rpResponse.data ?? []) as SalidaEntry[];
      const qirRecords = (qualityQirResponse.data ?? []).map((record) => ({
        lostCost: record.lost_cost ?? null,
      })) as QirRecord[];
      const stockRecords = (qualityStockResponse.data ?? []).map((record) => ({
        approxWeight: record.approx_weight ?? null,
      })) as StockRecord[];
      const scrapRecords = (qualityScrapResponse.data ?? []).map((record) => ({
        approxWeight: record.approx_weight ?? null,
      })) as ScrapRecord[];
      const energyRecords = (energyResponse.data ?? []).map((record) => ({
        generationKwh: record.generation_kwh ?? 0,
        consumptionKwh: record.consumption_kwh ?? 0,
        energyAcquiredCfeKwh: record.energy_acquired_cfe_kwh ?? 0,
        energySoldCfeKwh: record.energy_sold_cfe_kwh ?? 0,
        cfeBill: record.cfe_bill ?? 0,
      })) as EnergyRecord[];
      const waterRecords = (waterResponse.data ?? []).map((record) => ({
        date: record.date,
        month: record.month,
        year: record.year,
        quantityM3: record.quantity_m3 ?? 0,
        totalCost: record.total_cost ?? 0,
        type: record.type ?? null,        
      })) as WaterRecord[];
      const gasolineRecords = (gasolineResponse.data ?? []).map((record) => ({
        budget: record.budget ?? 0,
        equivalentLiters: record.equivalent_liters ?? 0,
        averageCostPerLiter: record.average_cost_per_liter ?? 0,
      })) as FuelGasolineRecord[];
      const gasRecords = (gasResponse.data ?? []).map((record) => ({
        totalLiters: record.total_liters ?? 0,
        totalCost: record.total_cost ?? 0,
        costUnit: record.cost_unit ?? null,
        quantity: record.quantity ?? 0,
        provider: record.provider ?? null,
      })) as FuelGasRecord[];
      const auditSessions = (auditSessionsResponse.data ?? []) as AuditSession[];
      const auditResponses = (auditResponsesResponse.data ?? []) as AuditResponse[];
      const ncRecords = (ncResponse.data ?? []) as NonconformityRecord[];
      const improvementRecords = (improvementResponse.data ?? []) as ContinuousImprovementRecord[];

      const riskResidualNotAcceptable = riskRecords.filter((record) => {
        const level = getLevel(record.p_residual, record.i_residual);
        const acceptable = getAcceptable(level);
        return acceptable === false;
      }).length;

      const riskCritical = riskRecords.filter((record) => {
        const level = getLevel(record.p_residual, record.i_residual);
        const classification = getClassification(level, record.i_residual);
        return classification === "critical";
      }).length;

      const riskLegal = riskRecords.filter((record) => Boolean(record.linked_to_legal)).length;

      const aspectTotal = aspects.length;
      const significant = aspects.filter((aspect) => computeExpectedValue(aspect) >= 10).length;
      const aspectIds = new Set(aspects.map((aspect) => aspect.id));
      const actionAspectIds = new Set(
        actions.filter((action) => action.source_type === "aspect" && action.source_id)
          .map((action) => action.source_id as string)
          .filter((id) => aspectIds.has(id)),
      );
      const aspectsWithAction = actionAspectIds.size;
      const aspectsActionPercent = aspectTotal
        ? (aspectsWithAction / aspectTotal) * 100
        : 0;

      const legalTotal = legalRequirements.length;
      const legalComplies = legalRequirements.filter(
        (item) => item.compliance_status === "complies",
      ).length;
      const legalNonComplies = legalRequirements.filter(
        (item) => item.compliance_status === "non_complies",
      ).length;
      const legalNotEvaluated = legalRequirements.filter(
        (item) => item.compliance_status === "not_evaluated",
      ).length;
      const legalOpenActions = legalRequirements.filter(
        (item) => item.compliance_status === "non_complies" && item.action_status !== "closed",
      ).length;

      const actionsWithStatus = actions.map((action) => ({
        ...action,
        derivedStatus: deriveStatus(action),
      }));
      const actionTotal = actionsWithStatus.length;
      const actionCompleted = actionsWithStatus.filter((action) => action.derivedStatus === "completed");
      const actionDelayed = actionsWithStatus.filter((action) => action.derivedStatus === "delayed");
      const actionWithoutOwner = actionsWithStatus.filter((action) => !action.owner);
      const actionCompletionRate = actionTotal
        ? Math.round((actionCompleted.length / actionTotal) * 100)
        : 0;

      const objectiveTotal = objectives.length;
      const objectivesCompliance = objectives.filter((objective) => objective.status === "Cumplido").length;
      const objectivesRisk = objectives.filter((objective) => objective.status === "Retrasado").length;
      const objectivesNotMet = objectives.filter((objective) => objective.status === "Cancelado").length;
      const objectiveCountermeasures = objectives.filter(
        (objective) => objective.countermeasures && objective.countermeasures.trim() !== "",
      ).length;

      const indicatorsTotal = indicators.length;
      const indicatorsUpdated = indicators.filter((record) => record.last_updated_at).length;
      const indicatorsUpdatedPercent = indicatorsTotal
        ? (indicatorsUpdated / indicatorsTotal) * 100
        : 0;
      const indicatorsOutOfTarget = indicators.filter((record) => {
        const summary = computeAnnualSummary(record);
        const match = formatAnnualTargetMatch(summary, record);
        return match === false;
      }).length;

      const resourcesTotal = resources.length;
      const resourceTypeCounts = resourceTypeOptions.map((option) => ({
        ...option,
        count: resources.filter((resource) => resource.resource_type === option.value).length,
      }));
      const topResourceType = resourceTypeCounts.sort((a, b) => b.count - a.count)[0];
      const resourceStatusCounts = resourceStatusOptions.map((option) => ({
        ...option,
        count: resources.filter((resource) => resource.status === option.value).length,
      }));
      const topResourceStatus = resourceStatusCounts.sort((a, b) => b.count - a.count)[0];
      const activeProcesses = processes.filter((process) => process.is_active ?? true);
      const totalProcesses = activeProcesses.length;
      const processesWithResources = new Set(
        resources.map((resource) => resource.process_activity).filter(Boolean) as string[],
      ).size;
      const criticalNotCurrent = resources.filter(
        (resource) => resource.is_critical && resource.status !== "current",
      ).length;

      const competenceScores = competenceResults.map((record) => {
        if (record.overall_score != null) return record.overall_score;
        const scores = competenceDimensions.map((dimension) =>
          toNumber(record[dimension.key]),
        );
        return scores.reduce((sum, value) => sum + value, 0) / scores.length;
      });
      const competenceScoreGlobal = competenceScores.length
        ? competenceScores.reduce((sum, value) => sum + value, 0) / competenceScores.length
        : 0;
      const dimensionAverages = competenceDimensions.map((dimension) => {
        const values = competenceResults.map((record) => toNumber(record[dimension.key]));
        const average = values.length
          ? values.reduce((sum, value) => sum + value, 0) / values.length
          : 0;
        return { ...dimension, average };
      });
      const lowestDimension = dimensionAverages.sort((a, b) => a.average - b.average)[0];

      const lessonCountByModule = awarenessLessons.reduce<Record<string, number>>((acc, lesson) => {
        acc[lesson.module_id] = (acc[lesson.module_id] ?? 0) + 1;
        return acc;
      }, {});

      const lessonLookup = awarenessLessons.reduce<Record<string, string>>((acc, lesson) => {
        acc[lesson.id] = lesson.module_id;
        return acc;
      }, {});

      const viewedCountByModule = awarenessViews.reduce<Record<string, number>>((acc, view) => {
        const moduleId = lessonLookup[view.lesson_id];
        if (!moduleId) return acc;
        acc[moduleId] = (acc[moduleId] ?? 0) + 1;
        return acc;
      }, {});

      const progressByModule = awarenessProgress.reduce<Record<string, AwarenessProgress>>((acc, record) => {
        acc[record.module_id] = record;
        return acc;
      }, {});

      const modulesWithStatus = awarenessModules.map((module) => {
        const totalLessons = lessonCountByModule[module.id] ?? 0;
        const viewedLessons = viewedCountByModule[module.id] ?? 0;
        const percent = totalLessons > 0 ? (viewedLessons / totalLessons) * 100 : 0;
        const progressRecord = progressByModule[module.id];
        const status = progressRecord?.status === "completed"
          ? "Completado"
          : viewedLessons > 0
          ? "En progreso"
          : "No iniciado";

        return {
          ...module,
          totalLessons,
          viewedLessons,
          progressPercent: percent,
          status,
          progressRecord,
        };
      });

      const availableModules = modulesWithStatus.length;
      const completedModules = modulesWithStatus.filter((module) => module.status === "Completado");
      const avgProgress = availableModules
        ? modulesWithStatus.reduce((total, module) => total + module.progressPercent, 0) /
          availableModules
        : 0;
      const weightedScoreNumerator = completedModules.reduce((total, module) => {
        const points = module.points_available ?? 0;
        const score = module.progressRecord?.final_score ?? 0;
        return total + score * points;
      }, 0);
      const weightedScoreDenominator = completedModules.reduce((total, module) => {
        return total + (module.points_available ?? 0);
      }, 0);
      const globalScore = weightedScoreDenominator
        ? weightedScoreNumerator / weightedScoreDenominator
        : 0;
      const pointsObtained = modulesWithStatus.reduce((total, module) => {
        return total + (module.progressRecord?.points_obtained ?? 0);
      }, 0);
      const pointsAvailable = modulesWithStatus.reduce((total, module) => {
        return total + (module.points_available ?? 0);
      }, 0);

      const matrixCount = matrixRecords.length;
      const now = new Date();
      const currentMonth = now.getMonth();
      const currentYear = now.getFullYear();
      const publishedPosts = posts.filter((post) => post.status === "published");
      const postsInPeriod = publishedPosts.filter((post) => {
        if (!post.publish_date) return false;
        const date = new Date(post.publish_date);
        if (Number.isNaN(date.getTime())) return false;
        return date.getMonth() === currentMonth && date.getFullYear() === currentYear;
      });
      const postCategoryCounts = publishedPosts.reduce<Record<string, number>>((acc, post) => {
        if (!post.category) return acc;
        acc[post.category] = (acc[post.category] ?? 0) + 1;
        return acc;
      }, {});
      const topPostCategory = Object.keys(postCategoryCounts).sort(
        (a, b) => (postCategoryCounts[b] ?? 0) - (postCategoryCounts[a] ?? 0),
      )[0];

      const classificationCounts = documents.reduce<Record<string, number>>((acc, doc) => {
        acc[doc.classification] = (acc[doc.classification] ?? 0) + 1;
        return acc;
      }, {});
      const topClassification = Object.keys(classificationCounts).sort(
        (a, b) => (classificationCounts[b] ?? 0) - (classificationCounts[a] ?? 0),
      )[0];
      const delayedDocuments = documents.filter(
        (doc) => computeDocumentStatus(doc.target_date, doc.start_date, doc.issue_date) === "Retrasado",
      ).length;
      const revisionsInPeriod = revisions.filter((revision) => {
        if (!revision.revision_date) return false;
        const date = new Date(revision.revision_date);
        if (Number.isNaN(date.getTime())) return false;
        return date.getMonth() === currentMonth && date.getFullYear() === currentYear;
      }).length;

      const evaluatedSuppliers = new Set(
        evaluations.map((evaluation) => evaluation.supplier_id).filter(Boolean),
      ).size;
      const rejectedEvaluations = evaluations.filter((evaluation) => evaluation.status === "REPROBADO").length;

      const sumQuantity = (entries: SalidaEntry[]) =>
        entries.reduce((acc, entry) => acc + toNumber(entry.cantidad), 0);

      const rsuTotal = sumQuantity(rsuSalidas);
      const rmeTotal = sumQuantity(rmeSalidas);
      const rpTotal = sumQuantity(rpSalidas);

      const qualityLostCost = qirRecords.reduce((sum, record) => sum + (record.lostCost ?? 0), 0);
      const stockWeight = stockRecords.reduce((sum, record) => sum + (record.approxWeight ?? 0), 0);
      const scrapWeight = scrapRecords.reduce((sum, record) => sum + (record.approxWeight ?? 0), 0);
      const totalWeight = stockWeight + scrapWeight;
      const stockPercent = totalWeight ? (stockWeight / totalWeight) * 100 : 0;
      const scrapPercent = totalWeight ? (scrapWeight / totalWeight) * 100 : 0;

      const energyTotals = energyRecords.reduce(
        (acc, record) => {
          acc.generation += record.generationKwh;
          acc.consumption += record.consumptionKwh;
          acc.acquired += record.energyAcquiredCfeKwh;
          acc.sold += record.energySoldCfeKwh;
          acc.cfeBill += record.cfeBill;
          return acc;
        },
        { generation: 0, consumption: 0, acquired: 0, sold: 0, cfeBill: 0 },
      );
      const coverageGross = safeDivide(energyTotals.generation, energyTotals.consumption);
      const coverageNet = safeDivide(energyTotals.generation - energyTotals.sold, energyTotals.consumption);
      const dependencyGross = safeDivide(energyTotals.acquired, energyTotals.consumption);
      const dependencyNet = safeDivide(energyTotals.acquired - energyTotals.sold, energyTotals.consumption);
      const selfSufficiency = coverageNet - dependencyNet;

      const waterTotalsByType = waterRecords.reduce(
        (acc, record) => {
          const typeKey = record.type === "RO" ? "ro" : "red";
          acc[typeKey] += record.quantityM3;
          return acc;
        },
        { red: 0, ro: 0 },
      );
      const waterTotalCostsByType = waterRecords.reduce(
        (acc, record) => {
          const typeKey = record.type === "RO" ? "ro" : "red";
          acc[typeKey] += record.totalCost;
          return acc;
        },
        { red: 0, ro: 0 },
      );
      const waterMonthlyTotalsByType = waterRecords.reduce(
        (acc, record) => {
          const typeKey = record.type === "RO" ? "ro" : "red";
          const key = `${record.year}-${record.month}`;
          acc[typeKey][key] = (acc[typeKey][key] ?? 0) + record.quantityM3;
          return acc;
        },
        { red: {} as Record<string, number>, ro: {} as Record<string, number> },
      );
        const waterMonthlyTotals = waterRecords.reduce<Record<string, number>>((acc, record) => {
        const key = `${record.year}-${record.month}`;
        acc[key] = (acc[key] ?? 0) + record.quantityM3;
        return acc;
      }, {});
      const buildWaterTrend = (totals: Record<string, number>) => {
        const months = Object.keys(totals).sort((a, b) => a.localeCompare(b));
        const lastKey = months[months.length - 1];
        const prevKey = months[months.length - 2];
        return lastKey && prevKey ? totals[lastKey] - totals[prevKey] : 0;
      };      
      const waterMonths = Object.keys(waterMonthlyTotals).sort((a, b) => a.localeCompare(b));
      const lastMonthKey = waterMonths[waterMonths.length - 1];
      const prevMonthKey = waterMonths[waterMonths.length - 2];
      const waterTrendValue = lastMonthKey && prevMonthKey
        ? waterMonthlyTotals[lastMonthKey] - waterMonthlyTotals[prevMonthKey]
        : 0;      
      const waterRedTrendValue = buildWaterTrend(waterMonthlyTotalsByType.red);
      const waterRoTrendValue = buildWaterTrend(waterMonthlyTotalsByType.ro);

      const gasolineBudget = gasolineRecords.reduce((sum, record) => sum + record.budget, 0);
      const gasolineLiters = gasolineRecords.reduce((sum, record) => sum + record.equivalentLiters, 0);
      const gasolineAvgCost = gasolineLiters
        ? gasolineBudget / gasolineLiters
        : gasolineRecords.reduce((sum, record) => sum + record.averageCostPerLiter, 0) /
          (gasolineRecords.length || 1);

      const gasTotalLiters = gasRecords.reduce((sum, record) => sum + record.totalLiters, 0);
      const gasTotalCost = gasRecords.reduce((sum, record) => sum + record.totalCost, 0);
      const providerCount = new Set(gasRecords.map((record) => record.provider).filter(Boolean)).size;

      const auditTotal = auditSessions.length;
      const classificationByResponse: Record<string, string> = {
        "Cumple correctamente": "Conformidad",
        "Cumple, pero puede mejorar": "Oportunidad de mejora",
        "Cumple, pero hay riesgo futuro": "Observación",
        "Cumple de forma destacada": "Buena práctica",
        "No cumple parcialmente": "No conformidad menor",
        "No cumple totalmente o hay riesgo grave": "No conformidad mayor",
      };
      const responseCounts = auditResponses.reduce<Record<string, number>>((acc, response) => {
        const classification = classificationByResponse[response.response];
        if (!classification) return acc;
        acc[classification] = (acc[classification] ?? 0) + 1;
        return acc;
      }, {});
      const observationCount = responseCounts["Observación"] ?? 0;
      const opportunityCount = responseCounts["Oportunidad de mejora"] ?? 0;
      const minorNc = responseCounts["No conformidad menor"] ?? 0;
      const majorNc = responseCounts["No conformidad mayor"] ?? 0;
      const totalResponses = Object.values(responseCounts).reduce((sum, value) => sum + value, 0);
      const compliantCount = Math.max(totalResponses - minorNc - majorNc, 0);
      const auditClosedPercent = totalResponses
        ? (compliantCount / totalResponses) * 100
        : 0;

      const ncTotal = ncRecords.length;
      const ncStatusCounts = ncRecords.reduce<Record<string, number>>((acc, record) => {
        const status = record.status ?? "Sin asignar";
        acc[status] = (acc[status] ?? 0) + 1;
        return acc;
      }, {});
      const topNcStatus = Object.keys(ncStatusCounts).sort(
        (a, b) => (ncStatusCounts[b] ?? 0) - (ncStatusCounts[a] ?? 0),
      )[0];
      const ncClassificationCounts = ncRecords.reduce<Record<string, number>>((acc, record) => {
        const classification = record.classification ?? "Sin clasificación";
        acc[classification] = (acc[classification] ?? 0) + 1;
        return acc;
      }, {});
      const topNcClassification = Object.keys(ncClassificationCounts).sort(
        (a, b) => (ncClassificationCounts[b] ?? 0) - (ncClassificationCounts[a] ?? 0),
      )[0];

      const improvementTotal = improvementRecords.length;
      const improvementClosed = improvementRecords.filter((record) => record.status === "closed").length;
      const improvementClosedPercent = improvementTotal
        ? (improvementClosed / improvementTotal) * 100
        : 0;
      const improvementSourceCounts = improvementRecords.reduce<Record<string, number>>((acc, record) => {
        if (!record.source_type) return acc;
        acc[record.source_type] = (acc[record.source_type] ?? 0) + 1;
        return acc;
      }, {});
      const topImprovementSource = Object.keys(improvementSourceCounts).sort(
        (a, b) => (improvementSourceCounts[b] ?? 0) - (improvementSourceCounts[a] ?? 0),
      )[0];

      setStats({
        riskTotal: numberFormatter.format(riskRecords.length),
        riskResidualNotAcceptable: numberFormatter.format(riskResidualNotAcceptable),
        riskCritical: numberFormatter.format(riskCritical),
        riskLegal: numberFormatter.format(riskLegal),
        aspectsTotal: numberFormatter.format(aspectTotal),
        aspectsSignificant: numberFormatter.format(significant),
        aspectsActionPercent: formatPercent(aspectsActionPercent),
        legalComplies: formatPercent(legalTotal ? (legalComplies / legalTotal) * 100 : 0),
        legalNonComplies: formatPercent(legalTotal ? (legalNonComplies / legalTotal) * 100 : 0),
        legalNotEvaluated: formatPercent(legalTotal ? (legalNotEvaluated / legalTotal) * 100 : 0),
        legalOpenActions: numberFormatter.format(legalOpenActions),
        actionPlanTotal: numberFormatter.format(actionTotal),
        actionPlanCompletion: formatPercent(actionCompletionRate, 1),
        actionPlanDelayed: numberFormatter.format(actionDelayed.length),
        actionPlanWithoutOwner: numberFormatter.format(actionWithoutOwner.length),
        objectivesTotal: numberFormatter.format(objectiveTotal),
        objectivesCompliance: formatPercent(objectiveTotal ? (objectivesCompliance / objectiveTotal) * 100 : 0),
        objectivesRisk: formatPercent(objectiveTotal ? (objectivesRisk / objectiveTotal) * 100 : 0),
        objectivesNotMet: formatPercent(objectiveTotal ? (objectivesNotMet / objectiveTotal) * 100 : 0),
        objectivesCountermeasures: numberFormatter.format(objectiveCountermeasures),
        indicatorsUpdatedPercent: formatPercent(indicatorsUpdatedPercent, 1),
        indicatorsOutOfTarget: numberFormatter.format(indicatorsOutOfTarget),
        resourcesDistribution: topResourceType
          ? `${topResourceType.label} ${formatPercent(resourcesTotal ? (topResourceType.count / resourcesTotal) * 100 : 0, 0)}`
          : "-",
        resourcesStatus: topResourceStatus
          ? `${topResourceStatus.label} ${formatPercent(resourcesTotal ? (topResourceStatus.count / resourcesTotal) * 100 : 0, 0)}`
          : "-",
        resourcesCoverage: `${processesWithResources}/${totalProcesses}`,
        resourcesCriticalWithoutBackup: numberFormatter.format(criticalNotCurrent),
        competenceScoreGlobal: numberFormatter.format(competenceScoreGlobal),
        competenceMainGap: lowestDimension ? lowestDimension.label : "-",
        awarenessModulesCompleted: `${completedModules.length} / ${availableModules}`,
        awarenessProgress: formatPercent(avgProgress, 1),
        awarenessScore: numberFormatter.format(globalScore),
        awarenessPoints: `${numberFormatter.format(pointsObtained)} / ${numberFormatter.format(pointsAvailable)}`,
        communicationMatrixItems: numberFormatter.format(matrixCount),
        communicationPostsPeriod: numberFormatter.format(postsInPeriod.length),
        communicationTopCategory: topPostCategory ?? "-",
        documentControlClassification: documentClassificationLabels[topClassification] ?? topClassification ?? "-",
        documentControlDelayed: numberFormatter.format(delayedDocuments),
        documentControlRevisionsPeriod: numberFormatter.format(revisionsInPeriod),
        externalEvaluated: `${evaluatedSuppliers} / ${suppliers.length}`,
        externalCurrent: numberFormatter.format(suppliers.length),
        externalLinkedNc: numberFormatter.format(rejectedEvaluations),
        rsuTotal: numberFormatter.format(rsuTotal),
        rsuIncidents: numberFormatter.format(rsuSalidas.length),
        rmeTotal: numberFormatter.format(rmeTotal),
        rmeIncidents: numberFormatter.format(rmeSalidas.length),
        rpTotal: numberFormatter.format(rpTotal),
        rpIncidents: numberFormatter.format(rpSalidas.length),
        qualityQir: numberFormatter.format(qirRecords.length),
        qualityStockPercent: formatPercent(stockPercent, 1),
        qualityScrapPercent: formatPercent(scrapPercent, 1),
        qualityStockScrap: `${numberFormatter.format(stockWeight)} / ${numberFormatter.format(scrapWeight)}`,
        qualityLostCost: currencyFormatter.format(qualityLostCost),
        qualityScrapVolume: `${numberFormatter.format(scrapWeight)} kg`,
        energySelfSufficiency: formatPercent(selfSufficiency * 100, 1),
        energyDependency: `${formatPercent(dependencyGross * 100, 1)} / ${formatPercent(dependencyNet * 100, 1)}`,
        energyBill: currencyFormatter.format(energyTotals.cfeBill),
        energyGeneratedConsumed: `${numberFormatter.format(energyTotals.generation)} kWh / ${numberFormatter.format(energyTotals.consumption)} kWh`,
        waterRedConsumption: `${numberFormatter.format(waterTotalsByType.red)} m³`,
        waterRoConsumption: `${numberFormatter.format(waterTotalsByType.ro)} m³`,
        waterTrend: `${numberFormatter.format(waterTrendValue)} m³`,
        waterRedTrend: `${numberFormatter.format(waterRedTrendValue)} m³`,
        waterRoTrend: `${numberFormatter.format(waterRoTrendValue)} m³`,
        waterRedCost: currencyFormatter.format(waterTotalCostsByType.red),
        waterRoCost: currencyFormatter.format(waterTotalCostsByType.ro),
        fuelsGasolineLiters: numberFormatter.format(gasolineLiters),
        fuelsGasolineSpend: currencyFormatter.format(gasolineBudget),
        fuelsGasolineAvgCost: currencyFormatter.format(gasolineAvgCost),
        fuelsGasLiters: numberFormatter.format(gasTotalLiters),
        fuelsGasSpend: currencyFormatter.format(gasTotalCost),
        fuelsSuppliers: numberFormatter.format(providerCount),
        auditTotal: numberFormatter.format(auditTotal),
        auditFindings: `${numberFormatter.format(opportunityCount)} / ${numberFormatter.format(observationCount)}`,
        auditNcMajorMinor: `${numberFormatter.format(minorNc)} / ${numberFormatter.format(majorNc)}`,
        auditClosedPercent: formatPercent(auditClosedPercent, 1),
        ncRecords: numberFormatter.format(ncTotal),
        ncStatus: topNcStatus ?? "-",
        ncClassification: topNcClassification ?? "-",
        improvementTotal: numberFormatter.format(improvementTotal),
        improvementClosedPercent: formatPercent(improvementClosedPercent, 1),
        improvementTopSource: topImprovementSource ?? "-",
      });

      setIsLoading(false);
    };

    loadStats();

    return () => {
      isMounted = false;
    };
  }, [currencyFormatter, numberFormatter]);

  const sections = [
    {
      id: "planning",
      title: "Planeación",
      description: "Riesgos, aspectos, legales y objetivos.",
      href: "/meiwa-core/meiwa-sgi/planning",
      modules: [
        {
          title: "Riesgos y oportunidades",
          href: "/meiwa-core/meiwa-sgi/planning/risk-and-opportunities",
          kpi: { label: "Residual no aceptable", value: stats.riskResidualNotAcceptable },          
          stats: [
            { label: "Total", value: stats.riskTotal },
            { label: "Residual no aceptable", value: stats.riskResidualNotAcceptable },
            { label: "Críticos", value: stats.riskCritical },
            { label: "Ligados a legal", value: stats.riskLegal },
          ],
        },
        {
          title: "Aspectos e impactos",
          href: "/meiwa-core/meiwa-sgi/planning/aspects-impacts",
          kpi: { label: "% significativos", value: "100%" },
          stats: [
            { label: "Aspectos totales", value: stats.aspectsTotal },
            { label: "Significativos", value: stats.aspectsSignificant },
            { label: "% con acción", value: stats.aspectsActionPercent },
          ],
        },
        {
          title: "Requisitos legales",
          href: "/meiwa-core/meiwa-sgi/planning/legal-requirements",
          kpi: { label: "% cumple", value: stats.legalComplies },
          stats: [
            { label: "% Cumple", value: stats.legalComplies },
            { label: "% No cumple", value: stats.legalNonComplies },
            { label: "% No evaluado", value: stats.legalNotEvaluated },
            { label: "Acciones abiertas", value: stats.legalOpenActions },
          ],
        },
        {
          title: "Plan de acciones (6.1.4)",
          href: "/meiwa-core/meiwa-sgi/planning/action-plan",
          kpi: { label: "% completas", value: stats.actionPlanCompletion },
          stats: [
            { label: "Acciones", value: stats.actionPlanTotal },
            { label: "% completas", value: stats.actionPlanCompletion },
            { label: "Retrasadas", value: stats.actionPlanDelayed },
            { label: "Sin responsable", value: stats.actionPlanWithoutOwner },
          ],
        },
        {
          title: "Objetivos ambientales",
          href: "/meiwa-core/meiwa-sgi/planning/environmental-objectives",
          kpi: { label: "% en cumplimiento", value: stats.objectivesCompliance },
          stats: [
            { label: "Objetivos", value: stats.objectivesTotal },
            { label: "% en cumplimiento", value: stats.objectivesCompliance },
            { label: "% en riesgo", value: stats.objectivesRisk },
            { label: "% no cumplidos", value: stats.objectivesNotMet },
            { label: "Contramedidas", value: stats.objectivesCountermeasures },
          ],
        },
        {
          title: "Indicadores ambientales",
          href: "/meiwa-core/meiwa-sgi/planning/environmental-indicators",
          kpi: { label: "% actualizados", value: stats.indicatorsUpdatedPercent },
          stats: [
            { label: "% actualizados", value: stats.indicatorsUpdatedPercent },
            { label: "Fuera de meta", value: stats.indicatorsOutOfTarget },
          ],
        },
      ],
    },
    {
      id: "support",
      title: "Soporte",
      description: "Recursos, competencia y comunicación.",
      href: "/meiwa-core/meiwa-sgi/support",
      modules: [
        {
          title: "Recursos",
          href: "/meiwa-core/meiwa-sgi/support/resources",
          kpi: { label: "Cobertura", value: stats.resourcesCoverage },
          stats: [
            { label: "Distribución", value: stats.resourcesDistribution },
            { label: "Estado", value: stats.resourcesStatus },
            { label: "Cobertura", value: stats.resourcesCoverage },
            { label: "Críticos sin respaldo", value: stats.resourcesCriticalWithoutBackup },
          ],
        },
        {
          title: "Competencia",
          href: "/meiwa-core/meiwa-sgi/support/competence",
          kpi: { label: "Score global", value: stats.competenceScoreGlobal },
          stats: [
            { label: "Score global", value: stats.competenceScoreGlobal },
            { label: "Brecha principal", value: stats.competenceMainGap },
          ],
        },
        {
          title: "Toma de conciencia",
          href: "/meiwa-core/meiwa-sgi/support/awareness",
          kpi: { label: "Calificación global", value: stats.awarenessScore },
          stats: [
            { label: "Módulos completados", value: stats.awarenessModulesCompleted },
            { label: "% avance", value: stats.awarenessProgress },
            { label: "Calificación", value: stats.awarenessScore },
            { label: "Puntos", value: stats.awarenessPoints },
          ],
        },
        {
          title: "Comunicación",
          href: "/meiwa-core/meiwa-sgi/support/comunication",
          kpi: { label: "Comunicados en periodo", value: stats.communicationPostsPeriod },
          stats: [
            { label: "Items en matriz", value: stats.communicationMatrixItems },
            { label: "Comunicados en periodo", value: stats.communicationPostsPeriod },
            { label: "Top categoría", value: stats.communicationTopCategory },
          ],
        },
        {
          title: "Control documental",
          href: "/meiwa-core/meiwa-sgi/support/document-control",
          kpi: { label: "Revisiones en periodo", value: stats.documentControlRevisionsPeriod },
          stats: [
            { label: "Clasificación", value: stats.documentControlClassification },
            { label: "Retrasados", value: stats.documentControlDelayed },
            { label: "Revisiones en periodo", value: stats.documentControlRevisionsPeriod },
          ],
        },
      ],
    },
    {
      id: "operation",
      title: "Operación",
      description: "Servicios, residuos, energía y calidad.",
      href: "/meiwa-core/meiwa-sgi/operation",
      modules: [
        {
          title: "Servicios externalizados",
          href: "/meiwa-core/meiwa-sgi/operation/external",
          kpi: { label: "Evaluados / vigentes", value: stats.externalEvaluated },
          stats: [
            { label: "Evaluados / vigentes", value: stats.externalEvaluated },
            { label: "NC ligadas", value: stats.externalLinkedNc },
          ],
        },
        {
          title: "RSU",
          href: "/meiwa-core/meiwa-sgi/operation/rsu",
          kpi: { label: "Total periodo (kg)", value: stats.rsuTotal },
          stats: [
            { label: "Total periodo (kg)", value: stats.rsuTotal },
            { label: "Incidencias", value: stats.rsuIncidents },
          ],
        },
        {
          title: "RME",
          href: "/meiwa-core/meiwa-sgi/operation/rme",
          kpi: { label: "Total periodo (kg)", value: stats.rmeTotal },
          stats: [
            { label: "Total periodo (kg)", value: stats.rmeTotal },
            { label: "Incidencias", value: stats.rmeIncidents },
          ],
        },
        {
          title: "Residuos peligrosos (RP)",
          href: "/meiwa-core/meiwa-sgi/operation/rp",
          kpi: { label: "Total periodo (kg)", value: stats.rpTotal },
          stats: [
            { label: "Total periodo (kg)", value: stats.rpTotal },
            { label: "Incidencias", value: stats.rpIncidents },
          ],
        },
        {
          title: "Calidad (QIR)",
          href: "/meiwa-core/meiwa-sgi/operation/quality",
          kpi: { label: "Volumen en scrap", value: stats.qualityScrapVolume },
          stats: [
            { label: "QIR periodo", value: stats.qualityQir },
            { label: "% stock", value: stats.qualityStockPercent },
            { label: "% scrap", value: stats.qualityScrapPercent },
            { label: "Stock vs scrap (kg)", value: stats.qualityStockScrap },
            { label: "Costo perdido", value: stats.qualityLostCost },
          ],
        },
        {
          title: "Energía",
          href: "/meiwa-core/meiwa-sgi/operation/energy",
          kpi: { label: "Autosuficiencia energética", value: stats.energySelfSufficiency},
          stats: [
            { label: "Autosuficiencia energética", value: stats.energySelfSufficiency },            
            { label: "Dependencia CFE bruta/neta", value: stats.energyDependency },
            { label: "Factura CFE", value: stats.energyBill },
            { label: "Generada/consumida", value: stats.energyGeneratedConsumed },
          ],
        },
        {
          title: "Agua de Red",
          href: "/meiwa-core/meiwa-sgi/operation/water",
          kpi: { label: "Consumo total", value: stats.waterRedConsumption },
          stats: [
            { label: "Consumo total", value: stats.waterRedConsumption },
            { label: "Tendencia mensual", value: stats.waterRedTrend },
            { label: "Costo total", value: stats.waterRedCost },
          ],
        },
        {
          title: "Agua de RO",
          href: "/meiwa-core/meiwa-sgi/operation/water",
          kpi: { label: "Consumo total", value: stats.waterRoConsumption },
          stats: [
            { label: "Consumo total", value: stats.waterRoConsumption },
            { label: "Tendencia mensual", value: stats.waterRoTrend },
            { label: "Costo total", value: stats.waterRoCost },
          ],
        },
        {
          title: "Combustibles",
          href: "/meiwa-core/meiwa-sgi/operation/fuels",
          kpi: { label: "Gasolina litros", value: stats.fuelsGasolineLiters },
          stats: [
            { label: "Gasolina litros", value: stats.fuelsGasolineLiters },
            { label: "Gasto gasolina", value: stats.fuelsGasolineSpend },
            { label: "Costo prom/L", value: stats.fuelsGasolineAvgCost },
            { label: "Gases litros", value: stats.fuelsGasLiters },
            { label: "Gasto gases", value: stats.fuelsGasSpend },
            { label: "Proveedores", value: stats.fuelsSuppliers },
          ],
        },
      ],
    },
    {
      id: "evaluation",
      title: "Evaluación",
      description: "Auditoría interna y hallazgos.",
      href: "/meiwa-core/meiwa-sgi/evaluation",
      modules: [
        {
          title: "Auditoría interna",
          href: "/meiwa-core/meiwa-sgi/evaluation/internal-audit",
          kpi: { label: "% Cumplimiento global", value: stats.auditClosedPercent },
          stats: [
            { label: "Auditorías", value: stats.auditTotal },
            { label: "Hallazgos (Opp/Obs)", value: stats.auditFindings },
            { label: "NC menor/mayor", value: stats.auditNcMajorMinor },
            { label: "% cerrados", value: stats.auditClosedPercent },
          ],
        },
      ],
    },
    {
      id: "improvement",
      title: "Mejora",
      description: "NC/AC y mejora continua.",
      href: "/meiwa-core/meiwa-sgi/improvement",
      modules: [
        {
          title: "NC / Acción correctiva",
          href: "/meiwa-core/meiwa-sgi/improvement/nonconformity-corrective-action",
          kpi: { label: "No conformidades", value: stats.ncRecords },
          stats: [
            { label: "Registros", value: stats.ncRecords },
            { label: "Por estado", value: stats.ncStatus },
            { label: "Por clasificación", value: stats.ncClassification },
          ],
        },
        {
          title: "Mejora continua (10.3)",
          href: "/meiwa-core/meiwa-sgi/improvement/continuous-improvement",
          kpi: { label: "% cerradas", value: stats.improvementClosedPercent },
          stats: [
            { label: "Mejoras", value: stats.improvementTotal },
            { label: "% cerradas", value: stats.improvementClosedPercent },
            { label: "Fuentes", value: stats.improvementTopSource },
          ],
        },
      ],
    },
  ];


  const moduleHighlights = useMemo(
    () =>
      sections.flatMap((section) =>
        section.modules.map((module) => ({
          sectionTitle: section.title,
          ...module,
        })),
      ),
    [sections],
  );

  const moduleOptions = useMemo(
    () =>
      Array.from(new Set(moduleHighlights.map((module) => module.title))).sort((a, b) =>
        a.localeCompare(b),
      ),
    [moduleHighlights],
  );

  const normalizedSearch = searchTerm.trim().toLowerCase();
  const matchesModuleFilters = (module: { title: string }, sectionTitle: string) => {
    if (moduleFilter !== "all" && module.title !== moduleFilter) return false;
    if (!normalizedSearch) return true;
    return `${module.title} ${sectionTitle}`.toLowerCase().includes(normalizedSearch);
  };

  const filteredModuleHighlights = useMemo(
    () =>
      moduleHighlights.filter((module) => matchesModuleFilters(module, module.sectionTitle)),
    [moduleHighlights, moduleFilter, normalizedSearch],
  );

  const filteredSections = useMemo(
    () =>
      sections
        .map((section) => ({
          ...section,
          modules: section.modules.filter((module) => matchesModuleFilters(module, section.title)),
        }))
        .filter((section) => section.modules.length > 0),
    [sections, moduleFilter, normalizedSearch],
  );

  const sparklineTemplate = [0.4, 0.6, 0.8, 0.7, 0.9];

  const parseNumericValue = (value: string) => {
    const match = value.match(/-?[\d,.]+/);
    if (!match) return null;
    const numeric = Number.parseFloat(match[0].replace(/,/g, ""));
    return Number.isNaN(numeric) ? null : numeric;
  };

  const buildSparklineHeights = (value: string) => {
    const numericValue = parseNumericValue(value);
    const normalized = numericValue === null ? 60 : Math.min(100, Math.max(10, numericValue));
    return sparklineTemplate.map((multiplier) =>
      Math.round(Math.min(100, Math.max(18, normalized * multiplier))),
    );
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              Sistema de Gestión Integral
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Seguimiento, Medición, Análisis, Evaluación
            </h1>
            <p className="max-w-4xl text-base text-gray-600 sm:text-lg">
            Consolida los indicadores clave de Planeación, Soporte, Operación, Evaluación y Mejora
            en un solo tablero para una lectura rápida del estado del SGI.
            </p>
          {isLoading && (
            <p className="mt-4 text-sm text-slate-500">
              Actualizando indicadores...
            </p>
          )}            
          </div>
        </section>        

        <Tabs
          value={viewMode}
          onValueChange={(value) => setViewMode(value as "panorama" | "modules")}
          className="space-y-8"
        >
          <section className="space-y-5 rounded-2xl border border-slate-200 bg-white p-6 shadow-sm">
            <div className="flex flex-col gap-4 lg:flex-row lg:items-end lg:justify-between">
              <div className="space-y-1">
                <h2 className="text-xl font-semibold text-slate-900">
                  Filtros y vistas
                </h2>
                <p className="text-sm text-slate-600">
                  Filtra por módulo o búsqueda rápida y alterna entre panorama y vista por módulos.
                </p>
              </div>
              <TabsList className="grid w-full grid-cols-2 lg:w-[260px]">
                <TabsTrigger value="panorama">Panorama</TabsTrigger>
                <TabsTrigger value="modules">Módulos</TabsTrigger>
              </TabsList>
            </div>
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <p className="text-sm font-medium text-slate-700">Búsqueda</p>
                <Input
                  value={searchTerm}
                  onChange={(event) => setSearchTerm(event.target.value)}
                  placeholder="Buscar por módulo o sección"
                />
              </div>
              <div className="space-y-2">
                <p className="text-sm font-medium text-slate-700">Módulo</p>
                <Select value={moduleFilter} onValueChange={setModuleFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todos los módulos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos los módulos</SelectItem>
                    {moduleOptions.map((option) => (
                      <SelectItem key={option} value={option}>
                        {option}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </section>            

          <TabsContent value="panorama" className="space-y-4">
            <div className="flex flex-col gap-2">
              <h2 className="text-2xl font-semibold text-slate-900">
                Panorama de módulos
              </h2>
              <p className="text-sm text-slate-600">
                Un KPI clave por módulo con tendencia visual para identificar prioridades.
              </p>
            </div>
            {filteredModuleHighlights.length ? (
              <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-3">
                {filteredModuleHighlights.map((module) => {
                  const sparklineHeights = buildSparklineHeights(module.kpi.value);

                  return (
                    <Card key={module.title} className="border-slate-200">
                      <CardContent className="flex h-full flex-col gap-4 p-5">
                        <div className="flex items-start justify-between gap-4">
                          <div>
                            <p className="text-xs font-semibold uppercase tracking-wide text-emerald-600">
                              {module.sectionTitle}
                            </p>
                            <h3 className="text-lg font-semibold text-slate-900">
                              {module.title}
                            </h3>
                          </div>
                          <div className="flex items-end gap-1">
                            {sparklineHeights.map((height, index) => (
                              <span
                                key={`${module.title}-spark-${index}`}
                                className="w-2 rounded-full bg-emerald-500/80"
                                style={{ height: `${height}%` }}
                              />
                            ))}
                          </div>
                        </div>
                        <div className="rounded-xl bg-slate-50 p-3">
                          <p className="text-xs font-semibold uppercase tracking-wide text-slate-500">
                            {module.kpi.label}
                          </p>
                          <p className="text-xl font-semibold text-slate-900">
                            {module.kpi.value}
                          </p>
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            ) : (
              <Card className="border-slate-200">
                <CardContent className="p-6 text-sm text-slate-600">
                  No hay módulos que coincidan con los filtros actuales.
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="modules" className="space-y-10">
            {filteredSections.length ? (
              filteredSections.map((section) => (
                <section key={section.id} className="space-y-6">
                  <div className="flex flex-col gap-4 rounded-2xl border border-slate-100 bg-slate-50 px-6 py-4 sm:flex-row sm:items-center sm:justify-between">
                    <div>
                      <h2 className="text-2xl font-semibold text-slate-900">{section.title}</h2>
                      <p className="text-sm text-slate-600">{section.description}</p>
                    </div>
                    <Button asChild variant="outline" size="sm">
                      <Link href={section.href}>Ver módulos</Link>
                    </Button>
                  </div>

                  <div className="grid gap-6 lg:grid-cols-2">
                    {section.modules.map((module) => (
                      <Card key={module.title} className="border-slate-200">
                        <CardHeader>
                          <CardTitle className="text-xl text-slate-900">{module.title}</CardTitle>
                        </CardHeader>
                        <CardContent>
                          <dl className="grid gap-4 sm:grid-cols-2">
                            {module.stats.map((stat) => (
                              <div key={stat.label} className="space-y-1 rounded-lg bg-slate-50 p-3">
                                <dt className="text-xs font-semibold uppercase tracking-wide text-slate-500">
                                  {stat.label}
                                </dt>
                                <dd className="text-lg font-semibold text-slate-900">{stat.value}</dd>
                              </div>
                            ))}
                          </dl>
                          <div className="mt-6 flex flex-wrap gap-3">
                            <Button asChild variant="secondary" size="sm">
                              <Link href={module.href}>Ver detalle</Link>
                            </Button>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </section>
              ))
            ) : (
              <Card className="border-slate-200">
                <CardContent className="p-6 text-sm text-slate-600">
                  No hay módulos que coincidan con los filtros actuales.
                </CardContent>
              </Card>
            )}
          </TabsContent>
        </Tabs>
      </div>
    </SGIDashboardLayout>
  );
}