"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import type { User } from "@supabase/supabase-js";
import {
  CalendarDays,
  ClipboardList,
  FileSearch,
  Filter,
  ListChecks,
  Plus,
  RefreshCcw,
  ShieldCheck,
} from "lucide-react";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

const classificationOptions = [
  "Oportunidad de mejora",
  "Observación",
  "No conformidad menor",
  "No conformidad mayor",
];

const statusOptions = [
  "COMPLETO",
  "RETRASADO",
  "EN PROCESO",
  "PLANEADO",
  "SIN ASIGNAR",
  "REABIERTO",  
] as const;

type StatusOption = (typeof statusOptions)[number];

const resultOptions = ["Conformidad", "No conformidad"] as const;

type ResultOption = (typeof resultOptions)[number];

type AuditResponse = {
  id: string;
  session_id: string;
  section_id: string | null;
  group_title: string | null;
  question: string | null;
  response: string | null;
  classification: string | null;
  comment: string | null;
  created_at?: string | null;
};

type AuditSession = {
  id: string;
  audit_date: string | null;
  auditors: string[] | null;
};

type NcCorrectiveAction = {
  id: string;
  source: "audit" | "manual";
  audit_session_id: string | null;
  audit_response_id: string | null;
  created_at: string | null;
  category: string | null;
  auditors: string | string[] | null;
  question: string | null;
  answer: string | null;
  classification: string | null;
  comments: string | null;
  root_cause_analysis: string | null;
  action_plan: string | null;
  implementation_owner: string | null;
  target_implementation_date: string | null;
  start_date: string | null;
  end_date: string | null;
  evidence_url: string | null;
  result: ResultOption | null;
  result_comment: string | null;
  review_date: string | null;
  status: StatusOption | null;
  updated_at: string | null;
  updated_by: string | null;
};

type FormState = {
  category: string;
  auditors: string;
  question: string;
  answer: string;
  classification: string;
  comments: string;
  root_cause_analysis: string;
  action_plan: string;
  implementation_owner: string;
  target_implementation_date: string;
  start_date: string;
  end_date: string;
  evidence_url: string;
  result: ResultOption | "";
  result_comment: string;
  review_date: string;
};

const buildEmptyFormState = (): FormState => ({
  category: "",
  auditors: "",
  question: "",
  answer: "",
  classification: "",
  comments: "",
  root_cause_analysis: "",
  action_plan: "",
  implementation_owner: "",
  target_implementation_date: "",
  start_date: "",
  end_date: "",
  evidence_url: "",
  result: "",
  result_comment: "",
  review_date: "",
});

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return "Sin fecha";
  return parsed.toLocaleDateString("es-MX");
};

const formatDateInput = (value?: string | null) => {
  if (!value) return "";
  return new Date(value).toISOString().split("T")[0] ?? "";
};

const normalizeDate = (value?: string | null) => {
  if (!value) return null;
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return null;
  parsed.setHours(0, 0, 0, 0);
  return parsed;
};

const getStatusBadge = (status: StatusOption) => {
  const styles: Record<StatusOption, string> = {
    COMPLETO: "bg-emerald-100 text-emerald-700",
    RETRASADO: "bg-red-100 text-red-700",
    "EN PROCESO": "bg-amber-100 text-amber-700",
    PLANEADO: "bg-blue-100 text-blue-700",
    "SIN ASIGNAR": "bg-slate-200 text-slate-700",
    REABIERTO: "bg-orange-100 text-orange-700",
  };
  return styles[status] ?? "bg-slate-100 text-slate-700";
};

const getResultBadge = (result: ResultOption | null) => {
  if (result === "Conformidad") return "bg-emerald-100 text-emerald-700";
  if (result === "No conformidad") return "bg-red-100 text-red-700";
  return "bg-slate-100 text-slate-700";
};

const getClassificationBadge = (classification: string | null) => {
  if (!classification) return "bg-slate-100 text-slate-700";
  if (classification.includes("mayor")) return "bg-red-100 text-red-700";
  if (classification.includes("menor")) return "bg-amber-100 text-amber-700";
  if (classification.includes("Observación")) return "bg-blue-100 text-blue-700";
  return "bg-emerald-100 text-emerald-700";
};

const formatAuditors = (auditors: string | string[] | null) => {
  if (!auditors) return "N/A";
  if (Array.isArray(auditors)) return auditors.filter(Boolean).join(", ");
  if (auditors.trim() === "") return "N/A";
  return auditors;
};

const computeStatus = (payload: {
  implementation_owner?: string | null;
  target_implementation_date?: string | null;
  start_date?: string | null;
  end_date?: string | null;
  result?: ResultOption | null;
}): StatusOption => {
  if (payload.result === "No conformidad") return "REABIERTO";
  if (payload.end_date) return "COMPLETO";
  const today = normalizeDate(new Date().toISOString());
  const targetDate = normalizeDate(payload.target_implementation_date ?? null);
  if (targetDate && today && today > targetDate) return "RETRASADO";
  if (payload.start_date) return "EN PROCESO";
  if (payload.implementation_owner || payload.target_implementation_date) return "PLANEADO";
  return "SIN ASIGNAR";
};

export default function NonconformityCorrectiveActionPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [records, setRecords] = useState<NcCorrectiveAction[]>([]);
  const [searchText, setSearchText] = useState("");
  const [classificationFilter, setClassificationFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [ownerFilter, setOwnerFilter] = useState("all");
  const [categoryFilter, setCategoryFilter] = useState("all");
  const [createdFrom, setCreatedFrom] = useState("");
  const [createdTo, setCreatedTo] = useState("");
  const [targetFrom, setTargetFrom] = useState("");
  const [targetTo, setTargetTo] = useState("");
  const [endFrom, setEndFrom] = useState("");
  const [endTo, setEndTo] = useState("");
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<"new" | "edit">("new");
  const [activeRecord, setActiveRecord] = useState<NcCorrectiveAction | null>(null);
  const [formState, setFormState] = useState<FormState>(buildEmptyFormState());
  const [saving, setSaving] = useState(false);

  const canManage = allowedManagers.has(user?.email ?? "");

  const syncAuditFindings = useCallback(async () => {
    const { data: auditResponses, error } = await supabase
      .from("internal_audit_responses")
      .select(
        "id, session_id, section_id, group_title, question, response, classification, comment, created_at"
      )
      .in("classification", classificationOptions);

    if (error) {
      toast.error("No pudimos sincronizar los hallazgos de auditoría.");
      return;
    }

    if (!auditResponses || auditResponses.length === 0) return;

    const responseIds = auditResponses.map((response) => response.id);
    const sessionIds = Array.from(
      new Set(auditResponses.map((response) => response.session_id))
    );

    const { data: sessionsData, error: sessionsError } = await supabase
      .from("internal_audit_sessions")
      .select("id, audit_date, auditors")
      .in("id", sessionIds);

    if (sessionsError) {
      toast.error("No pudimos leer las sesiones de auditoría.");
      return;
    }

    const sessionsMap = new Map<string, AuditSession>();
    (sessionsData ?? []).forEach((session) => sessionsMap.set(session.id, session));

    const { data: existingActions, error: actionsError } = await supabase
      .from("nc_corrective_actions")
      .select("audit_response_id")
      .in("audit_response_id", responseIds);

    if (actionsError) {
      toast.error("No pudimos validar los registros existentes.");
      return;
    }

    const existingIds = new Set(
      (existingActions ?? [])
        .map((record) => record.audit_response_id)
        .filter((value): value is string => Boolean(value))
    );

    const payload = (auditResponses as AuditResponse[])
      .filter((response) => !existingIds.has(response.id))
      .map((response) => {
        const session = sessionsMap.get(response.session_id);
        const createdAt = response.created_at ?? session?.audit_date ?? null;

        return {
          source: "audit" as const,
          audit_session_id: response.session_id,
          audit_response_id: response.id,
          created_at: createdAt,
          category: response.group_title ?? response.section_id ?? "Sin categoría",
          auditors: session?.auditors?.join(", ") ?? "N/A",
          question: response.question ?? "",
          answer: response.response ?? "",
          classification: response.classification ?? "",
          comments: response.comment ?? "",
          root_cause_analysis: "",
          action_plan: "",
          implementation_owner: "",
          target_implementation_date: null,
          start_date: null,
          end_date: null,
          evidence_url: null,
          result: null,
          result_comment: null,
          review_date: null,
          status: "SIN ASIGNAR" as StatusOption,
        };
      });

    if (payload.length === 0) return;

    const { error: insertError } = await supabase
      .from("nc_corrective_actions")
      .insert(payload);

    if (insertError) {
      toast.error("No pudimos importar los hallazgos de auditoría.");
      return;
    }
  }, []);

  const fetchRecords = useCallback(async () => {
    const { data, error } = await supabase
      .from("nc_corrective_actions")
      .select("*")
      .order("created_at", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar los hallazgos.");
      return;
    }

    const normalized = (data ?? []).map((record: NcCorrectiveAction) => {
      const computedStatus = computeStatus({
        implementation_owner: record.implementation_owner,
        target_implementation_date: record.target_implementation_date,
        start_date: record.start_date,
        end_date: record.end_date,
        result: record.result,
      });

      return {
        ...record,
        status: computedStatus,
      };
    });

    setRecords(normalized);
  }, []);

  const loadSession = useCallback(async () => {
    setLoading(true);
    const { data } = await supabase.auth.getSession();
    const currentUser = data.session?.user ?? null;
    if (!currentUser) {
      router.push("/");
      return;
    }

    setUser(currentUser);
    await syncAuditFindings();
    await fetchRecords();
    setLoading(false);
  }, [fetchRecords, router, syncAuditFindings]);

  useEffect(() => {
    loadSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      const nextUser = session?.user ?? null;
      setUser(nextUser);
      if (!nextUser) {
        router.push("/");
      }
    });

    return () => {
      subscription.unsubscribe();
    };
  }, [loadSession, router]);

  const uniqueCategories = useMemo(() => {
    const values = new Set(
      records
        .map((record) => record.category)
        .filter((value): value is string => Boolean(value))
    );
    return ["all", ...Array.from(values)];
  }, [records]);

  const uniqueOwners = useMemo(() => {
    const values = new Set(
      records
        .map((record) => record.implementation_owner)
        .filter((value): value is string => Boolean(value))
    );
    return ["all", ...Array.from(values)];
  }, [records]);

  const filteredRecords = useMemo(() => {
    return records.filter((record) => {
      if (classificationFilter !== "all" && record.classification !== classificationFilter) {
        return false;
      }
      if (statusFilter !== "all" && record.status !== statusFilter) {
        return false;
      }
      if (ownerFilter !== "all" && record.implementation_owner !== ownerFilter) {
        return false;
      }
      if (categoryFilter !== "all" && record.category !== categoryFilter) {
        return false;
      }

      const search = searchText.trim().toLowerCase();
      if (search) {
        const haystack = [
          record.question,
          record.action_plan,
          record.root_cause_analysis,
          record.comments,
          record.category,
        ]
          .filter(Boolean)
          .join(" ")
          .toLowerCase();
        if (!haystack.includes(search)) return false;
      }

      const createdDate = normalizeDate(record.created_at ?? null);
      const createdFromDate = normalizeDate(createdFrom || null);
      const createdToDate = normalizeDate(createdTo || null);

      if (createdFromDate && createdDate && createdDate < createdFromDate) return false;
      if (createdToDate && createdDate && createdDate > createdToDate) return false;

      const targetDate = normalizeDate(record.target_implementation_date ?? null);
      const targetFromDate = normalizeDate(targetFrom || null);
      const targetToDate = normalizeDate(targetTo || null);

      if (targetFromDate && targetDate && targetDate < targetFromDate) return false;
      if (targetToDate && targetDate && targetDate > targetToDate) return false;

      const endDate = normalizeDate(record.end_date ?? null);
      const endFromDate = normalizeDate(endFrom || null);
      const endToDate = normalizeDate(endTo || null);

      if (endFromDate && endDate && endDate < endFromDate) return false;
      if (endToDate && endDate && endDate > endToDate) return false;

      return true;
    });
  }, [
    categoryFilter,
    classificationFilter,
    createdFrom,
    createdTo,
    endFrom,
    endTo,
    ownerFilter,
    records,
    searchText,
    statusFilter,
    targetFrom,
    targetTo,
  ]);

  const totalsByStatus = useMemo(() => {
    return statusOptions.reduce<Record<StatusOption, number>>((acc, status) => {
      acc[status] = records.filter((record) => record.status === status).length;
      return acc;
    }, {} as Record<StatusOption, number>);
  }, [records]);

  const totalsByClassification = useMemo(() => {
    return classificationOptions.reduce<Record<string, number>>((acc, classification) => {
      acc[classification] = records.filter(
        (record) => record.classification === classification,
      ).length;
      return acc;
    }, {});
  }, [records]);

  const handleOpenNew = () => {
    setDialogMode("new");
    setActiveRecord(null);
    setFormState(buildEmptyFormState());
    setDialogOpen(true);
  };

  const handleOpenEdit = (record: NcCorrectiveAction) => {
    setDialogMode("edit");
    setActiveRecord(record);
    setFormState({
      category: record.category ?? "",
      auditors: formatAuditors(record.auditors === "N/A" ? "" : record.auditors),
      question: record.question ?? "",
      answer: record.answer ?? "",
      classification: record.classification ?? "",
      comments: record.comments ?? "",
      root_cause_analysis: record.root_cause_analysis ?? "",
      action_plan: record.action_plan ?? "",
      implementation_owner: record.implementation_owner ?? "",
      target_implementation_date: formatDateInput(record.target_implementation_date),
      start_date: formatDateInput(record.start_date),
      end_date: formatDateInput(record.end_date),
      evidence_url: record.evidence_url ?? "",
      result: record.result ?? "",
      result_comment: record.result_comment ?? "",
      review_date: formatDateInput(record.review_date),
    });
    setDialogOpen(true);
  };

  const handleFormChange = (field: keyof FormState, value: string) => {
    setFormState((prev) => ({ ...prev, [field]: value }));
  };

  const handleSave = async () => {
    if (!user) {
      toast.error("Necesitas iniciar sesión para guardar.");
      return;
    }

    if (!formState.classification) {
      toast.error("Selecciona una clasificación válida.");
      return;
    }

    if (!formState.category) {
      toast.error("La categoría es obligatoria.");
      return;
    }

    setSaving(true);

    const computedStatus = computeStatus({
      implementation_owner: formState.implementation_owner,
      target_implementation_date: formState.target_implementation_date,
      start_date: formState.start_date,
      end_date: formState.end_date,
      result: formState.result || null,
    });

    const basePayload = {
      category: formState.category,
      auditors: formState.auditors || "N/A",
      question: formState.question,
      answer: formState.answer,
      classification: formState.classification,
      comments: formState.comments,
      root_cause_analysis: formState.root_cause_analysis,
      action_plan: formState.action_plan,
      implementation_owner: formState.implementation_owner || null,
      target_implementation_date: formState.target_implementation_date || null,
      start_date: formState.start_date || null,
      end_date: formState.end_date || null,
      evidence_url: formState.evidence_url || null,
      result: formState.result || null,
      result_comment: formState.result_comment || null,
      review_date: formState.review_date || null,
      status: computedStatus,
      updated_at: new Date().toISOString(),
      updated_by: user.email ?? user.id,
    };

    if (dialogMode === "new") {
      const { error } = await supabase.from("nc_corrective_actions").insert({
        ...basePayload,
        source: "manual",
        created_at: new Date().toISOString(),
      });

      if (error) {
        toast.error("No pudimos guardar el hallazgo.");
        setSaving(false);
        return;
      }

      toast.success("Hallazgo manual registrado.");
    } else if (activeRecord) {
      const { error } = await supabase
        .from("nc_corrective_actions")
        .update(basePayload)
        .eq("id", activeRecord.id);

      if (error) {
        toast.error("No pudimos actualizar el registro.");
        setSaving(false);
        return;
      }

      toast.success("Seguimiento actualizado.");
    }

    await fetchRecords();
    setSaving(false);
    setDialogOpen(false);
  };

  const handleRefresh = async () => {
    setLoading(true);
    await syncAuditFindings();
    await fetchRecords();
    setLoading(false);
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              MEJORA CONTINUA
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              No Conformidad y Acción Correctiva
            </h1>
            <p className="max-w-4xl text-base text-gray-600 sm:text-lg">
              Matriz integral para seguimiento de hallazgos de auditoría interna y acciones
              correctivas manuales. Centraliza análisis, planes de acción y responsables.
            </p>
          </div>
        </section>

        <section className="grid gap-6 lg:grid-cols-4">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Total de hallazgos</CardTitle>
              <ListChecks className="h-5 w-5 text-emerald-600" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{records.length}</p>
              <p className="text-xs text-gray-500">Incluye auditorías y registros manuales.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">En proceso / planeado</CardTitle>
              <CalendarDays className="h-5 w-5 text-amber-500" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">
                {(totalsByStatus["EN PROCESO"] ?? 0) + (totalsByStatus["PLANEADO"] ?? 0)}
              </p>
              <p className="text-xs text-gray-500">Acciones activas o por iniciar.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Completos</CardTitle>
              <ShieldCheck className="h-5 w-5 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{totalsByStatus.COMPLETO ?? 0}</p>
              <p className="text-xs text-gray-500">Acciones cerradas correctamente.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Retrasados</CardTitle>
              <FileSearch className="h-5 w-5 text-red-500" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{totalsByStatus.RETRASADO ?? 0}</p>
              <p className="text-xs text-gray-500">Requieren atención prioritaria.</p>
            </CardContent>
          </Card>
        </section>

        <section className="space-y-4 rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <div className="flex items-center gap-2 text-sm font-semibold text-emerald-700">
            <ClipboardList className="h-4 w-4" />
            Estadísticas por clasificación
          </div>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            {classificationOptions.map((classification) => (
              <Card key={classification} className="border-emerald-100">
                <CardHeader className="space-y-1">
                  <CardTitle className="text-sm font-semibold text-gray-600">{classification}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-bold text-gray-900">
                    {totalsByClassification[classification] ?? 0}
                  </p>
                  <p className="text-xs text-gray-500">Registros clasificados.</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </section>
        
        <section className="space-y-4 rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <div className="flex flex-wrap items-center justify-between gap-4">
            <div className="flex items-center gap-2 text-sm font-semibold text-emerald-700">
              <Filter className="h-4 w-4" />
              Filtros y acciones
            </div>
            <div className="flex flex-wrap gap-2">
              <Button
                variant="outline"
                className="border-emerald-200 text-emerald-700"
                onClick={handleRefresh}
                disabled={loading}
              >
                <RefreshCcw className="mr-2 h-4 w-4" />
                Sincronizar
              </Button>
              {canManage && (
                <Button onClick={handleOpenNew} className="bg-emerald-600 text-white">
                  <Plus className="mr-2 h-4 w-4" />
                  Agregar hallazgo
                </Button>
              )}
            </div>
          </div>

          <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
            <div className="space-y-2">
              <Label>Clasificación</Label>
              <Select value={classificationFilter} onValueChange={setClassificationFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todas" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas</SelectItem>
                  {classificationOptions.map((classification) => (
                    <SelectItem key={classification} value={classification}>
                      {classification}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Estado</Label>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todos" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos</SelectItem>
                  {statusOptions.map((status) => (
                    <SelectItem key={status} value={status}>
                      {status}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Responsable</Label>
              <Select value={ownerFilter} onValueChange={setOwnerFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todos" />
                </SelectTrigger>
                <SelectContent>
                  {uniqueOwners.map((owner) => (
                    <SelectItem key={owner} value={owner}>
                      {owner === "all" ? "Todos" : owner}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Categoría</Label>
              <Select value={categoryFilter} onValueChange={setCategoryFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todas" />
                </SelectTrigger>
                <SelectContent>
                  {uniqueCategories.map((category) => (
                    <SelectItem key={category} value={category}>
                      {category === "all" ? "Todas" : category}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
            <div className="space-y-2">
              <Label>Búsqueda</Label>
              <Input
                placeholder="Buscar por pregunta, plan, comentarios..."
                value={searchText}
                onChange={(event) => setSearchText(event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Creación (desde/hasta)</Label>
              <div className="flex gap-2">
                <Input type="date" value={createdFrom} onChange={(e) => setCreatedFrom(e.target.value)} />
                <Input type="date" value={createdTo} onChange={(e) => setCreatedTo(e.target.value)} />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Meta (desde/hasta)</Label>
              <div className="flex gap-2">
                <Input type="date" value={targetFrom} onChange={(e) => setTargetFrom(e.target.value)} />
                <Input type="date" value={targetTo} onChange={(e) => setTargetTo(e.target.value)} />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Término (desde/hasta)</Label>
              <div className="flex gap-2">
                <Input type="date" value={endFrom} onChange={(e) => setEndFrom(e.target.value)} />
                <Input type="date" value={endTo} onChange={(e) => setEndTo(e.target.value)} />
              </div>
            </div>
          </div>
        </section>

        <section className="rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <div className="mb-4 flex items-center justify-between">
            <div>
              <h2 className="text-lg font-semibold text-gray-900">Matriz de seguimiento</h2>
              <p className="text-sm text-gray-500">
                {filteredRecords.length} registros visibles de {records.length} en total.
              </p>
            </div>
          </div>

          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Fecha de creación</TableHead>
                  <TableHead>Categoría</TableHead>
                  <TableHead>Auditores</TableHead>
                  <TableHead>Pregunta</TableHead>
                  <TableHead>Respuesta</TableHead>
                  <TableHead>Clasificación</TableHead>
                  <TableHead>Comentarios</TableHead>
                  <TableHead>Análisis de causa raíz</TableHead>
                  <TableHead>Plan de acción</TableHead>
                  <TableHead>Responsable</TableHead>
                  <TableHead>Fecha meta</TableHead>
                  <TableHead>Fecha inicio</TableHead>
                  <TableHead>Fecha término</TableHead>
                  <TableHead>Evidencia</TableHead>
                  <TableHead>Resultado</TableHead>
                  <TableHead>Comentario de resultado</TableHead>
                  <TableHead>Fecha revisión</TableHead>
                  <TableHead>Estado</TableHead>
                  <TableHead></TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredRecords.map((record) => (
                  <TableRow key={record.id}>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.created_at)}
                    </TableCell>
                    <TableCell className="min-w-[220px]">
                      <div className="space-y-1">
                        <p className="font-medium text-gray-900">{record.category ?? "Sin categoría"}</p>
                        <Badge variant="outline" className="text-xs">
                          {record.source === "audit" ? "Auditoría" : "Manual"}
                        </Badge>
                      </div>
                    </TableCell>
                    <TableCell className="min-w-[180px] text-sm text-gray-600">
                      {formatAuditors(record.auditors)}
                    </TableCell>
                    <TableCell className="min-w-[220px] text-sm text-gray-600">
                      {record.question || "Sin pregunta"}
                    </TableCell>
                    <TableCell className="min-w-[200px] text-sm text-gray-600">
                      {record.answer || "Sin respuesta"}
                    </TableCell>
                    <TableCell>
                      <Badge className={getClassificationBadge(record.classification)}>
                        {record.classification || "Sin clasificación"}
                      </Badge>
                    </TableCell>
                    <TableCell className="min-w-[200px] text-sm text-gray-600">
                      {record.comments || "Sin comentarios"}
                    </TableCell>
                    <TableCell className="min-w-[240px] text-sm text-gray-600">
                      {record.root_cause_analysis || "Sin análisis"}
                    </TableCell>
                    <TableCell className="min-w-[240px] text-sm text-gray-600">
                      {record.action_plan || "Sin plan"}
                    </TableCell>
                    <TableCell className="min-w-[180px] text-sm text-gray-600">
                      {record.implementation_owner || "Sin responsable"}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.target_implementation_date)}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.start_date)}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.end_date)}
                    </TableCell>
                    <TableCell className="min-w-[180px] text-sm text-emerald-700">
                      {record.evidence_url ? (
                        <a
                          href={record.evidence_url}
                          className="font-medium underline decoration-emerald-300 underline-offset-4 hover:text-emerald-900"
                          target="_blank"
                          rel="noreferrer"
                        >
                          Ver evidencia
                        </a>
                      ) : (
                        <span className="text-gray-500">Sin evidencia</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge className={getResultBadge(record.result)}>
                        {record.result ?? "Sin resultado"}
                      </Badge>
                    </TableCell>
                    <TableCell className="min-w-[200px] text-sm text-gray-600">
                      {record.result_comment || "Sin comentario"}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.review_date)}
                    </TableCell>
                    <TableCell>
                      <Badge className={getStatusBadge(record.status ?? "SIN ASIGNAR")}> 
                        {record.status ?? "SIN ASIGNAR"}
                      </Badge>
                    </TableCell>
                    <TableCell className="text-right">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleOpenEdit(record)}
                      >
                        Ver / Editar
                      </Button>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>
        </section>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto sm:max-w-3xl">
          <DialogHeader>
            <DialogTitle>
              {dialogMode === "new"
                ? "Registrar hallazgo manual"
                : "Detalle y seguimiento del hallazgo"}
            </DialogTitle>
          </DialogHeader>

          {activeRecord && dialogMode === "edit" && (
            <div className="rounded-2xl border border-emerald-100 bg-emerald-50/40 p-4 text-sm text-gray-700">
              <p className="font-semibold text-emerald-700">Datos de auditoría</p>
              <div className="mt-2 grid gap-2 md:grid-cols-2">
                <div>
                  <span className="text-xs uppercase text-gray-500">Fuente</span>
                  <p>{activeRecord.source === "audit" ? "Auditoría interna" : "Manual"}</p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-500">Auditores</span>
                  <p>{formatAuditors(activeRecord.auditors)}</p>
                </div>
                <div className="md:col-span-2">
                  <span className="text-xs uppercase text-gray-500">Pregunta/Descripción</span>
                  <p>{activeRecord.question || "Sin pregunta"}</p>
                </div>
                <div className="md:col-span-2">
                  <span className="text-xs uppercase text-gray-500">Respuesta</span>
                  <p>{activeRecord.answer || "Sin respuesta"}</p>
                </div>
              </div>
            </div>
          )}

          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label>Categoría</Label>
              <Input
                value={formState.category}
                onChange={(event) => handleFormChange("category", event.target.value)}
                disabled={dialogMode === "edit" && activeRecord?.source === "audit"}
              />
            </div>
            <div className="space-y-2">
              <Label>Auditores</Label>
              <Input
                value={formState.auditors}
                onChange={(event) => handleFormChange("auditors", event.target.value)}
                disabled={dialogMode === "edit" && activeRecord?.source === "audit"}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Pregunta / Descripción del hallazgo</Label>
              <Textarea
                value={formState.question}
                onChange={(event) => handleFormChange("question", event.target.value)}
                disabled={dialogMode === "edit" && activeRecord?.source === "audit"}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Respuesta</Label>
              <Textarea
                value={formState.answer}
                onChange={(event) => handleFormChange("answer", event.target.value)}
                disabled={dialogMode === "edit" && activeRecord?.source === "audit"}
              />
            </div>
            <div className="space-y-2">
              <Label>Clasificación</Label>
              <Select
                value={formState.classification}
                onValueChange={(value) => handleFormChange("classification", value)}
                disabled={dialogMode === "edit" && activeRecord?.source === "audit"}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona" />
                </SelectTrigger>
                <SelectContent>
                  {classificationOptions.map((classification) => (
                    <SelectItem key={classification} value={classification}>
                      {classification}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Comentarios</Label>
              <Input
                value={formState.comments}
                onChange={(event) => handleFormChange("comments", event.target.value)}
                disabled={dialogMode === "edit" && activeRecord?.source === "audit"}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Análisis de causa raíz</Label>
              <Textarea
                value={formState.root_cause_analysis}
                onChange={(event) => handleFormChange("root_cause_analysis", event.target.value)}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Plan de acción</Label>
              <Textarea
                value={formState.action_plan}
                onChange={(event) => handleFormChange("action_plan", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Responsable de implementación</Label>
              <Input
                value={formState.implementation_owner}
                onChange={(event) => handleFormChange("implementation_owner", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha meta de implementación</Label>
              <Input
                type="date"
                value={formState.target_implementation_date}
                onChange={(event) => handleFormChange("target_implementation_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha de inicio</Label>
              <Input
                type="date"
                value={formState.start_date}
                onChange={(event) => handleFormChange("start_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha de término</Label>
              <Input
                type="date"
                value={formState.end_date}
                onChange={(event) => handleFormChange("end_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Evidencia (link)</Label>
              <Input
                type="url"
                placeholder="https://"
                value={formState.evidence_url}
                onChange={(event) => handleFormChange("evidence_url", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Resultado</Label>
              <Select
                value={formState.result}
                onValueChange={(value) => handleFormChange("result", value)}
                disabled={!canManage}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona" />
                </SelectTrigger>
                <SelectContent>
                  {resultOptions.map((option) => (
                    <SelectItem key={option} value={option}>
                      {option}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Comentario de resultado</Label>
              <Input
                value={formState.result_comment}
                onChange={(event) => handleFormChange("result_comment", event.target.value)}
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha de revisión</Label>
              <Input
                type="date"
                value={formState.review_date}
                onChange={(event) => handleFormChange("review_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Estado calculado</Label>
              <Input
                value={computeStatus({
                  implementation_owner: formState.implementation_owner,
                  target_implementation_date: formState.target_implementation_date,
                  start_date: formState.start_date,
                  end_date: formState.end_date,
                  result: formState.result || null,
                })}
                readOnly
              />
            </div>
          </div>

          <DialogFooter className="mt-4">
            <Button variant="outline" onClick={() => setDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSave} disabled={saving || !canManage}>
              {saving ? "Guardando..." : "Guardar"}
            </Button>
          </DialogFooter>
          {!canManage && (
            <p className="text-xs text-red-500">
              Solo los managers autorizados pueden guardar cambios.
            </p>
          )}
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}