"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { supabase, type Database } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Checkbox } from "@/components/ui/checkbox";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  addMonths,
  calculateEnvironmentalScore,
  calculateFinalScore,
  getQuarterFromDate,
  getQuarterLabel,
  resolveStatus,
} from "@/lib/supplier-evaluations";
import { ArrowLeft, Pencil, Plus, Trash2 } from "lucide-react";
import { toast } from "sonner";
import { User } from "@supabase/supabase-js";

type Supplier = Database["public"]["Tables"]["suppliers"]["Row"];
type SupplierEvaluation = Database["public"]["Tables"]["supplier_evaluations"]["Row"];

type EvaluationFormState = {
  evaluationDate: string;
  envCert: boolean;
  envRse: boolean;
  envPolicy: boolean;
  techSpecScore: number;
  deliveryScore: number;
  costScore: number;
  paymentTermsScore: number;
  qualityCertsScore: number;
  notes: string;
};

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "comercial@meiwa.com.mx",
  "compras@meiwa.com.mx",
  "qc2.mexico@meiwa.com.mx",  
]);

const initialFormState = (): EvaluationFormState => ({
  evaluationDate: new Date().toISOString().split("T")[0],
  envCert: false,
  envRse: false,
  envPolicy: false,
  techSpecScore: 3,
  deliveryScore: 3,
  costScore: 3,
  paymentTermsScore: 3,
  qualityCertsScore: 3,
  notes: "",
});

const formatDate = (value: string) => new Date(value).toLocaleDateString("es-MX");

const clampScore = (value: number) => Math.min(5, Math.max(1, value));

export default function ExternalSupplierEvaluationsPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const [user, setUser] = useState<User | null>(null);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [selectedSupplierId, setSelectedSupplierId] = useState<string>("");
  const [evaluations, setEvaluations] = useState<SupplierEvaluation[]>([]);
  const [loadingSuppliers, setLoadingSuppliers] = useState(true);
  const [loadingEvaluations, setLoadingEvaluations] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingEvaluation, setEditingEvaluation] = useState<SupplierEvaluation | null>(null);
  const [formState, setFormState] = useState<EvaluationFormState>(initialFormState);
  const [confirmDuplicateOpen, setConfirmDuplicateOpen] = useState(false);
  const [pendingPayload, setPendingPayload] = useState<Database["public"]["Tables"]["supplier_evaluations"]["Insert"] | null>(null);

  const canManage = allowedManagers.has(user?.email ?? "");

  const selectedSupplier = useMemo(
    () => suppliers.find((supplier) => supplier.id === selectedSupplierId) ?? null,
    [selectedSupplierId, suppliers]
  );

  const fetchSuppliers = useCallback(async () => {
    setLoadingSuppliers(true);
    const { data: { session } } = await supabase.auth.getSession();

    if (!session?.user) {
      router.push("/");
      return;
    }

    setUser(session.user);

    const { data, error } = await supabase.from("suppliers").select("*").order("name");
    if (error) {
      toast.error("No se pudieron cargar los proveedores.");
      console.error(error);
      setLoadingSuppliers(false);
      return;
    }

    setSuppliers(data ?? []);
    const preferredId = searchParams.get("supplierId") ?? "";
    const resolvedId = data?.some((supplier) => supplier.id === preferredId)
      ? preferredId
      : data?.[0]?.id ?? "";
    setSelectedSupplierId(resolvedId);
    setLoadingSuppliers(false);
  }, [router, searchParams]);

  const fetchEvaluations = useCallback(
    async (supplierId: string) => {
      if (!supplierId) {
        setEvaluations([]);
        return;
      }
      setLoadingEvaluations(true);
      const { data, error } = await supabase
        .from("supplier_evaluations")
        .select("*")
        .eq("supplier_id", supplierId)
        .order("evaluation_date", { ascending: false });

      if (error) {
        toast.error("No se pudieron cargar las evaluaciones.");
        console.error(error);
        setLoadingEvaluations(false);
        return;
      }

      setEvaluations(data ?? []);
      setLoadingEvaluations(false);
    },
    []
  );

  useEffect(() => {
    fetchSuppliers();
  }, [fetchSuppliers]);

  useEffect(() => {
    void fetchEvaluations(selectedSupplierId);
  }, [fetchEvaluations, selectedSupplierId]);

  const envScore = useMemo(
    () =>
      calculateEnvironmentalScore({
        envCert: formState.envCert,
        envRse: formState.envRse,
        envPolicy: formState.envPolicy,
      }),
    [formState.envCert, formState.envRse, formState.envPolicy]
  );

  const finalScore = useMemo(
    () =>
      calculateFinalScore({
        envScore,
        techSpecScore: formState.techSpecScore,
        deliveryScore: formState.deliveryScore,
        costScore: formState.costScore,
        paymentTermsScore: formState.paymentTermsScore,
        qualityCertsScore: formState.qualityCertsScore,
      }),
    [
      envScore,
      formState.techSpecScore,
      formState.deliveryScore,
      formState.costScore,
      formState.paymentTermsScore,
      formState.qualityCertsScore,
    ]
  );

  const status = useMemo(() => resolveStatus(finalScore), [finalScore]);

  const currentQuarterLabel = useMemo(() => {
    const date = new Date(formState.evaluationDate);
    if (Number.isNaN(date.getTime())) return "Selecciona una fecha";
    const year = date.getFullYear();
    const quarter = getQuarterFromDate(date);
    return getQuarterLabel(year, quarter);
  }, [formState.evaluationDate]);

  const latestEvaluation = evaluations[0];
  const nextEvaluation = latestEvaluation ? addMonths(new Date(latestEvaluation.evaluation_date), 3) : null;

  const handleOpenCreate = () => {
    if (!canManage) {
      toast.error("No tienes permisos para registrar evaluaciones.");
      return;
    }
    if (!selectedSupplierId) {
      toast.error("Selecciona un proveedor.");
      return;
    }
    setEditingEvaluation(null);
    setFormState(initialFormState());
    setDialogOpen(true);
  };

  const handleOpenEdit = (evaluation: SupplierEvaluation) => {
    if (!canManage) {
      toast.error("No tienes permisos para editar evaluaciones.");
      return;
    }
    setEditingEvaluation(evaluation);
    setFormState({
      evaluationDate: evaluation.evaluation_date,
      envCert: evaluation.env_cert,
      envRse: evaluation.env_rse,
      envPolicy: evaluation.env_policy,
      techSpecScore: evaluation.tech_spec_score,
      deliveryScore: evaluation.delivery_score,
      costScore: evaluation.cost_score,
      paymentTermsScore: evaluation.payment_terms_score,
      qualityCertsScore: evaluation.quality_certs_score,
      notes: evaluation.notes ?? "",
    });
    setDialogOpen(true);
  };

  const createEvaluation = async (payload: Database["public"]["Tables"]["supplier_evaluations"]["Insert"]) => {
    const { error } = await supabase.from("supplier_evaluations").insert(payload);
    if (error) {
      toast.error("No se pudo registrar la evaluación.");
      console.error(error);
      return;
    }
    toast.success("Evaluación registrada.");
    setDialogOpen(false);
    fetchEvaluations(selectedSupplierId);
  };

  const updateEvaluation = async (payload: Database["public"]["Tables"]["supplier_evaluations"]["Update"]) => {
    if (!editingEvaluation) return;
    const { error } = await supabase.from("supplier_evaluations").update(payload).eq("id", editingEvaluation.id);
    if (error) {
      toast.error("No se pudo actualizar la evaluación.");
      console.error(error);
      return;
    }
    toast.success("Evaluación actualizada.");
    setDialogOpen(false);
    fetchEvaluations(selectedSupplierId);
  };

  const handleSaveEvaluation = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) {
      toast.error("No tienes permisos para guardar evaluaciones.");
      return;
    }
    if (!selectedSupplierId) {
      toast.error("Selecciona un proveedor.");
      return;
    }
    const date = new Date(formState.evaluationDate);
    if (Number.isNaN(date.getTime())) {
      toast.error("Selecciona una fecha válida.");
      return;
    }

    const year = date.getFullYear();
    const quarter = getQuarterFromDate(date) as 1 | 2 | 3 | 4;
    const { data: { session } } = await supabase.auth.getSession();

    const basePayload = {
      supplier_id: selectedSupplierId,
      evaluation_date: formState.evaluationDate,
      year,
      quarter,
      env_cert: formState.envCert,
      env_rse: formState.envRse,
      env_policy: formState.envPolicy,
      env_score: envScore,
      tech_spec_score: clampScore(formState.techSpecScore),
      delivery_score: clampScore(formState.deliveryScore),
      cost_score: clampScore(formState.costScore),
      payment_terms_score: clampScore(formState.paymentTermsScore),
      quality_certs_score: clampScore(formState.qualityCertsScore),
      final_score: finalScore,
      status,
      notes: formState.notes.trim() || null,
    };

    const hasDuplicate =
      !editingEvaluation &&
      evaluations.some((evaluation) => evaluation.year === year && evaluation.quarter === quarter);

    if (hasDuplicate) {
      setPendingPayload({
        ...basePayload,
        created_by: session?.user?.id ?? null,
      });
      setConfirmDuplicateOpen(true);
      return;
    }

    if (editingEvaluation) {
      await updateEvaluation(basePayload);
    } else {
      await createEvaluation({
        ...basePayload,
        created_by: session?.user?.id ?? null,
      });
    }
  };

  const handleConfirmDuplicate = async () => {
    if (!pendingPayload) return;
    setConfirmDuplicateOpen(false);
    await createEvaluation(pendingPayload);
    setPendingPayload(null);
  };

  const handleDeleteEvaluation = async (evaluation: SupplierEvaluation) => {
    if (!canManage) {
      toast.error("No tienes permisos para eliminar evaluaciones.");
      return;
    }
    const confirmation = window.confirm("¿Deseas eliminar esta evaluación?");
    if (!confirmation) return;

    const { error } = await supabase.from("supplier_evaluations").delete().eq("id", evaluation.id);
    if (error) {
      toast.error("No se pudo eliminar la evaluación.");
      console.error(error);
      return;
    }

    setEvaluations((prev) => prev.filter((item) => item.id !== evaluation.id));
    toast.success("Evaluación eliminada.");
  };

  return (
    <SGIDashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-wrap items-start justify-between gap-4">
          <div>
            <Button variant="ghost" className="mb-2 gap-2" onClick={() => router.push("/meiwa-core/meiwa-sgi/operation/external")}>
              <ArrowLeft className="h-4 w-4" />
              Volver a proveedores
            </Button>
            <h1 className="text-2xl font-semibold text-gray-900">Evaluación de proveedores</h1>
            <p className="text-sm text-gray-500">
              Selecciona un proveedor para revisar el historial y registrar nuevas evaluaciones.
            </p>
          </div>
          <Button onClick={handleOpenCreate} className="gap-2" disabled={!canManage || !selectedSupplierId}>
            <Plus className="h-4 w-4" />
            Nueva evaluación
          </Button>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Proveedor seleccionado</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {loadingSuppliers ? (
              <p className="text-sm text-gray-500">Cargando proveedores...</p>
            ) : (
              <div className="grid gap-4 md:grid-cols-[320px,1fr]">
                <div className="space-y-2">
                  <Label>Proveedor</Label>
                  <Select value={selectedSupplierId} onValueChange={setSelectedSupplierId}>
                    <SelectTrigger>
                      <SelectValue placeholder="Selecciona un proveedor" />
                    </SelectTrigger>
                    <SelectContent>
                      {suppliers.length === 0 ? (
                        <SelectItem value="sin-proveedores" disabled>
                          No hay proveedores registrados
                        </SelectItem>
                      ) : (
                        suppliers.map((supplier) => (
                          <SelectItem key={supplier.id} value={supplier.id}>
                            {supplier.name}
                          </SelectItem>
                        ))
                      )}
                    </SelectContent>
                  </Select>
                </div>
                <div className="rounded-md border bg-slate-50 p-4 text-sm text-gray-600">
                  {selectedSupplier ? (
                    <div className="space-y-1">
                      <p className="text-base font-semibold text-gray-900">{selectedSupplier.name}</p>
                      <p>
                        {selectedSupplier.material_type ?? "Material no especificado"} ·{" "}
                        {selectedSupplier.location ?? "Ubicación no especificada"}
                      </p>
                      <p>{selectedSupplier.contact_email_or_contact ?? "Sin contacto asignado"}</p>
                    </div>
                  ) : (
                    "Selecciona un proveedor para visualizar información."
                  )}
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        <div className="grid gap-4 md:grid-cols-3">
          <Card>
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Última evaluación</CardTitle>
            </CardHeader>
            <CardContent>
              {latestEvaluation ? (
                <div className="space-y-1">
                  <p className="text-lg font-semibold text-gray-900">{formatDate(latestEvaluation.evaluation_date)}</p>
                  <p className="text-sm text-gray-500">
                    {getQuarterLabel(latestEvaluation.year, latestEvaluation.quarter)}
                  </p>
                </div>
              ) : (
                <p className="text-sm text-gray-500">Pendiente</p>
              )}
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Próxima evaluación</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-lg font-semibold text-gray-900">
                {nextEvaluation ? nextEvaluation.toLocaleDateString("es-MX") : "Pendiente"}
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Estado actual</CardTitle>
            </CardHeader>
            <CardContent>
              {latestEvaluation ? (
                <Badge variant={latestEvaluation.status === "APROBADO" ? "default" : "destructive"}>
                  {latestEvaluation.status}
                </Badge>
              ) : (
                <Badge variant="outline">Sin evaluación</Badge>
              )}
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Historial de evaluaciones</CardTitle>
          </CardHeader>
          <CardContent>
            {loadingEvaluations ? (
              <p className="text-sm text-gray-500">Cargando evaluaciones...</p>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Fecha</TableHead>
                    <TableHead>Periodo</TableHead>
                    <TableHead>Consideraciones ambientales (10%)</TableHead>
                    <TableHead>Especificaciones técnicas (25%)</TableHead>
                    <TableHead>Tiempo de entrega (25%)</TableHead>
                    <TableHead>Costo (20%)</TableHead>
                    <TableHead>Condiciones de pago (10%)</TableHead>
                    <TableHead>Certificados de calidad (10%)</TableHead>
                    <TableHead>Score final</TableHead>
                    <TableHead>Estatus</TableHead>
                    <TableHead className="text-right">Acciones</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {!selectedSupplierId ? (
                    <TableRow>
                      <TableCell colSpan={11} className="text-center text-sm text-gray-500">
                        Selecciona un proveedor para ver sus evaluaciones.
                      </TableCell>
                    </TableRow>
                  ) : evaluations.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={11} className="text-center text-sm text-gray-500">
                        No hay evaluaciones registradas aún.
                      </TableCell>
                    </TableRow>
                  ) : (
                    evaluations.map((evaluation) => (
                      <TableRow key={evaluation.id}>
                        <TableCell>{formatDate(evaluation.evaluation_date)}</TableCell>
                        <TableCell>{getQuarterLabel(evaluation.year, evaluation.quarter)}</TableCell>
                        <TableCell>{evaluation.env_score.toFixed(1)}</TableCell>
                        <TableCell>{evaluation.tech_spec_score.toFixed(1)}</TableCell>
                        <TableCell>{evaluation.delivery_score.toFixed(1)}</TableCell>
                        <TableCell>{evaluation.cost_score.toFixed(1)}</TableCell>
                        <TableCell>{evaluation.payment_terms_score.toFixed(1)}</TableCell>
                        <TableCell>{evaluation.quality_certs_score.toFixed(1)}</TableCell>                        
                        <TableCell>{evaluation.final_score.toFixed(1)}</TableCell>
                        <TableCell>
                          <Badge variant={evaluation.status === "APROBADO" ? "default" : "destructive"}>
                            {evaluation.status}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right">
                          <div className="flex justify-end gap-2">
                            <Button variant="ghost" size="icon" onClick={() => handleOpenEdit(evaluation)} disabled={!canManage}>
                              <Pencil className="h-4 w-4" />
                            </Button>
                            <Button variant="ghost" size="icon" onClick={() => handleDeleteEvaluation(evaluation)} disabled={!canManage}>
                              <Trash2 className="h-4 w-4 text-rose-500" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            )}
          </CardContent>
        </Card>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>{editingEvaluation ? "Editar evaluación" : "Nueva evaluación trimestral"}</DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSaveEvaluation} className="space-y-6">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label htmlFor="evaluation-date">Fecha de evaluación</Label>
                <Input
                  id="evaluation-date"
                  type="date"
                  value={formState.evaluationDate}
                  onChange={(event) => setFormState((prev) => ({ ...prev, evaluationDate: event.target.value }))}
                  required
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Periodo</Label>
                <div className="rounded-md border px-3 py-2 text-sm text-gray-700">{currentQuarterLabel}</div>
              </div>
            </div>

            <div className="space-y-4">
              <div>
                <Label className="text-sm font-semibold text-gray-700">Consideraciones ambientales (10%)</Label>
                <p className="text-xs text-gray-500">Selecciona los elementos disponibles para calcular el puntaje.</p>
              </div>
              <div className="grid gap-3 md:grid-cols-3">
                <label className="flex items-center gap-2 text-sm">
                  <Checkbox
                    checked={formState.envCert}
                    onCheckedChange={(value) =>
                      setFormState((prev) => ({ ...prev, envCert: value === true }))
                    }
                    disabled={!canManage}
                  />
                  Certificados ambientales
                </label>
                <label className="flex items-center gap-2 text-sm">
                  <Checkbox
                    checked={formState.envRse}
                    onCheckedChange={(value) =>
                      setFormState((prev) => ({ ...prev, envRse: value === true }))
                    }
                    disabled={!canManage}
                  />
                  RSE ambiental
                </label>
                <label className="flex items-center gap-2 text-sm">
                  <Checkbox
                    checked={formState.envPolicy}
                    onCheckedChange={(value) =>
                      setFormState((prev) => ({ ...prev, envPolicy: value === true }))
                    }
                    disabled={!canManage}
                  />
                  Política ambiental
                </label>
              </div>
              <div className="rounded-md border px-3 py-2 text-sm text-gray-700">
                Puntaje ambiental calculado: <span className="font-semibold">{envScore}</span>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Especificaciones técnicas (25%)</Label>
                <Input
                  type="number"
                  min={1}
                  max={5}
                  value={formState.techSpecScore}
                  onChange={(event) =>
                    setFormState((prev) => ({
                      ...prev,
                      techSpecScore: clampScore(Number(event.target.value)),
                    }))
                  }
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Tiempo de entrega (25%)</Label>
                <Input
                  type="number"
                  min={1}
                  max={5}
                  value={formState.deliveryScore}
                  onChange={(event) =>
                    setFormState((prev) => ({
                      ...prev,
                      deliveryScore: clampScore(Number(event.target.value)),
                    }))
                  }
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Costo (20%)</Label>
                <Input
                  type="number"
                  min={1}
                  max={5}
                  value={formState.costScore}
                  onChange={(event) =>
                    setFormState((prev) => ({
                      ...prev,
                      costScore: clampScore(Number(event.target.value)),
                    }))
                  }
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Condiciones de pago (10%)</Label>
                <Input
                  type="number"
                  min={1}
                  max={5}
                  value={formState.paymentTermsScore}
                  onChange={(event) =>
                    setFormState((prev) => ({
                      ...prev,
                      paymentTermsScore: clampScore(Number(event.target.value)),
                    }))
                  }
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Certificados de calidad (10%)</Label>
                <Input
                  type="number"
                  min={1}
                  max={5}
                  value={formState.qualityCertsScore}
                  onChange={(event) =>
                    setFormState((prev) => ({
                      ...prev,
                      qualityCertsScore: clampScore(Number(event.target.value)),
                    }))
                  }
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Score final</Label>
                <div className="flex items-center gap-2 rounded-md border px-3 py-2 text-sm text-gray-700">
                  <span className="font-semibold">{finalScore.toFixed(1)}</span>
                  <Badge variant={status === "APROBADO" ? "default" : "destructive"}>{status}</Badge>
                </div>
              </div>
            </div>

            <div className="space-y-2">
              <Label>Notas</Label>
              <Textarea
                value={formState.notes}
                onChange={(event) => setFormState((prev) => ({ ...prev, notes: event.target.value }))}
                placeholder="Observaciones adicionales"
                disabled={!canManage}
              />
            </div>

            <div className="flex justify-end gap-2">
              <Button type="button" variant="ghost" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit" disabled={!canManage}>
                {editingEvaluation ? "Guardar cambios" : "Registrar evaluación"}
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      <AlertDialog open={confirmDuplicateOpen} onOpenChange={setConfirmDuplicateOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Ya existe una evaluación para este trimestre</AlertDialogTitle>
          </AlertDialogHeader>
          <p className="text-sm text-gray-500">
            Este proveedor ya cuenta con una evaluación en el mismo trimestre. ¿Deseas registrar una re-evaluación?
          </p>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={handleConfirmDuplicate}>Registrar re-evaluación</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}