"use client";

import { type FormEvent, useEffect, useMemo, useState } from "react";
import { type User } from "@supabase/supabase-js";
import { toast } from "sonner";
import {
  AlertCircle,
  ClipboardCheck,
  ClipboardList,
  Plus,
  RefreshCcw,
  Trash2,
} from "lucide-react";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

const departmentOptions = [
  "01_ADMIN",
  "02_MARKETING",
  "03_DS",
  "04_PE",
  "05_P1",
  "06_P2",
  "07_INSP",
  "08_COMERCIAL",
  "09_SUB",
  "10_QC",
  "11_ADMIN PRO",
  "12_OTRO",
  "13_CLIENTE",
  "14_VENTAS",
  "15_PROVEEDOR",
];

const defectOptions = [
  "1_Fuera de tolerancia (Falta cortar)",
  "2_Falta Proceso",
  "3_Restos de procesamiento",
  "4_Golpe (herramientas)",
  "5_Step",
  "6_Grabado (QC)",
  "7_Burry (rebaba)",
  "8_Golpe (otros)",
  "20_Sin dibujo",
  "21_Part List",
  "23_3D Model",
  "24_Condiciones de plano",
  "25_Interferencias",
  "27_Problema de soldadura",
  "30_Error de proceso",
];

const factorOptions = ["A", "B", "C", "D"];
const documentOptions = ["Form A", "QIR completo"];
const scopeOptions = [
  "Interno",
  "Aprobación de cliente",
  "Reclamo de cliente",
];
const classificationOptions = ["Stock", "Scrap"] as const;
const stockDestinationOptions = ["retrabajo", "scrap", "stock"] as const;

const parseNumber = (value: string) =>
  value.trim() === "" ? null : Number(value);

const formatDate = (value?: string | null) =>
  value
    ? new Date(`${value}T00:00:00`).toLocaleDateString("es-MX", {
        day: "2-digit",
        month: "short",
        year: "numeric",
      })
    : "-";

const calculateVolume = (l?: number | null, w?: number | null, h?: number | null) => {
  if (!l || !w || !h) return null;
  return Number((l * w * h).toFixed(2));
};

const calculateWeight = (volume?: number | null) => {
  if (!volume) return null;
  return Number((volume * 0.00786).toFixed(2));
};

const calculateDays = (entry?: string | null, exit?: string | null) => {
  if (!entry) return null;
  const start = new Date(`${entry}T00:00:00`).getTime();
  const end = new Date(`${exit ?? new Date().toISOString().slice(0, 10)}T00:00:00`).getTime();
  if (Number.isNaN(start) || Number.isNaN(end)) return null;
  const diff = Math.ceil((end - start) / (1000 * 60 * 60 * 24));
  return diff >= 0 ? diff : null;
};

type QirRecord = {
  id: string;
  qirNumber: number | null;
  date: string | null;
  detectedByDepartment: string | null;
  jobNumber: string | null;
  partName: string | null;
  jobName: string | null;
  defectType: string | null;
  defectDetail: string | null;
  correctiveAction: string | null;
  timeManDs: number | null;
  timeManPe: number | null;
  timeManP1: number | null;
  timeCnc: number | null;
  timeEdm: number | null;
  timeWc: number | null;
  timeMm: number | null;
  timeLathe: number | null;
  timeManIns: number | null;
  timeCmm: number | null;
  timeKeyence: number | null;
  timeManP2: number | null;
  timeOven: number | null;
  timeWelding: number | null;
  timeManCommercial: number | null;
  materialCost: number | null;
  lostCost: number | null;
  errorPerson: string | null;
  factorIdentification: string | null;
  responsibleArea: string | null;
  documentType: string | null;
  scope: string | null;
  materialClassification: (typeof classificationOptions)[number] | null;
  sentToStockOrScrapDate: string | null;
  whyWhySentDate: string | null;
  countermeasure: string | null;
  socialization: string | null;
  application: string | null;
  applicationJobNumber: string | null;
  qcDelay: string | null;
  standardDelay: string | null;
  socializationDelay: string | null;
  applicationDelay: string | null;
};

type StockRecord = {
  id: string;
  qirId: string | null;
  qirNumber: number | null;
  qirDate: string | null;
  stockEntryDate: string | null;
  originalJobNumber: string | null;
  materialType: string | null;
  sizeL: number | null;
  sizeW: number | null;
  sizeH: number | null;
  volumeCm3: number | null;
  approxWeight: number | null;
  destination: (typeof stockDestinationOptions)[number] | null;
  newJobNumber: string | null;
  status: string | null;
  stockExitDate: string | null;
  comments: string | null;
};

type ScrapRecord = {
  id: string;
  qirId: string | null;
  stockId: string | null;
  qirNumber: number | null;
  qirDate: string | null;
  stockEntryDate: string | null;
  originalJobNumber: string | null;
  materialType: string | null;
  sizeL: number | null;
  sizeW: number | null;
  sizeH: number | null;
  volumeCm3: number | null;
  approxWeight: number | null;
  destination: string | null;
  scrapExitDate: string | null;
  comments: string | null;
};

const buildEmptyQirForm = (): QirRecord => ({
  id: "",
  qirNumber: null,
  date: "",
  detectedByDepartment: "",
  jobNumber: "",
  partName: "",
  jobName: "",
  defectType: "",
  defectDetail: "",
  correctiveAction: "",
  timeManDs: null,
  timeManPe: null,
  timeManP1: null,
  timeCnc: null,
  timeEdm: null,
  timeWc: null,
  timeMm: null,
  timeLathe: null,
  timeManIns: null,
  timeCmm: null,
  timeKeyence: null,
  timeManP2: null,
  timeOven: null,
  timeWelding: null,
  timeManCommercial: null,
  materialCost: null,
  lostCost: null,
  errorPerson: "",
  factorIdentification: "",
  responsibleArea: "",
  documentType: "",
  scope: "",
  materialClassification: "Stock",
  sentToStockOrScrapDate: "",
  whyWhySentDate: "",
  countermeasure: "",
  socialization: "",
  application: "",
  applicationJobNumber: "",
  qcDelay: "",
  standardDelay: "",
  socializationDelay: "",
  applicationDelay: "",
});

const buildEmptyStockForm = (): StockRecord => ({
  id: "",
  qirId: null,
  qirNumber: null,
  qirDate: null,
  stockEntryDate: "",
  originalJobNumber: "",
  materialType: "",
  sizeL: null,
  sizeW: null,
  sizeH: null,
  volumeCm3: null,
  approxWeight: null,
  destination: "stock",
  newJobNumber: "",
  status: "stock",
  stockExitDate: "",
  comments: "",
});

const buildEmptyScrapForm = (): ScrapRecord => ({
  id: "",
  qirId: null,
  stockId: null,
  qirNumber: null,
  qirDate: null,
  stockEntryDate: "",
  originalJobNumber: "",
  materialType: "",
  sizeL: null,
  sizeW: null,
  sizeH: null,
  volumeCm3: null,
  approxWeight: null,
  destination: "residuos de manejo especial",
  scrapExitDate: "",
  comments: "",
});

const mapQirRecord = (record: Record<string, unknown>): QirRecord => ({
  id: String(record.id ?? ""),
  qirNumber:
    typeof record.qir_number === "number" ? record.qir_number : Number(record.qir_number ?? 0) || null,
  date: (record.date as string) ?? null,
  detectedByDepartment: (record.detected_by_department as string) ?? null,
  jobNumber: (record.job_number as string) ?? null,
  partName: (record.part_name as string) ?? null,
  jobName: (record.job_name as string) ?? null,
  defectType: (record.defect_type as string) ?? null,
  defectDetail: (record.defect_detail as string) ?? null,
  correctiveAction: (record.corrective_action as string) ?? null,
  timeManDs: record.time_man_ds as number | null,
  timeManPe: record.time_man_pe as number | null,
  timeManP1: record.time_man_p1 as number | null,
  timeCnc: record.time_cnc as number | null,
  timeEdm: record.time_edm as number | null,
  timeWc: record.time_wc as number | null,
  timeMm: record.time_mm as number | null,
  timeLathe: record.time_lathe as number | null,
  timeManIns: record.time_man_ins as number | null,
  timeCmm: record.time_cmm as number | null,
  timeKeyence: record.time_keyence as number | null,
  timeManP2: record.time_man_p2 as number | null,
  timeOven: record.time_oven as number | null,
  timeWelding: record.time_welding as number | null,
  timeManCommercial: record.time_man_commercial as number | null,
  materialCost: record.material_cost as number | null,
  lostCost: record.lost_cost as number | null,
  errorPerson: (record.error_person as string) ?? null,
  factorIdentification: (record.factor_identification as string) ?? null,
  responsibleArea: (record.responsible_area as string) ?? null,
  documentType: (record.document_type as string) ?? null,
  scope: (record.scope as string) ?? null,
  materialClassification: (record.material_classification as
    | (typeof classificationOptions)[number]
    | null) ?? null,
  sentToStockOrScrapDate: (record.sent_to_stock_or_scrap_date as string) ?? null,
  whyWhySentDate: (record.why_why_sent_date as string) ?? null,
  countermeasure: (record.countermeasure as string) ?? null,
  socialization: (record.socialization as string) ?? null,
  application: (record.application as string) ?? null,
  applicationJobNumber: (record.application_job_number as string) ?? null,
  qcDelay: (record.qc_delay as string) ?? null,
  standardDelay: (record.standard_delay as string) ?? null,
  socializationDelay: (record.socialization_delay as string) ?? null,
  applicationDelay: (record.application_delay as string) ?? null,
});

const mapStockRecord = (record: Record<string, unknown>): StockRecord => ({
  id: String(record.id ?? ""),
  qirId: (record.qir_id as string) ?? null,
  qirNumber: typeof record.qir_number === "number" ? record.qir_number : Number(record.qir_number ?? 0) || null,
  qirDate: (record.qir_date as string) ?? null,
  stockEntryDate: (record.stock_entry_date as string) ?? null,
  originalJobNumber: (record.original_job_number as string) ?? null,
  materialType: (record.material_type as string) ?? null,
  sizeL: record.size_l as number | null,
  sizeW: record.size_w as number | null,
  sizeH: record.size_h as number | null,
  volumeCm3: record.volume_cm3 as number | null,
  approxWeight: record.approx_weight as number | null,
  destination: (record.destination as StockRecord["destination"]) ?? null,
  newJobNumber: (record.new_job_number as string) ?? null,
  status: (record.status as string) ?? null,
  stockExitDate: (record.stock_exit_date as string) ?? null,
  comments: (record.comments as string) ?? null,
});

const mapScrapRecord = (record: Record<string, unknown>): ScrapRecord => ({
  id: String(record.id ?? ""),
  qirId: (record.qir_id as string) ?? null,
  stockId: (record.stock_id as string) ?? null,
  qirNumber: typeof record.qir_number === "number" ? record.qir_number : Number(record.qir_number ?? 0) || null,
  qirDate: (record.qir_date as string) ?? null,
  stockEntryDate: (record.stock_entry_date as string) ?? null,
  originalJobNumber: (record.original_job_number as string) ?? null,
  materialType: (record.material_type as string) ?? null,
  sizeL: record.size_l as number | null,
  sizeW: record.size_w as number | null,
  sizeH: record.size_h as number | null,
  volumeCm3: record.volume_cm3 as number | null,
  approxWeight: record.approx_weight as number | null,
  destination: (record.destination as string) ?? null,
  scrapExitDate: (record.scrap_exit_date as string) ?? null,
  comments: (record.comments as string) ?? null,
});

export default function QualityPage() {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [qirRecords, setQirRecords] = useState<QirRecord[]>([]);
  const [stockRecords, setStockRecords] = useState<StockRecord[]>([]);
  const [scrapRecords, setScrapRecords] = useState<ScrapRecord[]>([]);

  const [qirDialogOpen, setQirDialogOpen] = useState(false);
  const [stockDialogOpen, setStockDialogOpen] = useState(false);
  const [scrapDialogOpen, setScrapDialogOpen] = useState(false);
  const [qirDeleteTarget, setQirDeleteTarget] = useState<QirRecord | null>(null);
  const [stockDeleteTarget, setStockDeleteTarget] = useState<StockRecord | null>(null);
  const [scrapDeleteTarget, setScrapDeleteTarget] = useState<ScrapRecord | null>(null);

  const [qirForm, setQirForm] = useState<QirRecord>(buildEmptyQirForm());
  const [stockForm, setStockForm] = useState<StockRecord>(buildEmptyStockForm());
  const [scrapForm, setScrapForm] = useState<ScrapRecord>(buildEmptyScrapForm());

  const [qirFilters, setQirFilters] = useState({
    date: "",
    department: "",
    defect: "",
    responsibleArea: "",
    scope: "",
    classification: "",
    search: "",
  });

  const [stockFilters, setStockFilters] = useState({
    status: "",
    destination: "",
    daysRange: "",
    search: "",
  });

  const [scrapFilters, setScrapFilters] = useState({
    exitDate: "",
    daysRange: "",
    search: "",
  });

  const canManage = allowedManagers.has(user?.email ?? "");
  const currencyFormatter = useMemo(
    () =>
      new Intl.NumberFormat("es-MX", {
        style: "currency",
        currency: "MXN",
        maximumFractionDigits: 2,
      }),
    [],
  );
  const numberFormatter = useMemo(
    () =>
      new Intl.NumberFormat("es-MX", {
        maximumFractionDigits: 2,
      }),
    [],
  );

  const qualityStats = useMemo(() => {
    const totalLostCost = qirRecords.reduce(
      (sum, record) => sum + (record.lostCost ?? 0),
      0,
    );
    const totalScrapVolume = scrapRecords.reduce(
      (sum, record) => sum + (record.volumeCm3 ?? 0),
      0,
    );
    return {
      totalQir: qirRecords.length,
      totalLostCost,
      totalScrapPieces: scrapRecords.length,
      totalScrapVolume,
    };
  }, [qirRecords, scrapRecords]);

  const monthlyTrend = useMemo(() => {
    const totals = new Map<string, number>();

    qirRecords.forEach((record) => {
      if (!record.date) return;
      const monthKey = record.date.slice(0, 7);
      totals.set(monthKey, (totals.get(monthKey) ?? 0) + 1);
    });

    return Array.from(totals.entries())
      .sort(([a], [b]) => a.localeCompare(b))
      .map(([month, count]) => ({ month, count }));
  }, [qirRecords]);

  const maxMonthlyCount = useMemo(
    () => (monthlyTrend.length ? Math.max(...monthlyTrend.map((item) => item.count)) : 0),
    [monthlyTrend],
  );

  const timeLossStats = useMemo(() => {
    const totals = {
      "Mano de obra DS": 0,
      "Mano de obra PE": 0,
      "Mano de obra P1": 0,
      "Mano de obra P2": 0,
      CNC: 0,
      EDM: 0,
      WC: 0,
      MM: 0,
      Torno: 0,
      Inspección: 0,
      CMM: 0,
      Keyence: 0,
      Horno: 0,
      Soldadura: 0,
      Comercial: 0,
    };

    qirRecords.forEach((record) => {
      totals["Mano de obra DS"] += record.timeManDs ?? 0;
      totals["Mano de obra PE"] += record.timeManPe ?? 0;
      totals["Mano de obra P1"] += record.timeManP1 ?? 0;
      totals["Mano de obra P2"] += record.timeManP2 ?? 0;
      totals.CNC += record.timeCnc ?? 0;
      totals.EDM += record.timeEdm ?? 0;
      totals.WC += record.timeWc ?? 0;
      totals.MM += record.timeMm ?? 0;
      totals.Torno += record.timeLathe ?? 0;
      totals.Inspección += record.timeManIns ?? 0;
      totals.CMM += record.timeCmm ?? 0;
      totals.Keyence += record.timeKeyence ?? 0;
      totals.Horno += record.timeOven ?? 0;
      totals.Soldadura += record.timeWelding ?? 0;
      totals.Comercial += record.timeManCommercial ?? 0;
    });

    return Object.entries(totals)
      .map(([label, value]) => ({ label, value }))
      .filter((entry) => entry.value > 0)
      .sort((a, b) => b.value - a.value);
  }, [qirRecords]);

  const factorStats = useMemo(() => {
    const totals = new Map<string, number>();
    qirRecords.forEach((record) => {
      if (!record.factorIdentification) return;
      totals.set(
        record.factorIdentification,
        (totals.get(record.factorIdentification) ?? 0) + 1,
      );
    });

    return Array.from(totals.entries())
      .map(([label, count]) => ({ label, count }))
      .sort((a, b) => b.count - a.count);
  }, [qirRecords]);

  const responsibleAreaStats = useMemo(() => {
    const totals = new Map<string, number>();
    qirRecords.forEach((record) => {
      if (!record.responsibleArea) return;
      totals.set(
        record.responsibleArea,
        (totals.get(record.responsibleArea) ?? 0) + 1,
      );
    });

    return Array.from(totals.entries())
      .map(([label, count]) => ({ label, count }))
      .sort((a, b) => b.count - a.count);
  }, [qirRecords]);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      setUser(data.session?.user ?? null);
    };

    const loadRecords = async () => {
      setLoading(true);
      const [{ data: qirData, error: qirError }, { data: stockData, error: stockError }, { data: scrapData, error: scrapError }] =
        await Promise.all([
          supabase.from("quality_qir").select("*").order("qir_number", { ascending: false }),
          supabase.from("quality_stock").select("*").order("created_at", { ascending: false }),
          supabase.from("quality_scrap").select("*").order("created_at", { ascending: false }),
        ]);

      if (qirError || stockError || scrapError) {
        toast.error("No pudimos cargar los registros de calidad.");
      }

      setQirRecords((qirData ?? []).map((record) => mapQirRecord(record)));
      setStockRecords((stockData ?? []).map((record) => mapStockRecord(record)));
      setScrapRecords((scrapData ?? []).map((record) => mapScrapRecord(record)));
      setLoading(false);
    };

    loadSession();
    loadRecords();

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
    });

    return () => subscription.unsubscribe();
  }, []);

  const reloadData = async () => {
    setLoading(true);
    const [{ data: qirData }, { data: stockData }, { data: scrapData }] = await Promise.all([
      supabase.from("quality_qir").select("*").order("qir_number", { ascending: false }),
      supabase.from("quality_stock").select("*").order("created_at", { ascending: false }),
      supabase.from("quality_scrap").select("*").order("created_at", { ascending: false }),
    ]);

    setQirRecords((qirData ?? []).map((record) => mapQirRecord(record)));
    setStockRecords((stockData ?? []).map((record) => mapStockRecord(record)));
    setScrapRecords((scrapData ?? []).map((record) => mapScrapRecord(record)));
    setLoading(false);
  };

  const openQirDialog = (record?: QirRecord) => {
    setQirForm(record ? { ...record } : buildEmptyQirForm());
    setQirDialogOpen(true);
  };

  const openStockDialog = (record?: StockRecord) => {
    setStockForm(record ? { ...record } : buildEmptyStockForm());
    setStockDialogOpen(true);
  };

  const openScrapDialog = (record?: ScrapRecord) => {
    setScrapForm(record ? { ...record } : buildEmptyScrapForm());
    setScrapDialogOpen(true);
  };

  const buildQirPayload = (form: QirRecord) => ({
    date: form.date || null,
    detected_by_department: form.detectedByDepartment || null,
    job_number: form.jobNumber || null,
    part_name: form.partName || null,
    job_name: form.jobName || null,
    defect_type: form.defectType || null,
    defect_detail: form.defectDetail || null,
    corrective_action: form.correctiveAction || null,
    time_man_ds: form.timeManDs,
    time_man_pe: form.timeManPe,
    time_man_p1: form.timeManP1,
    time_cnc: form.timeCnc,
    time_edm: form.timeEdm,
    time_wc: form.timeWc,
    time_mm: form.timeMm,
    time_lathe: form.timeLathe,
    time_man_ins: form.timeManIns,
    time_cmm: form.timeCmm,
    time_keyence: form.timeKeyence,
    time_man_p2: form.timeManP2,
    time_oven: form.timeOven,
    time_welding: form.timeWelding,
    time_man_commercial: form.timeManCommercial,
    material_cost: form.materialCost,
    lost_cost: form.lostCost,
    error_person: form.errorPerson || null,
    factor_identification: form.factorIdentification || null,
    responsible_area: form.responsibleArea || null,
    document_type: form.documentType || null,
    scope: form.scope || null,
    material_classification: form.materialClassification || null,
    sent_to_stock_or_scrap_date: form.sentToStockOrScrapDate || null,
    why_why_sent_date: form.whyWhySentDate || null,
    countermeasure: form.countermeasure || null,
    socialization: form.socialization || null,
    application: form.application || null,
    application_job_number: form.applicationJobNumber || null,
    qc_delay: form.qcDelay || null,
    standard_delay: form.standardDelay || null,
    socialization_delay: form.socializationDelay || null,
    application_delay: form.applicationDelay || null,
  });

  const syncStockFromQir = async (qir: QirRecord) => {
    const { data: existing } = await supabase
      .from("quality_stock")
      .select("*")
      .eq("qir_id", qir.id)
      .maybeSingle();

    const volume = calculateVolume(existing?.size_l ?? null, existing?.size_w ?? null, existing?.size_h ?? null);
    const approxWeight = calculateWeight(volume ?? existing?.volume_cm3 ?? null);

    const payload = {
      qir_id: qir.id,
      qir_number: qir.qirNumber,
      qir_date: qir.date,
      stock_entry_date:
        existing?.stock_entry_date ?? qir.sentToStockOrScrapDate ?? null,
      original_job_number: qir.jobNumber,
      material_type: qir.defectType,
      size_l: existing?.size_l ?? null,
      size_w: existing?.size_w ?? null,
      size_h: existing?.size_h ?? null,
      volume_cm3: volume ?? existing?.volume_cm3 ?? null,
      approx_weight: approxWeight ?? existing?.approx_weight ?? null,
      destination: existing?.destination ?? "stock",
      new_job_number: existing?.new_job_number ?? null,
      status: existing?.status ?? "stock",
      stock_exit_date: existing?.stock_exit_date ?? null,
      comments: existing?.comments ?? null,
    };

    const { data, error } = await supabase
      .from("quality_stock")
      .upsert(payload, { onConflict: "qir_id" })
      .select()
      .single();

    if (error) {
      toast.error("No pudimos sincronizar el registro de stock.");
      return null;
    }

    return mapStockRecord(data);
  };

  const syncScrapFromQir = async (qir: QirRecord) => {
    const { data: existing } = await supabase
      .from("quality_scrap")
      .select("*")
      .eq("qir_id", qir.id)
      .maybeSingle();

    const volume = calculateVolume(existing?.size_l ?? null, existing?.size_w ?? null, existing?.size_h ?? null);
    const approxWeight = calculateWeight(volume ?? existing?.volume_cm3 ?? null);

    const payload = {
      qir_id: qir.id,
      qir_number: qir.qirNumber,
      qir_date: qir.date,
      stock_entry_date: existing?.stock_entry_date ?? qir.sentToStockOrScrapDate ?? null,
      original_job_number: qir.jobNumber,
      material_type: qir.defectType,
      size_l: existing?.size_l ?? null,
      size_w: existing?.size_w ?? null,
      size_h: existing?.size_h ?? null,
      volume_cm3: volume ?? existing?.volume_cm3 ?? null,
      approx_weight: approxWeight ?? existing?.approx_weight ?? null,
      destination: existing?.destination ?? "residuos de manejo especial",
      scrap_exit_date: existing?.scrap_exit_date ?? null,
      comments: existing?.comments ?? null,
    };

    const { data, error } = await supabase
      .from("quality_scrap")
      .upsert(payload, { onConflict: "qir_id" })
      .select()
      .single();

    if (error) {
      toast.error("No pudimos sincronizar el registro de scrap.");
      return null;
    }

    return mapScrapRecord(data);
  };

  const syncScrapFromStock = async (stock: StockRecord) => {
    const volume = calculateVolume(stock.sizeL, stock.sizeW, stock.sizeH);
    const approxWeight = calculateWeight(volume ?? stock.volumeCm3 ?? null);

    const payload = {
      stock_id: stock.id,
      qir_id: stock.qirId,
      qir_number: stock.qirNumber,
      qir_date: stock.qirDate,
      stock_entry_date: stock.stockEntryDate,
      original_job_number: stock.originalJobNumber,
      material_type: stock.materialType,
      size_l: stock.sizeL,
      size_w: stock.sizeW,
      size_h: stock.sizeH,
      volume_cm3: volume ?? stock.volumeCm3 ?? null,
      approx_weight: approxWeight ?? stock.approxWeight ?? null,
      destination: "residuos de manejo especial",
    };

    const { data, error } = await supabase
      .from("quality_scrap")
      .upsert(payload, { onConflict: "stock_id" })
      .select()
      .single();

    if (error) {
      toast.error("No pudimos generar el registro de scrap.");
      return null;
    }

    return mapScrapRecord(data);
  };

  const handleSaveQir = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) {
      toast.error("No tienes permisos para editar QIR.");
      return;
    }
    const payload = buildQirPayload(qirForm);

    if (!payload.date || !payload.detected_by_department) {
      toast.error("Fecha y departamento son obligatorios.");
      return;
    }

    let saved: QirRecord | null = null;

    if (qirForm.id) {
      const { data, error } = await supabase
        .from("quality_qir")
        .update(payload)
        .eq("id", qirForm.id)
        .select()
        .single();
      if (error || !data) {
        toast.error("No pudimos actualizar el QIR.");
        return;
      }
      saved = mapQirRecord(data);
    } else {
      if (!user) {
        toast.error("Necesitas iniciar sesión para crear el QIR.");
        return;
      }

      const { data, error } = await supabase
        .from("quality_qir")
        .insert({ ...payload, created_by: user.id })
        .select()
        .single();
      if (error || !data) {
        toast.error("No pudimos crear el QIR.");
        return;
      }
      saved = mapQirRecord(data);
    }

    if (!saved) return;

    if (saved.materialClassification === "Stock") {
      await supabase.from("quality_scrap").delete().eq("qir_id", saved.id);
      const syncedStock = await syncStockFromQir(saved);
      if (syncedStock) {
        setStockRecords((prev) => {
          const exists = prev.some((item) => item.id === syncedStock.id);
          return exists
            ? prev.map((item) => (item.id === syncedStock.id ? syncedStock : item))
            : [syncedStock, ...prev];
        });
      }
    }

    if (saved.materialClassification === "Scrap") {
      await supabase.from("quality_stock").delete().eq("qir_id", saved.id);
      const syncedScrap = await syncScrapFromQir(saved);
      if (syncedScrap) {
        setScrapRecords((prev) => {
          const exists = prev.some((item) => item.id === syncedScrap.id);
          return exists
            ? prev.map((item) => (item.id === syncedScrap.id ? syncedScrap : item))
            : [syncedScrap, ...prev];
        });
      }
    }

    setQirRecords((prev) => {
      const exists = prev.some((item) => item.id === saved?.id);
      return exists
        ? prev.map((item) => (item.id === saved?.id ? saved : item))
        : [saved, ...prev];
    });

    setQirDialogOpen(false);
    toast.success("QIR guardado.");
  };

  const handleSaveStock = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) return;

    const volume = calculateVolume(stockForm.sizeL, stockForm.sizeW, stockForm.sizeH);
    const approxWeight = calculateWeight(volume ?? stockForm.volumeCm3 ?? null);

    const status =
      stockForm.destination === "scrap"
        ? "scrap"
        : stockForm.destination === "retrabajo" && stockForm.newJobNumber
          ? "retrabajado"
          : "stock";

    const payload = {
      qir_id: stockForm.qirId || null,
      qir_number: stockForm.qirNumber ?? null,
      qir_date: stockForm.qirDate || null,        
      stock_entry_date: stockForm.stockEntryDate || null,
      original_job_number: stockForm.originalJobNumber || null,
      material_type: stockForm.materialType || null,
      size_l: stockForm.sizeL,
      size_w: stockForm.sizeW,
      size_h: stockForm.sizeH,
      volume_cm3: volume ?? stockForm.volumeCm3 ?? null,
      approx_weight: approxWeight ?? stockForm.approxWeight ?? null,
      destination: stockForm.destination || "stock",
      new_job_number: stockForm.newJobNumber || null,
      status,
      stock_exit_date: stockForm.stockExitDate || null,
      comments: stockForm.comments || null,
    };

    const { data, error } = stockForm.id
      ? await supabase
          .from("quality_stock")
          .update(payload)
          .eq("id", stockForm.id)
          .select()
          .single()
      : await supabase
          .from("quality_stock")
          .insert(payload)
          .select()
          .single();

    if (error || !data) {
      toast.error(stockForm.id ? "No pudimos actualizar el stock." : "No pudimos crear el stock.");
      return;
    }

    const saved = mapStockRecord(data);

    setStockRecords((prev) => {
      const exists = prev.some((item) => item.id === saved.id);
      return exists
        ? prev.map((item) => (item.id === saved.id ? saved : item))
        : [saved, ...prev];
    });

    if (saved.destination === "scrap") {
      const syncedScrap = await syncScrapFromStock(saved);
      if (syncedScrap) {
        setScrapRecords((prev) => {
          const exists = prev.some((item) => item.id === syncedScrap.id);
          return exists
            ? prev.map((item) => (item.id === syncedScrap.id ? syncedScrap : item))
            : [syncedScrap, ...prev];
        });
      }
    } else {
      await supabase.from("quality_scrap").delete().eq("stock_id", saved.id);
    }

    setStockDialogOpen(false);
    toast.success(stockForm.id ? "Stock actualizado." : "Stock registrado.");
  };

  const handleSaveScrap = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) return;

    const volume = calculateVolume(scrapForm.sizeL, scrapForm.sizeW, scrapForm.sizeH);
    const approxWeight = calculateWeight(volume ?? scrapForm.volumeCm3 ?? null);

    const payload = {
      qir_id: scrapForm.qirId || null,
      stock_id: scrapForm.stockId || null,
      qir_number: scrapForm.qirNumber ?? null,
      qir_date: scrapForm.qirDate || null,        
      stock_entry_date: scrapForm.stockEntryDate || null,
      original_job_number: scrapForm.originalJobNumber || null,
      material_type: scrapForm.materialType || null,
      size_l: scrapForm.sizeL,
      size_w: scrapForm.sizeW,
      size_h: scrapForm.sizeH,
      volume_cm3: volume ?? scrapForm.volumeCm3 ?? null,
      approx_weight: approxWeight ?? scrapForm.approxWeight ?? null,
      destination: scrapForm.destination || "residuos de manejo especial",
      scrap_exit_date: scrapForm.scrapExitDate || null,
      comments: scrapForm.comments || null,
    };

    const { data, error } = scrapForm.id
      ? await supabase
          .from("quality_scrap")
          .update(payload)
          .eq("id", scrapForm.id)
          .select()
          .single()
      : await supabase
          .from("quality_scrap")
          .insert(payload)
          .select()
          .single();

    if (error || !data) {
      toast.error(scrapForm.id ? "No pudimos actualizar el scrap." : "No pudimos crear el scrap.");
      return;
    }

    const saved = mapScrapRecord(data);
    setScrapRecords((prev) => {
      const exists = prev.some((item) => item.id === saved.id);
      return exists
        ? prev.map((item) => (item.id === saved.id ? saved : item))
        : [saved, ...prev];
    });

    setScrapDialogOpen(false);
    toast.success(scrapForm.id ? "Scrap actualizado." : "Scrap registrado.");
  };

  const handleDeleteQir = async () => {
    if (!qirDeleteTarget) return;
    const { error } = await supabase
      .from("quality_qir")
      .delete()
      .eq("id", qirDeleteTarget.id);

    if (error) {
      toast.error("No pudimos eliminar el QIR.");
      return;
    }

    await supabase.from("quality_stock").delete().eq("qir_id", qirDeleteTarget.id);
    await supabase.from("quality_scrap").delete().eq("qir_id", qirDeleteTarget.id);

    setQirRecords((prev) => prev.filter((item) => item.id !== qirDeleteTarget.id));
    setStockRecords((prev) => prev.filter((item) => item.qirId !== qirDeleteTarget.id));
    setScrapRecords((prev) => prev.filter((item) => item.qirId !== qirDeleteTarget.id));
    setQirDeleteTarget(null);
    toast.success("QIR eliminado.");
  };

  const handleDeleteStock = async () => {
    if (!stockDeleteTarget) return;
    const { error } = await supabase
      .from("quality_stock")
      .delete()
      .eq("id", stockDeleteTarget.id);

    if (error) {
      toast.error("No pudimos eliminar el registro de stock.");
      return;
    }

    await supabase.from("quality_scrap").delete().eq("stock_id", stockDeleteTarget.id);

    setStockRecords((prev) => prev.filter((item) => item.id !== stockDeleteTarget.id));
    setScrapRecords((prev) => prev.filter((item) => item.stockId !== stockDeleteTarget.id));
    setStockDeleteTarget(null);
    toast.success("Stock eliminado.");
  };

  const handleDeleteScrap = async () => {
    if (!scrapDeleteTarget) return;
    const { error } = await supabase
      .from("quality_scrap")
      .delete()
      .eq("id", scrapDeleteTarget.id);

    if (error) {
      toast.error("No pudimos eliminar el registro de scrap.");
      return;
    }

    setScrapRecords((prev) => prev.filter((item) => item.id !== scrapDeleteTarget.id));
    setScrapDeleteTarget(null);
    toast.success("Scrap eliminado.");
  };

  const filteredQirRecords = useMemo(() => {
    return qirRecords.filter((record) => {
      const matchesDate = !qirFilters.date || record.date === qirFilters.date;
      const matchesDepartment =
        !qirFilters.department || record.detectedByDepartment === qirFilters.department;
      const matchesDefect = !qirFilters.defect || record.defectType === qirFilters.defect;
      const matchesResponsible =
        !qirFilters.responsibleArea || record.responsibleArea === qirFilters.responsibleArea;
      const matchesScope = !qirFilters.scope || record.scope === qirFilters.scope;
      const matchesClassification =
        !qirFilters.classification ||
        record.materialClassification === qirFilters.classification;
      const searchTarget = `${record.jobNumber ?? ""} ${record.partName ?? ""} ${
        record.jobName ?? ""
      } ${record.errorPerson ?? ""}`.toLowerCase();
      const matchesSearch = !qirFilters.search || searchTarget.includes(qirFilters.search.toLowerCase());

      return (
        matchesDate &&
        matchesDepartment &&
        matchesDefect &&
        matchesResponsible &&
        matchesScope &&
        matchesClassification &&
        matchesSearch
      );
    });
  }, [qirFilters, qirRecords]);

  const filteredStockRecords = useMemo(() => {
    return stockRecords.filter((record) => {
      const matchesStatus = !stockFilters.status || record.status === stockFilters.status;
      const matchesDestination =
        !stockFilters.destination || record.destination === stockFilters.destination;
      const days = calculateDays(record.stockEntryDate, record.stockExitDate);
      const matchesDaysRange = (() => {
        if (!stockFilters.daysRange) return true;
        if (!days) return false;
        if (stockFilters.daysRange === "0-7") return days <= 7;
        if (stockFilters.daysRange === "8-30") return days > 7 && days <= 30;
        if (stockFilters.daysRange === "31-90") return days > 30 && days <= 90;
        if (stockFilters.daysRange === "90+") return days > 90;
        return true;
      })();
      const searchTarget = `${record.qirNumber ?? ""} ${record.originalJobNumber ?? ""} ${
        record.newJobNumber ?? ""
      }`.toLowerCase();
      const matchesSearch = !stockFilters.search || searchTarget.includes(stockFilters.search.toLowerCase());

      return matchesStatus && matchesDestination && matchesDaysRange && matchesSearch;
    });
  }, [stockFilters, stockRecords]);

  const filteredScrapRecords = useMemo(() => {
    return scrapRecords.filter((record) => {
      const matchesExitDate =
        !scrapFilters.exitDate || record.scrapExitDate === scrapFilters.exitDate;
      const days = calculateDays(record.stockEntryDate, record.scrapExitDate);
      const matchesDaysRange = (() => {
        if (!scrapFilters.daysRange) return true;
        if (!days) return false;
        if (scrapFilters.daysRange === "0-7") return days <= 7;
        if (scrapFilters.daysRange === "8-30") return days > 7 && days <= 30;
        if (scrapFilters.daysRange === "31-90") return days > 30 && days <= 90;
        if (scrapFilters.daysRange === "90+") return days > 90;
        return true;
      })();
      const searchTarget = `${record.qirNumber ?? ""} ${record.originalJobNumber ?? ""}`.toLowerCase();
      const matchesSearch = !scrapFilters.search || searchTarget.includes(scrapFilters.search.toLowerCase());

      return matchesExitDate && matchesDaysRange && matchesSearch;
    });
  }, [scrapFilters, scrapRecords]);

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-gradient-to-br from-slate-50 via-white to-sky-50">
        <section className="relative overflow-hidden rounded-3xl border border-slate-200 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-sky-500/10 to-indigo-500/10" />
          <div className="relative space-y-3">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-sky-600">
              Operación · Calidad
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">Quality</h1>
            <p className="max-w-5xl text-base text-gray-600">
              Registra los QIR y controla la salida ISO 14001 hacia stock o scrap con
              trazabilidad entre matrices.
            </p>
            <div className="mt-6 flex flex-wrap items-center gap-3">
              <Button
                variant="outline"
                className="gap-2"
                onClick={reloadData}
                disabled={loading}
              >
                <RefreshCcw className="h-4 w-4" />
                Actualizar datos
              </Button>
              {!canManage && (
                <span className="flex items-center gap-2 text-sm text-gray-500">
                  <AlertCircle className="h-4 w-4" />
                  Modo lectura en QIR, Stock y Scrap.
                </span>
              )}
            </div>
          </div>
        </section>

        <Tabs defaultValue="qir" className="space-y-6">
          <TabsList className="grid w-full max-w-2xl grid-cols-4">
            <TabsTrigger value="qir">Errores de Calidad (QIR)</TabsTrigger>
            <TabsTrigger value="stock">Stock de Material</TabsTrigger>
            <TabsTrigger value="scrap">Scrap</TabsTrigger>
            <TabsTrigger value="stats">Estadísticas</TabsTrigger>
          </TabsList>

          <TabsContent value="qir" className="space-y-6">
            <Card className="border-sky-100">
              <CardHeader className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle className="text-xl text-gray-900">Matriz QIR</CardTitle>
                  <p className="text-sm text-gray-500">
                    Registra los errores de calidad y define la clasificación del material.
                  </p>
                </div>
                {canManage && (
                  <Button className="gap-2" onClick={() => openQirDialog()}>
                    <Plus className="h-4 w-4" />
                    Nuevo QIR
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-3 md:grid-cols-3 xl:grid-cols-6">
                  <div className="space-y-1">
                    <Label>Fecha</Label>
                    <Input
                      type="date"
                      value={qirFilters.date}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, date: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Depto</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirFilters.department}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, department: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      {departmentOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Tipo defecto</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirFilters.defect}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, defect: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      {defectOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Área responsable</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirFilters.responsibleArea}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, responsibleArea: event.target.value }))
                      }
                    >
                      <option value="">Todas</option>
                      {departmentOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Alcance</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirFilters.scope}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, scope: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      {scopeOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Clasificación</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirFilters.classification}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, classification: event.target.value }))
                      }
                    >
                      <option value="">Todas</option>
                      {classificationOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1 md:col-span-3 xl:col-span-2">
                    <Label>Búsqueda</Label>
                    <Input
                      placeholder="Número trabajo, parte, trabajo, persona"
                      value={qirFilters.search}
                      onChange={(event) =>
                        setQirFilters((prev) => ({ ...prev, search: event.target.value }))
                      }
                    />
                  </div>
                </div>

                <div className="overflow-hidden rounded-xl border border-slate-200 bg-white">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead># QIR</TableHead>
                        <TableHead>Fecha</TableHead>
                        <TableHead>Depto detecta</TableHead>
                        <TableHead>Número de trabajo</TableHead>
                        <TableHead>Tipo defecto</TableHead>
                        <TableHead>Clasificación</TableHead>
                        <TableHead>Salida ISO 14001</TableHead>
                        <TableHead className="text-right">Acciones</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {loading && (
                        <TableRow>
                          <TableCell colSpan={8}>Cargando registros...</TableCell>
                        </TableRow>
                      )}
                      {!loading && filteredQirRecords.length === 0 && (
                        <TableRow>
                          <TableCell colSpan={8}>Sin registros para los filtros actuales.</TableCell>
                        </TableRow>
                      )}
                      {!loading &&
                        filteredQirRecords.map((record) => (
                          <TableRow key={record.id}>
                            <TableCell className="font-medium text-gray-900">
                              {record.qirNumber ?? "-"}
                            </TableCell>
                            <TableCell>{formatDate(record.date)}</TableCell>
                            <TableCell>{record.detectedByDepartment ?? "-"}</TableCell>
                            <TableCell>{record.jobNumber ?? "-"}</TableCell>
                            <TableCell>{record.defectType ?? "-"}</TableCell>
                            <TableCell>{record.materialClassification ?? "-"}</TableCell>
                            <TableCell>{formatDate(record.sentToStockOrScrapDate)}</TableCell>
                            <TableCell className="text-right">
                              <div className="flex justify-end gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => openQirDialog(record)}
                                >
                                  {canManage ? "Editar" : "Ver"}
                                </Button>
                                {canManage && (
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="gap-1 text-rose-600"
                                    onClick={() => setQirDeleteTarget(record)}
                                  >
                                    <Trash2 className="h-4 w-4" />
                                    Eliminar
                                  </Button>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="stats" className="space-y-6">
            <section className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
              <Card>
                <CardHeader className="space-y-1">
                  <CardTitle className="text-sm font-medium text-slate-500">
                    QIR registrados
                  </CardTitle>
                  <p className="text-2xl font-semibold text-slate-900">
                    {qualityStats.totalQir}
                  </p>
                </CardHeader>
                <CardContent className="text-xs text-slate-500">
                  Matrices con errores de calidad documentados.
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="space-y-1">
                  <CardTitle className="text-sm font-medium text-slate-500">
                    Costo perdido estimado
                  </CardTitle>
                  <p className="text-2xl font-semibold text-slate-900">
                    {currencyFormatter.format(qualityStats.totalLostCost)}
                  </p>
                </CardHeader>
                <CardContent className="text-xs text-slate-500">
                  Suma reportada en los QIR.
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="space-y-1">
                  <CardTitle className="text-sm font-medium text-slate-500">
                    Piezas en scrap
                  </CardTitle>
                  <p className="text-2xl font-semibold text-slate-900">
                    {qualityStats.totalScrapPieces}
                  </p>
                </CardHeader>
                <CardContent className="text-xs text-slate-500">
                  Registros totales enviados a scrap.
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="space-y-1">
                  <CardTitle className="text-sm font-medium text-slate-500">
                    Volumen en scrap
                  </CardTitle>
                  <p className="text-2xl font-semibold text-slate-900">
                    {numberFormatter.format(qualityStats.totalScrapVolume)} cm³
                  </p>
                </CardHeader>
                <CardContent className="text-xs text-slate-500">
                  Volumen acumulado de piezas en scrap.
                </CardContent>
              </Card>
            </section>

            <section className="grid gap-4 lg:grid-cols-[2fr_1fr]">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base font-semibold text-slate-900">
                    Tendencia mensual de QIR
                  </CardTitle>
                  <p className="text-sm text-slate-500">
                    Conteo de QIR registrados por mes.
                  </p>
                </CardHeader>
                <CardContent className="space-y-3">
                  {monthlyTrend.length === 0 && (
                    <p className="text-sm text-slate-500">
                      Sin registros mensuales todavía.
                    </p>
                  )}
                  {monthlyTrend.length > 0 && (
                    <div className="space-y-3">
                      {monthlyTrend.map(({ month, count }) => {
                        const width = maxMonthlyCount
                          ? Math.round((count / maxMonthlyCount) * 100)
                          : 0;
                        return (
                          <div key={month} className="space-y-1">
                            <div className="flex items-center justify-between text-xs text-slate-500">
                              <span>{month}</span>
                              <span>{count} QIR</span>
                            </div>
                            <div className="h-2 w-full rounded-full bg-slate-100">
                              <div
                                className="h-2 rounded-full bg-sky-500"
                                style={{ width: `${width}%` }}
                              />
                            </div>
                          </div>
                        );
                      })}
                    </div>
                  )}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-base font-semibold text-slate-900">
                    Tiempo perdido por área / máquina
                  </CardTitle>
                  <p className="text-sm text-slate-500">
                    Suma de horas reportadas en QIR.
                  </p>
                </CardHeader>
                <CardContent className="space-y-2">
                  {timeLossStats.length === 0 && (
                    <p className="text-sm text-slate-500">
                      Sin tiempos capturados.
                    </p>
                  )}
                  {timeLossStats.map((entry) => (
                    <div key={entry.label} className="flex items-center justify-between text-sm">
                      <span className="text-slate-600">{entry.label}</span>
                      <span className="font-medium text-slate-900">
                        {entry.value.toFixed(1)} h
                      </span>
                    </div>
                  ))}
                </CardContent>
              </Card>
            </section>

            <section className="grid gap-4 lg:grid-cols-2">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base font-semibold text-slate-900">
                    Factor identification
                  </CardTitle>
                  <p className="text-sm text-slate-500">
                    Distribución por tipo de factor reportado.
                  </p>
                </CardHeader>
                <CardContent className="space-y-2">
                  {factorStats.length === 0 && (
                    <p className="text-sm text-slate-500">
                      Sin factores reportados.
                    </p>
                  )}
                  {factorStats.map((entry) => (
                    <div key={entry.label} className="flex items-center justify-between text-sm">
                      <span className="text-slate-600">{entry.label}</span>
                      <span className="font-medium text-slate-900">
                        {entry.count}
                      </span>
                    </div>
                  ))}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-base font-semibold text-slate-900">
                    Área responsable
                  </CardTitle>
                  <p className="text-sm text-slate-500">
                    QIR agrupados por área responsable.
                  </p>
                </CardHeader>
                <CardContent className="space-y-2">
                  {responsibleAreaStats.length === 0 && (
                    <p className="text-sm text-slate-500">
                      Sin áreas responsables reportadas.
                    </p>
                  )}
                  {responsibleAreaStats.map((entry) => (
                    <div key={entry.label} className="flex items-center justify-between text-sm">
                      <span className="text-slate-600">{entry.label}</span>
                      <span className="font-medium text-slate-900">
                        {entry.count}
                      </span>
                    </div>
                  ))}
                </CardContent>
              </Card>
            </section>
          </TabsContent>

          <TabsContent value="stock" className="space-y-6">
            <Card className="border-emerald-100">
              <CardHeader className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle className="text-xl text-gray-900">Matriz de Stock</CardTitle>
                  <p className="text-sm text-gray-500">
                    Registros generados desde QIR con salida ISO 14001 = Stock.
                  </p>
                </div>
                {canManage && (
                  <Button className="gap-2" onClick={() => openStockDialog()}>
                    <Plus className="h-4 w-4" />
                    Agregar registro
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-3 md:grid-cols-3 xl:grid-cols-6">
                  <div className="space-y-1">
                    <Label>Estado</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={stockFilters.status}
                      onChange={(event) =>
                        setStockFilters((prev) => ({ ...prev, status: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      <option value="stock">stock</option>
                      <option value="scrap">scrap</option>
                      <option value="retrabajado">retrabajado</option>
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Destino</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={stockFilters.destination}
                      onChange={(event) =>
                        setStockFilters((prev) => ({ ...prev, destination: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      {stockDestinationOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Días en stock</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={stockFilters.daysRange}
                      onChange={(event) =>
                        setStockFilters((prev) => ({ ...prev, daysRange: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      <option value="0-7">0-7</option>
                      <option value="8-30">8-30</option>
                      <option value="31-90">31-90</option>
                      <option value="90+">90+</option>
                    </select>
                  </div>
                  <div className="space-y-1 md:col-span-3 xl:col-span-3">
                    <Label>Búsqueda</Label>
                    <Input
                      placeholder="QIR, trabajo original/nuevo"
                      value={stockFilters.search}
                      onChange={(event) =>
                        setStockFilters((prev) => ({ ...prev, search: event.target.value }))
                      }
                    />
                  </div>
                </div>

                <div className="overflow-hidden rounded-xl border border-slate-200 bg-white">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead># QIR</TableHead>
                        <TableHead>Ingreso a stock</TableHead>
                        <TableHead>Trabajo original</TableHead>
                        <TableHead>Destino</TableHead>
                        <TableHead>Estado</TableHead>
                        <TableHead>Días en stock</TableHead>
                        <TableHead className="text-right">Acciones</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {loading && (
                        <TableRow>
                          <TableCell colSpan={7}>Cargando registros...</TableCell>
                        </TableRow>
                      )}
                      {!loading && filteredStockRecords.length === 0 && (
                        <TableRow>
                          <TableCell colSpan={7}>Sin registros para los filtros actuales.</TableCell>
                        </TableRow>
                      )}
                      {!loading &&
                        filteredStockRecords.map((record) => (
                          <TableRow key={record.id}>
                            <TableCell className="font-medium text-gray-900">
                              {record.qirNumber ?? "-"}
                            </TableCell>
                            <TableCell>{formatDate(record.stockEntryDate)}</TableCell>
                            <TableCell>{record.originalJobNumber ?? "-"}</TableCell>
                            <TableCell>{record.destination ?? "-"}</TableCell>
                            <TableCell>{record.status ?? "-"}</TableCell>
                            <TableCell>
                              {calculateDays(record.stockEntryDate, record.stockExitDate) ?? "-"}
                            </TableCell>
                            <TableCell className="text-right">
                              <div className="flex justify-end gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => openStockDialog(record)}
                                  disabled={!canManage}
                                >
                                  Editar
                                </Button>
                                {canManage && (
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="gap-1 text-rose-600"
                                    onClick={() => setStockDeleteTarget(record)}
                                  >
                                    <Trash2 className="h-4 w-4" />
                                    Eliminar
                                  </Button>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="scrap" className="space-y-6">
            <Card className="border-rose-100">
              <CardHeader className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle className="text-xl text-gray-900">Matriz de Scrap</CardTitle>
                  <p className="text-sm text-gray-500">
                    Registros generados desde QIR o desde cambios de destino en stock.
                  </p>
                </div>
                {canManage && (
                  <Button className="gap-2" onClick={() => openScrapDialog()}>
                    <Plus className="h-4 w-4" />
                    Agregar registro
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-3 md:grid-cols-3 xl:grid-cols-6">
                  <div className="space-y-1">
                    <Label>Fecha salida</Label>
                    <Input
                      type="date"
                      value={scrapFilters.exitDate}
                      onChange={(event) =>
                        setScrapFilters((prev) => ({ ...prev, exitDate: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Días en scrap</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={scrapFilters.daysRange}
                      onChange={(event) =>
                        setScrapFilters((prev) => ({ ...prev, daysRange: event.target.value }))
                      }
                    >
                      <option value="">Todos</option>
                      <option value="0-7">0-7</option>
                      <option value="8-30">8-30</option>
                      <option value="31-90">31-90</option>
                      <option value="90+">90+</option>
                    </select>
                  </div>
                  <div className="space-y-1 md:col-span-2 xl:col-span-4">
                    <Label>Búsqueda</Label>
                    <Input
                      placeholder="QIR, trabajo original"
                      value={scrapFilters.search}
                      onChange={(event) =>
                        setScrapFilters((prev) => ({ ...prev, search: event.target.value }))
                      }
                    />
                  </div>
                </div>

                <div className="overflow-hidden rounded-xl border border-slate-200 bg-white">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead># QIR</TableHead>
                        <TableHead>Ingreso a scrap</TableHead>
                        <TableHead>Trabajo original</TableHead>
                        <TableHead>Destino</TableHead>
                        <TableHead>Días en scrap</TableHead>
                        <TableHead className="text-right">Acciones</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {loading && (
                        <TableRow>
                          <TableCell colSpan={6}>Cargando registros...</TableCell>
                        </TableRow>
                      )}
                      {!loading && filteredScrapRecords.length === 0 && (
                        <TableRow>
                          <TableCell colSpan={6}>Sin registros para los filtros actuales.</TableCell>
                        </TableRow>
                      )}
                      {!loading &&
                        filteredScrapRecords.map((record) => (
                          <TableRow key={record.id}>
                            <TableCell className="font-medium text-gray-900">
                              {record.qirNumber ?? "-"}
                            </TableCell>
                            <TableCell>{formatDate(record.stockEntryDate)}</TableCell>
                            <TableCell>{record.originalJobNumber ?? "-"}</TableCell>
                            <TableCell>{record.destination ?? "-"}</TableCell>
                            <TableCell>
                              {calculateDays(record.stockEntryDate, record.scrapExitDate) ?? "-"}
                            </TableCell>
                            <TableCell className="text-right">
                              <div className="flex justify-end gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => openScrapDialog(record)}
                                  disabled={!canManage}
                                >
                                  Editar
                                </Button>
                                {canManage && (
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="gap-1 text-rose-600"
                                    onClick={() => setScrapDeleteTarget(record)}
                                  >
                                    <Trash2 className="h-4 w-4" />
                                    Eliminar
                                  </Button>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>

      <Dialog open={qirDialogOpen} onOpenChange={setQirDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto sm:max-w-5xl">
          <DialogHeader>
            <DialogTitle>
              {qirForm.id
                ? `${canManage ? "Editar" : "Detalle"} QIR #${qirForm.qirNumber ?? ""}`
                : "Nuevo QIR"}
            </DialogTitle>
          </DialogHeader>
          <form className="space-y-8" onSubmit={handleSaveQir}>
            <fieldset disabled={!canManage} className="space-y-8">
              <div className="space-y-4">
                <div className="flex items-center gap-2 text-sm font-semibold text-sky-700">
                  <ClipboardList className="h-4 w-4" />
                  Descripción de la no conformidad
                </div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label>N° (Consecutivo)</Label>
                    <Input value={qirForm.qirNumber ?? "Auto"} disabled />
                  </div>
                  <div className="space-y-1">
                    <Label>Fecha</Label>
                    <Input
                      type="date"
                      value={qirForm.date ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, date: event.target.value }))
                      }
                      required
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Depto que detecta</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.detectedByDepartment ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, detectedByDepartment: event.target.value }))
                      }
                      required
                    >
                      <option value="">Selecciona</option>
                      {departmentOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Número de Trabajo</Label>
                    <Input
                      value={qirForm.jobNumber ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, jobNumber: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Nombre de la Parte</Label>
                    <Input
                      value={qirForm.partName ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, partName: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Nombre del Trabajo</Label>
                    <Input
                      value={qirForm.jobName ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, jobName: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1 md:col-span-2">
                    <Label>Tipo de Defecto</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.defectType ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, defectType: event.target.value }))
                      }
                    >
                      <option value="">Selecciona</option>
                      {defectOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1 md:col-span-2">
                    <Label>Detalle del Defecto</Label>
                    <Textarea
                      rows={3}
                      value={qirForm.defectDetail ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, defectDetail: event.target.value }))
                      }
                    />
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <div className="flex items-center gap-2 text-sm font-semibold text-indigo-700">
                  <ClipboardCheck className="h-4 w-4" />
                  Intervención en el Molde / Tiempo Pérdido
                </div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1 md:col-span-2">
                    <Label>Acción correctiva</Label>
                    <Textarea
                      rows={2}
                      value={qirForm.correctiveAction ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, correctiveAction: event.target.value }))
                      }
                    />
                  </div>
                  {[
                    { key: "timeManDs", label: "Tiempo Hombre DS" },
                    { key: "timeManPe", label: "Tiempo Hombre PE" },
                    { key: "timeManP1", label: "Tiempo Hombre P1" },
                    { key: "timeCnc", label: "Tiempo máquina CNC" },
                    { key: "timeEdm", label: "Tiempo máquina EDM" },
                    { key: "timeWc", label: "Tiempo máquina WC" },
                    { key: "timeMm", label: "Tiempo máquina MM" },
                    { key: "timeLathe", label: "Tiempo máquina Torno" },
                    { key: "timeManIns", label: "Tiempo Hombre INS" },
                    { key: "timeCmm", label: "Tiempo máquina CMM" },
                    { key: "timeKeyence", label: "Tiempo máquina KEYENCE" },
                    { key: "timeManP2", label: "Tiempo Hombre P2" },
                    { key: "timeOven", label: "Tiempo máquina HORNO" },
                    { key: "timeWelding", label: "Tiempo máquina SOLDADORA" },
                    { key: "timeManCommercial", label: "Tiempo Hombre COMERCIAL" },
                    { key: "materialCost", label: "Costo de Material Nuevo (USD)" },
                    { key: "lostCost", label: "Costo Perdido (USD)" },
                  ].map(({ key, label }) => (
                    <div key={key} className="space-y-1">
                      <Label>{label}</Label>
                      <Input
                        type="number"
                        step="0.01"
                        value={qirForm[key as keyof QirRecord] ?? ""}
                        onChange={(event) =>
                          setQirForm((prev) => ({
                            ...prev,
                            [key]: parseNumber(event.target.value),
                          }))
                        }
                      />
                    </div>
                  ))}
                </div>
              </div>

              <div className="space-y-4">
                <div className="text-sm font-semibold text-slate-700">Factores</div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label>Persona del Error</Label>
                    <Input
                      value={qirForm.errorPerson ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, errorPerson: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Factor Identification</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.factorIdentification ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({
                          ...prev,
                          factorIdentification: event.target.value,
                        }))
                      }
                    >
                      <option value="">Selecciona</option>
                      {factorOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Área responsable</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.responsibleArea ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, responsibleArea: event.target.value }))
                      }
                    >
                      <option value="">Selecciona</option>
                      {departmentOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <div className="text-sm font-semibold text-slate-700">Seguimiento QC</div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label>Documento</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.documentType ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, documentType: event.target.value }))
                      }
                    >
                      <option value="">Selecciona</option>
                      {documentOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Alcance</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.scope ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, scope: event.target.value }))
                      }
                    >
                      <option value="">Selecciona</option>
                      {scopeOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <div className="text-sm font-semibold text-slate-700">ISO 14001</div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label>Clasificación del Material</Label>
                    <select
                      className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                      value={qirForm.materialClassification ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({
                          ...prev,
                          materialClassification: event.target.value as QirRecord["materialClassification"],
                        }))
                      }
                    >
                      {classificationOptions.map((option) => (
                        <option key={option} value={option}>
                          {option}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="space-y-1">
                    <Label>Fecha envío a stock o scrap</Label>
                    <Input
                      type="date"
                      value={qirForm.sentToStockOrScrapDate ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({
                          ...prev,
                          sentToStockOrScrapDate: event.target.value,
                        }))
                      }
                    />
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <div className="text-sm font-semibold text-slate-700">
                  Medidas para evitar recurrencia
                </div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label>Fecha envío Why Why</Label>
                    <Input
                      type="date"
                      value={qirForm.whyWhySentDate ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, whyWhySentDate: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Contramedida (Estándar, Flujo, IT)</Label>
                    <Input
                      value={qirForm.countermeasure ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, countermeasure: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Socialización</Label>
                    <Input
                      value={qirForm.socialization ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, socialization: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Aplicación</Label>
                    <Input
                      value={qirForm.application ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, application: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Número trabajo aplicación</Label>
                    <Input
                      value={qirForm.applicationJobNumber ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, applicationJobNumber: event.target.value }))
                      }
                    />
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <div className="text-sm font-semibold text-slate-700">Cierre de QIR</div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label>Retraso envío a QC</Label>
                    <Input
                      value={qirForm.qcDelay ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, qcDelay: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Retraso en estándar</Label>
                    <Input
                      value={qirForm.standardDelay ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, standardDelay: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Retraso en Socialización</Label>
                    <Input
                      value={qirForm.socializationDelay ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, socializationDelay: event.target.value }))
                      }
                    />
                  </div>
                  <div className="space-y-1">
                    <Label>Retraso en Aplicación</Label>
                    <Input
                      value={qirForm.applicationDelay ?? ""}
                      onChange={(event) =>
                        setQirForm((prev) => ({ ...prev, applicationDelay: event.target.value }))
                      }
                    />
                  </div>
                </div>
              </div>
            </fieldset>

            <DialogFooter className="gap-2 sm:justify-end">
              <Button variant="outline" type="button" onClick={() => setQirDialogOpen(false)}>
                Cancelar
              </Button>
              {canManage && (
                <Button type="submit" className="gap-2">
                  Guardar QIR
                </Button>
              )}
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={stockDialogOpen} onOpenChange={setStockDialogOpen}>
        <DialogContent className="max-h-[85vh] overflow-y-auto sm:max-w-3xl">
          <DialogHeader>
            <DialogTitle>
              {`Stock QIR #${stockForm.qirNumber ?? ""}`}
            </DialogTitle>
          </DialogHeader>
          <form className="space-y-4" onSubmit={handleSaveStock}>
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-1">
                <Label>Número QIR</Label>
                <Input
                  type="number"
                  value={stockForm.qirNumber ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({
                      ...prev,
                      qirNumber: parseNumber(event.target.value),
                    }))
                  }
                  disabled={Boolean(stockForm.qirId)}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha QIR</Label>
                <Input
                  type="date"
                  value={stockForm.qirDate ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, qirDate: event.target.value }))
                  }
                  disabled={Boolean(stockForm.qirId)}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha ingreso a stock</Label>
                <Input
                  type="date"
                  value={stockForm.stockEntryDate ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, stockEntryDate: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Trabajo original</Label>
                <Input
                  value={stockForm.originalJobNumber ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, originalJobNumber: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1 md:col-span-2">
                <Label>Tipo de material</Label>
                <Input
                  value={stockForm.materialType ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, materialType: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Tamaño L (cm)</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={stockForm.sizeL ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, sizeL: parseNumber(event.target.value) }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Tamaño W (cm)</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={stockForm.sizeW ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, sizeW: parseNumber(event.target.value) }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Tamaño H (cm)</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={stockForm.sizeH ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, sizeH: parseNumber(event.target.value) }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Destino</Label>
                <select
                  className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                  value={stockForm.destination ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({
                      ...prev,
                      destination: event.target.value as StockRecord["destination"],
                    }))
                  }
                >
                  {stockDestinationOptions.map((option) => (
                    <option key={option} value={option}>
                      {option}
                    </option>
                  ))}
                </select>
              </div>
              <div className="space-y-1">
                <Label>Número trabajo nuevo</Label>
                <Input
                  value={stockForm.newJobNumber ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, newJobNumber: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha salida del stock</Label>
                <Input
                  type="date"
                  value={stockForm.stockExitDate ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, stockExitDate: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1 md:col-span-2">
                <Label>Comentarios</Label>
                <Textarea
                  rows={2}
                  value={stockForm.comments ?? ""}
                  onChange={(event) =>
                    setStockForm((prev) => ({ ...prev, comments: event.target.value }))
                  }
                />
              </div>
            </div>
            <div className="grid gap-2 text-sm text-gray-600 md:grid-cols-2">
              <span>Volumen estimado: {calculateVolume(stockForm.sizeL, stockForm.sizeW, stockForm.sizeH) ?? "-"} cm³</span>
              <span>Peso aproximado: {calculateWeight(calculateVolume(stockForm.sizeL, stockForm.sizeW, stockForm.sizeH)) ?? "-"} kg</span>
            </div>
            <DialogFooter className="gap-2 sm:justify-end">
              <Button variant="outline" type="button" onClick={() => setStockDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit">Guardar cambios</Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={scrapDialogOpen} onOpenChange={setScrapDialogOpen}>
        <DialogContent className="max-h-[85vh] overflow-y-auto sm:max-w-3xl">
          <DialogHeader>
            <DialogTitle>{`Scrap QIR #${scrapForm.qirNumber ?? ""}`}</DialogTitle>
          </DialogHeader>
          <form className="space-y-4" onSubmit={handleSaveScrap}>
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-1">
                <Label>Número QIR</Label>
                <Input
                  type="number"
                  value={scrapForm.qirNumber ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({
                      ...prev,
                      qirNumber: parseNumber(event.target.value),
                    }))
                  }
                  disabled={Boolean(scrapForm.qirId || scrapForm.stockId)}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha QIR</Label>
                <Input
                  type="date"
                  value={scrapForm.qirDate ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, qirDate: event.target.value }))
                  }
                  disabled={Boolean(scrapForm.qirId || scrapForm.stockId)}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha ingreso a scrap</Label>
                <Input
                  type="date"
                  value={scrapForm.stockEntryDate ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, stockEntryDate: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Trabajo original</Label>
                <Input
                  value={scrapForm.originalJobNumber ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, originalJobNumber: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1 md:col-span-2">
                <Label>Tipo de material</Label>
                <Input
                  value={scrapForm.materialType ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, materialType: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Tamaño L (cm)</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={scrapForm.sizeL ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, sizeL: parseNumber(event.target.value) }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Tamaño W (cm)</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={scrapForm.sizeW ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, sizeW: parseNumber(event.target.value) }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Tamaño H (cm)</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={scrapForm.sizeH ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, sizeH: parseNumber(event.target.value) }))
                  }
                />
              </div>
              <div className="space-y-1">
                <Label>Destino</Label>
                <Input value={scrapForm.destination ?? ""} disabled />
              </div>
              <div className="space-y-1">
                <Label>Fecha salida del scrap</Label>
                <Input
                  type="date"
                  value={scrapForm.scrapExitDate ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, scrapExitDate: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-1 md:col-span-2">
                <Label>Comentarios</Label>
                <Textarea
                  rows={2}
                  value={scrapForm.comments ?? ""}
                  onChange={(event) =>
                    setScrapForm((prev) => ({ ...prev, comments: event.target.value }))
                  }
                />
              </div>
            </div>
            <div className="grid gap-2 text-sm text-gray-600 md:grid-cols-2">
              <span>Volumen estimado: {calculateVolume(scrapForm.sizeL, scrapForm.sizeW, scrapForm.sizeH) ?? "-"} cm³</span>
              <span>Peso aproximado: {calculateWeight(calculateVolume(scrapForm.sizeL, scrapForm.sizeW, scrapForm.sizeH)) ?? "-"} kg</span>
            </div>
            <DialogFooter className="gap-2 sm:justify-end">
              <Button variant="outline" type="button" onClick={() => setScrapDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit">Guardar cambios</Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <AlertDialog
        open={Boolean(qirDeleteTarget)}
        onOpenChange={(open) => !open && setQirDeleteTarget(null)}
      >
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar QIR?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction className="bg-rose-600 hover:bg-rose-700" onClick={handleDeleteQir}>
              Eliminar
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <AlertDialog
        open={Boolean(stockDeleteTarget)}
        onOpenChange={(open) => !open && setStockDeleteTarget(null)}
      >
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar registro de stock?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction className="bg-rose-600 hover:bg-rose-700" onClick={handleDeleteStock}>
              Eliminar
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <AlertDialog
        open={Boolean(scrapDeleteTarget)}
        onOpenChange={(open) => !open && setScrapDeleteTarget(null)}
      >
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar registro de scrap?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction className="bg-rose-600 hover:bg-rose-700" onClick={handleDeleteScrap}>
              Eliminar
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}