"use client";

import { useEffect, useState, type FormEvent } from "react";
import Link from "next/link";
import { FileText, FileUp, Trash2 } from "lucide-react";
import { User } from "@supabase/supabase-js";
import { supabase, type Database } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { toast } from "sonner";

const residuosPeligrosos = [
  "Lámparas fluorescentes",
  "Material sólido impregnado con grasa y/o aceite",
  "Químicos contaminados",
  "Líquidos contaminados con aceite y/o grasa",
  "Sólido impregnado con solvente",
  "Lodos con residuos ferrosos",
  "Lodos contaminados",
];

const initialPermits = [
  {
    id: "permiso-municipal",
    title: "Permiso municipal de recolección RP",
    description: "Autorización vigente del municipio para manejo de RP.",
    href: "/meiwa-core/meiwa-sgi/operation/rp/registro-plan-manejo.pdf",
  },
  {
    id: "permiso-proveedor",
    title: "Permiso del proveedor externo",
    description: "Licencia del proveedor para transporte y disposición final.",
    href: "/meiwa-core/meiwa-sgi/operation/rp/registro-plan-manejo.pdf",
  },
];

const registroBase = {
  id: "registro-plan-manejo",
  title: "Registro y plan de manejo RP",
  description: "Documento oficial del plan de manejo de residuos peligrosos.",
  href: "/meiwa-core/meiwa-sgi/operation/rp/registro-plan-manejo.pdf",
};
const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);

type Registro = typeof registroBase;

type Permit = (typeof initialPermits)[number];

type RpFile = Database["public"]["Tables"]["rp_files"]["Row"];

type IngresoEntry = {
  id: string;
  residuo: string;
  fecha: string;
  cantidadKg: string;
  codigoPeligrosidad: string;
  areaProceso: string;
  responsable: string;
};

type SalidaEntry = {
  id: string;
  residuo: string;
  fecha: string;
  cantidad: string;
  codigoPeligrosidad: string;
  numeroManifiesto: string;
  registroAmbiental: string;
  responsable: string;
  transportista: string;
  autorizacionTransporte: string;
  responsableTransporte: string;
  destinoFinal: string;
  autorizacionDestinoFinal: string;
  responsableDestinoFinal: string;
};

type UploadedFile = {
  id: string;
  name: string;
  fileDate: string;
};

export default function RpPage() {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [registroId, setRegistroId] = useState<string | null>(null);
  const [registroDoc, setRegistroDoc] = useState<Registro>(registroBase);
  const [registroForm, setRegistroForm] = useState({
    title: registroBase.title,
    href: registroBase.href,
  });  
  const [permits, setPermits] = useState<Permit[]>(initialPermits);
  const [permitForm, setPermitForm] = useState({
    title: "",
    description: "",
    href: "",
  });  
  const [ingresoEntries, setIngresoEntries] = useState<IngresoEntry[]>([]);
  const [salidaEntries, setSalidaEntries] = useState<SalidaEntry[]>([]);
  const [facturas, setFacturas] = useState<UploadedFile[]>([]);
  const [manifiestos, setManifiestos] = useState<UploadedFile[]>([]);
  const [facturaFile, setFacturaFile] = useState<File | null>(null);
  const [manifiestoFile, setManifiestoFile] = useState<File | null>(null);
  const [ingresoForm, setIngresoForm] = useState({
    residuo: residuosPeligrosos[0],
    fecha: "",
    cantidadKg: "",
    codigoPeligrosidad: "",
    areaProceso: "",
    responsable: "",
  });
  const [salidaForm, setSalidaForm] = useState({
    residuo: residuosPeligrosos[0],
    fecha: "",
    cantidad: "",
    codigoPeligrosidad: "",
    numeroManifiesto: "",
    registroAmbiental: "",
    responsable: "",
    transportista: "",
    autorizacionTransporte: "",
    responsableTransporte: "",
    destinoFinal: "",
    autorizacionDestinoFinal: "",
    responsableDestinoFinal: "",
  });

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const sessionUser = data.session?.user ?? null;
      setUser(sessionUser);
      if (!sessionUser) {
        setLoading(false);
      }
    };

    loadSession();

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      const sessionUser = session?.user ?? null;
      setUser(sessionUser);
      if (!sessionUser) {
        setLoading(false);
      }
    });

    return () => subscription.unsubscribe();
  }, []);

  const canManageDocs = allowedManagers.has(user?.email ?? "");

  const loadRegistroDoc = async () => {
    const { data, error } = await supabase
      .from("rp_registro_docs")
      .select("*")
      .order("created_at", { ascending: false })
      .limit(1);

    if (error) {
      toast.error("No pudimos cargar el registro de manejo.");
      return;
    }

    const registro = data?.[0];
    if (registro) {
      setRegistroId(registro.id);
      setRegistroDoc({
        id: registro.id,
        title: registro.title,
        description: registro.description ?? registroBase.description,
        href: registro.href,
      });
      setRegistroForm({ title: registro.title, href: registro.href });
    } else {
      setRegistroId(null);
      setRegistroDoc(registroBase);
      setRegistroForm({ title: registroBase.title, href: registroBase.href });
    }
  };

  const loadPermits = async () => {
    const { data, error } = await supabase
      .from("rp_permits")
      .select("*")
      .order("created_at", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar los permisos.");
      return;
    }

    const mapped = (data ?? []).map((permit) => ({
      id: permit.id,
      title: permit.title,
      description: permit.description ?? "Permiso del proveedor externo.",
      href: permit.href,
    }));

    setPermits(mapped);
  };

  const loadIngresos = async () => {
    const { data, error } = await supabase
      .from("rp_ingresos")
      .select("*")
      .order("fecha", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar la bitácora de ingreso.");
      return;
    }

    const mapped = (data ?? []).map((entry) => ({
      id: entry.id,
      residuo: entry.residuo,
      fecha: entry.fecha,
      cantidadKg: entry.cantidad_kg,
      codigoPeligrosidad: entry.codigo_peligrosidad,
      areaProceso: entry.area_proceso,
      responsable: entry.responsable,
    }));

    setIngresoEntries(mapped);
  };

  const loadSalidas = async () => {
    const { data, error } = await supabase
      .from("rp_salidas")
      .select("*")
      .order("fecha", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar la bitácora de salida.");
      return;
    }

    const mapped = (data ?? []).map((entry) => ({
      id: entry.id,
      residuo: entry.residuo,
      fecha: entry.fecha,
      cantidad: entry.cantidad,
      codigoPeligrosidad: entry.codigo_peligrosidad ?? "",
      numeroManifiesto: entry.numero_manifiesto ?? "",
      registroAmbiental: entry.registro_ambiental ?? "",
      responsable: entry.responsable ?? "",
      transportista: entry.transportista ?? "",
      autorizacionTransporte: entry.autorizacion_transporte ?? "",
      responsableTransporte: entry.responsable_transporte ?? "",
      destinoFinal: entry.destino_final ?? "",
      autorizacionDestinoFinal: entry.autorizacion_destino_final ?? "",
      responsableDestinoFinal: entry.responsable_destino_final ?? "",
    }));

    setSalidaEntries(mapped);
  };

  const mapFileToUploaded = (file: RpFile) => ({
    id: file.id,
    name: file.file_name,
    fileDate: file.file_date,
  });

  const loadFiles = async () => {
    const { data, error } = await supabase
      .from("rp_files")
      .select("*")
      .order("created_at", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar los archivos RP.");
      return;
    }

    const facturaItems = (data ?? [])
      .filter((file) => file.file_type === "factura")
      .map(mapFileToUploaded);
    const manifiestoItems = (data ?? [])
      .filter((file) => file.file_type === "manifiesto")
      .map(mapFileToUploaded);

    setFacturas(facturaItems);
    setManifiestos(manifiestoItems);
  };

  const loadRsuData = async () => {
    await Promise.all([loadRegistroDoc(), loadPermits(), loadIngresos(), loadSalidas(), loadFiles()]);
    setLoading(false);
  };

  useEffect(() => {
    if (user) {
      setLoading(true);
      loadRsuData();
    }
  }, [user]);

  const saveRegistro = async (title: string, href: string) => {
    if (!user) return;
    const payload = {
      title: title.trim(),
      description: registroDoc.description ?? registroBase.description,
      href: href.trim(),
      created_by: user.id,
    };

    if (!payload.title || !payload.href) {
      toast.error("Necesitamos título y enlace para guardar el registro.");
      return;
    }

    const { data, error } = registroId
      ? await supabase.from("rp_registro_docs").update(payload).eq("id", registroId).select().single()
      : await supabase.from("rp_registro_docs").insert(payload).select().single();

    if (error) {
      toast.error("No pudimos guardar el registro de manejo.");
      return;
    }

    setRegistroId(data.id);
    setRegistroDoc({
      id: data.id,
      title: data.title,
      description: data.description ?? registroBase.description,
      href: data.href,
    });
    setRegistroForm({ title: data.title, href: data.href });
    toast.success("Registro actualizado.");
  };

  const handleEditRegistro = async () => {
    if (!canManageDocs) return;
    const newTitle = window.prompt("Editar nombre del documento", registroDoc.title);
    if (!newTitle) return;
    const newHref = window.prompt("Editar enlace del documento (URL)", registroDoc.href);
    if (!newHref) return;
    await saveRegistro(newTitle, newHref);
  };

  const handleEditPermit = async (permitId: string) => {
    if (!canManageDocs) return;
    const permit = permits.find((item) => item.id === permitId);
    if (!permit) return;
    const newTitle = window.prompt("Editar nombre del permiso", permit.title);
    if (!newTitle) return;
    const newDescription = window.prompt(
      "Editar descripción del permiso",
      permit.description ?? "",
    );
    if (!newDescription) return;
    const newHref = window.prompt("Editar enlace del permiso (URL)", permit.href);
    if (!newHref) return;

    const payload = {
      title: newTitle.trim(),
      description: newDescription.trim() || null,
      href: newHref.trim(),
    };

    const { error, data } = await supabase
      .from("rp_permits")
      .update(payload)
      .eq("id", permitId)
      .select()
      .single();

    if (error || !data) {
      toast.error("No pudimos actualizar el permiso.");
      return;
    }
 
    setPermits((prev) =>
      prev.map((item) =>
        item.id === permitId
          ? {
              ...item,
              title: data.title,
              description: data.description ?? "Permiso del proveedor externo.",
              href: data.href,
            }
          : item,
      ),
    );
    toast.success("Permiso actualizado.");    
  };

  const handleDeleteRegistro = async () => {
    if (!canManageDocs) return;
    const confirmed = window.confirm("¿Deseas eliminar este documento?");
    if (!confirmed) return;
    if (registroId) {
      const { error } = await supabase.from("rp_registro_docs").delete().eq("id", registroId);
      if (error) {
        toast.error("No pudimos eliminar el registro.");
        return;
      }
    }
    setRegistroId(null);    
    setRegistroDoc((prev) => ({ ...prev, title: "Documento eliminado", href: "" }));
    setRegistroForm({ title: "", href: "" });
    toast.success("Registro eliminado.");
  };

  const handleDeletePermit = async (permitId: string) => {
    if (!canManageDocs) return;
    const confirmed = window.confirm("¿Deseas eliminar este permiso?");
    if (!confirmed) return;
    const { error } = await supabase.from("rp_permits").delete().eq("id", permitId);
    if (error) {
      toast.error("No pudimos eliminar el permiso.");
      return;
    }    
    setPermits((prev) => prev.filter((item) => item.id !== permitId));
    toast.success("Permiso eliminado.");
  };

  const handleRegistroSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManageDocs || !registroForm.title || !registroForm.href) return;
    await saveRegistro(registroForm.title, registroForm.href);
  };

  const handlePermitSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManageDocs || !permitForm.title || !permitForm.href) return;
    if (!user) return;

    const payload = {
      title: permitForm.title.trim(),
      description: permitForm.description.trim() || null,
      href: permitForm.href.trim(),
      created_by: user.id,
    };

    const { data, error } = await supabase.from("rp_permits").insert(payload).select().single();
    if (error || !data) {
      toast.error("No pudimos registrar el permiso.");
      return;
    }

    setPermits((prev) => [
      {
        id: data.id,
        title: data.title,
        description: data.description ?? "Permiso de proveedor registrado.",
        href: data.href,
      },
      ...prev,
    ]);
    setPermitForm({ title: "", description: "", href: "" });
    toast.success("Permiso registrado.");    
  };

  const handleIngresoSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (
      !ingresoForm.fecha ||
      !ingresoForm.cantidadKg ||
      !ingresoForm.codigoPeligrosidad ||
      !ingresoForm.areaProceso ||
      !ingresoForm.responsable
    ) {
      return;
    }
    if (!user) return;

    const payload = {
      residuo: ingresoForm.residuo,
      fecha: ingresoForm.fecha,
      cantidad_kg: ingresoForm.cantidadKg,
      codigo_peligrosidad: ingresoForm.codigoPeligrosidad,
      area_proceso: ingresoForm.areaProceso,
      responsable: ingresoForm.responsable,
      created_by: user.id,
    };

    const { data, error } = await supabase.from("rp_ingresos").insert(payload).select().single();
    if (error || !data) {
      toast.error("No pudimos registrar el ingreso.");
      return;
    }

    setIngresoEntries((prev) => [
      {
        id: data.id,
        residuo: data.residuo,
        fecha: data.fecha,
        cantidadKg: data.cantidad_kg,
        codigoPeligrosidad: data.codigo_peligrosidad,
        areaProceso: data.area_proceso,
        responsable: data.responsable,
      },
      ...prev,
    ]);
    setIngresoForm({
      residuo: residuosPeligrosos[0],
      fecha: "",
      cantidadKg: "",
      codigoPeligrosidad: "",
      areaProceso: "",
      responsable: "",
    });
    toast.success("Ingreso registrado.");
  };

  const handleEditIngreso = async (entryId: string) => {
    if (!canManageDocs) return;
    const entry = ingresoEntries.find((item) => item.id === entryId);
    if (!entry) return;
    const residuo = window.prompt("Editar residuo", entry.residuo);
    if (!residuo) return;
    const fecha = window.prompt("Editar fecha (YYYY-MM-DD)", entry.fecha);
    if (!fecha) return;
    const cantidadKg = window.prompt("Editar cantidad en kg", entry.cantidadKg);
    if (!cantidadKg) return;
    const codigoPeligrosidad = window.prompt(
      "Editar código de peligrosidad",
      entry.codigoPeligrosidad,
    );
    if (!codigoPeligrosidad) return;
    const areaProceso = window.prompt("Editar área o proceso", entry.areaProceso);
    if (!areaProceso) return;
    const responsable = window.prompt("Editar responsable", entry.responsable);
    if (!responsable) return;

    const payload = {
      residuo: residuo.trim(),
      fecha: fecha.trim(),
      cantidad_kg: cantidadKg.trim(),
      codigo_peligrosidad: codigoPeligrosidad.trim(),
      area_proceso: areaProceso.trim(),
      responsable: responsable.trim(),
    };

    const { data, error } = await supabase
      .from("rp_ingresos")
      .update(payload)
      .eq("id", entryId)
      .select()
      .single();

    if (error || !data) {
      toast.error("No pudimos actualizar el ingreso.");
      return;
    }

    setIngresoEntries((prev) =>
      prev.map((item) =>
        item.id === entryId
          ? {
              ...item,
              residuo: data.residuo,
              fecha: data.fecha,
              cantidadKg: data.cantidad_kg,
              codigoPeligrosidad: data.codigo_peligrosidad,
              areaProceso: data.area_proceso,
              responsable: data.responsable,
            }
          : item,
      ),
    );
    toast.success("Ingreso actualizado.");
  };

  const handleDeleteIngreso = async (entryId: string) => {
    if (!canManageDocs) return;
    const confirmed = window.confirm("¿Deseas eliminar este ingreso?");
    if (!confirmed) return;
    const { error } = await supabase.from("rp_ingresos").delete().eq("id", entryId);
    if (error) {
      toast.error("No pudimos eliminar el ingreso.");
      return;
    }
    setIngresoEntries((prev) => prev.filter((item) => item.id !== entryId));
    toast.success("Ingreso eliminado.");
  };

  const handleSalidaSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (
      !salidaForm.fecha ||
      !salidaForm.cantidad ||
      !salidaForm.codigoPeligrosidad ||
      !salidaForm.numeroManifiesto
    ) {
      return;
    }
    if (!user) return;

    const payload = {
      residuo: salidaForm.residuo,
      fecha: salidaForm.fecha,
      cantidad: salidaForm.cantidad,
      codigo_peligrosidad: salidaForm.codigoPeligrosidad,
      numero_manifiesto: salidaForm.numeroManifiesto,
      registro_ambiental: salidaForm.registroAmbiental || null,
      responsable: salidaForm.responsable || null,
      transportista: salidaForm.transportista || null,
      autorizacion_transporte: salidaForm.autorizacionTransporte || null,
      responsable_transporte: salidaForm.responsableTransporte || null,
      destino_final: salidaForm.destinoFinal || null,
      autorizacion_destino_final: salidaForm.autorizacionDestinoFinal || null,
      responsable_destino_final: salidaForm.responsableDestinoFinal || null,
      created_by: user.id,
    };

    const { data, error } = await supabase.from("rp_salidas").insert(payload).select().single();
    if (error || !data) {
      toast.error("No pudimos registrar la salida.");
      return;
    }

    setSalidaEntries((prev) => [
      {
        id: data.id,
        residuo: data.residuo,
        fecha: data.fecha,
        cantidad: data.cantidad,
        codigoPeligrosidad: data.codigo_peligrosidad ?? "",
        numeroManifiesto: data.numero_manifiesto ?? "",
        registroAmbiental: data.registro_ambiental ?? "",
        responsable: data.responsable ?? "",
        transportista: data.transportista ?? "",
        autorizacionTransporte: data.autorizacion_transporte ?? "",
        responsableTransporte: data.responsable_transporte ?? "",
        destinoFinal: data.destino_final ?? "",
        autorizacionDestinoFinal: data.autorizacion_destino_final ?? "",
        responsableDestinoFinal: data.responsable_destino_final ?? "",
      },
      ...prev,
    ]);
    setSalidaForm({
      residuo: residuosPeligrosos[0],
      fecha: "",
      cantidad: "",
      codigoPeligrosidad: "",
      numeroManifiesto: "",
      registroAmbiental: "",
      responsable: "",
      transportista: "",
      autorizacionTransporte: "",
      responsableTransporte: "",
      destinoFinal: "",
      autorizacionDestinoFinal: "",
      responsableDestinoFinal: "",
    });
    toast.success("Salida registrada.");    
  };

  const handleEditSalida = async (entryId: string) => {
    if (!canManageDocs) return;
    const entry = salidaEntries.find((item) => item.id === entryId);
    if (!entry) return;
    const residuo = window.prompt("Editar residuo", entry.residuo);
    if (!residuo) return;
    const fecha = window.prompt("Editar fecha (YYYY-MM-DD)", entry.fecha);
    if (!fecha) return;
    const cantidad = window.prompt("Editar cantidad", entry.cantidad);
    if (!cantidad) return;
    const codigoPeligrosidad = window.prompt(
      "Editar código de peligrosidad",
      entry.codigoPeligrosidad,
    );
    if (!codigoPeligrosidad) return;
    const numeroManifiesto = window.prompt(
      "Editar número de manifiesto",
      entry.numeroManifiesto,
    );
    if (!numeroManifiesto) return;
    const registroAmbiental = window.prompt(
      "Editar registro ambiental",
      entry.registroAmbiental,
    );
    if (!registroAmbiental) return;
    const responsable = window.prompt("Editar responsable", entry.responsable);
    if (!responsable) return;
    const transportista = window.prompt("Editar transportista", entry.transportista);
    if (!transportista) return;
    const autorizacionTransporte = window.prompt(
      "Editar autorización transporte",
      entry.autorizacionTransporte,
    );
    if (!autorizacionTransporte) return;
    const responsableTransporte = window.prompt(
      "Editar responsable transporte",
      entry.responsableTransporte,
    );
    if (!responsableTransporte) return;
    const destinoFinal = window.prompt("Editar destino final", entry.destinoFinal);
    if (!destinoFinal) return;
    const autorizacionDestinoFinal = window.prompt(
      "Editar autorización destino final",
      entry.autorizacionDestinoFinal,
    );
    if (!autorizacionDestinoFinal) return;
    const responsableDestinoFinal = window.prompt(
      "Editar responsable destino final",
      entry.responsableDestinoFinal,
    );
    if (!responsableDestinoFinal) return;

    const payload = {
      residuo: residuo.trim(),
      fecha: fecha.trim(),
      cantidad: cantidad.trim(),
      codigo_peligrosidad: codigoPeligrosidad.trim(),
      numero_manifiesto: numeroManifiesto.trim(),
      registro_ambiental: registroAmbiental.trim() || null,
      responsable: responsable.trim() || null,
      transportista: transportista.trim() || null,
      autorizacion_transporte: autorizacionTransporte.trim() || null,
      responsable_transporte: responsableTransporte.trim() || null,
      destino_final: destinoFinal.trim() || null,
      autorizacion_destino_final: autorizacionDestinoFinal.trim() || null,
      responsable_destino_final: responsableDestinoFinal.trim() || null,
    };

    const { data, error } = await supabase
      .from("rp_salidas")
      .update(payload)
      .eq("id", entryId)
      .select()
      .single();

    if (error || !data) {
      toast.error("No pudimos actualizar la salida.");
      return;
    }

    setSalidaEntries((prev) =>
      prev.map((item) =>
        item.id === entryId
          ? {
              ...item,
              residuo: data.residuo,
              fecha: data.fecha,
              cantidad: data.cantidad,
              codigoPeligrosidad: data.codigo_peligrosidad ?? "",
              numeroManifiesto: data.numero_manifiesto ?? "",
              registroAmbiental: data.registro_ambiental ?? "",
              responsable: data.responsable ?? "",
              transportista: data.transportista ?? "",
              autorizacionTransporte: data.autorizacion_transporte ?? "",
              responsableTransporte: data.responsable_transporte ?? "",
              destinoFinal: data.destino_final ?? "",
              autorizacionDestinoFinal: data.autorizacion_destino_final ?? "",
              responsableDestinoFinal: data.responsable_destino_final ?? "",
            }
          : item,
      ),
    );
    toast.success("Salida actualizada.");
  };

  const handleDeleteSalida = async (entryId: string) => {
    if (!canManageDocs) return;
    const confirmed = window.confirm("¿Deseas eliminar esta salida?");
    if (!confirmed) return;
    const { error } = await supabase.from("rp_salidas").delete().eq("id", entryId);
    if (error) {
      toast.error("No pudimos eliminar la salida.");
      return;
    }
    setSalidaEntries((prev) => prev.filter((item) => item.id !== entryId));
    toast.success("Salida eliminada.");
  };

  const handleFacturaSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!facturaFile) return;
    if (!user) return;
    const fileDate = new Date().toISOString().split("T")[0];
    const payload = {
      file_name: facturaFile.name,
      file_type: "factura",
      file_date: fileDate,
      created_by: user.id,
    };
    const { data, error } = await supabase.from("rp_files").insert(payload).select().single();
    if (error || !data) {
      toast.error("No pudimos registrar la factura.");
      return;
    }    
    setFacturas((prev) => [
      {
        id: data.id,
        name: data.file_name,
        fileDate: data.file_date,
      },
      ...prev,
    ]);
    setFacturaFile(null);
    toast.success("Factura registrada.");    
  };

  const handleManifiestoSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!manifiestoFile) return;
    if (!user) return;
    const fileDate = new Date().toISOString().split("T")[0];
    const payload = {
      file_name: manifiestoFile.name,
      file_type: "manifiesto",
      file_date: fileDate,
      created_by: user.id,
    };
    const { data, error } = await supabase.from("rp_files").insert(payload).select().single();
    if (error || !data) {
      toast.error("No pudimos registrar el manifiesto.");
      return;
    }    
    setManifiestos((prev) => [
      {
        id: data.id,
        name: data.file_name,
        fileDate: data.file_date,
      },
      ...prev,
    ]);
    setManifiestoFile(null);
    toast.success("Manifiesto registrado.");    
  };

  const handleEditFile = async (fileId: string, fileType: "factura" | "manifiesto") => {
    if (!canManageDocs) return;
    const files = fileType === "factura" ? facturas : manifiestos;
    const file = files.find((item) => item.id === fileId);
    if (!file) return;
    const newName = window.prompt("Editar nombre del archivo", file.name);
    if (!newName) return;
    const newDate = window.prompt("Editar fecha (YYYY-MM-DD)", file.fileDate);
    if (!newDate) return;

    const payload = {
      file_name: newName.trim(),
      file_date: newDate.trim(),
    };

    const { data, error } = await supabase
      .from("rp_files")
      .update(payload)
      .eq("id", fileId)
      .select()
      .single();

    if (error || !data) {
      toast.error("No pudimos actualizar el archivo.");
      return;
    }

    const updateFileState = (prev: UploadedFile[]) =>
      prev.map((item) =>
        item.id === fileId
          ? {
              ...item,
              name: data.file_name,
              fileDate: data.file_date,
            }
          : item,
      );

    if (fileType === "factura") {
      setFacturas(updateFileState);
    } else {
      setManifiestos(updateFileState);
    }

    toast.success("Archivo actualizado.");
  };

  const handleDeleteFile = async (fileId: string, fileType: "factura" | "manifiesto") => {
    if (!canManageDocs) return;
    const confirmed = window.confirm("¿Deseas eliminar este archivo?");
    if (!confirmed) return;
    const { error } = await supabase.from("rp_files").delete().eq("id", fileId);
    if (error) {
      toast.error("No pudimos eliminar el archivo.");
      return;
    }
    if (fileType === "factura") {
      setFacturas((prev) => prev.filter((item) => item.id !== fileId));
    } else {
      setManifiestos((prev) => prev.filter((item) => item.id !== fileId));
    }
    toast.success("Archivo eliminado.");
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-sky-500/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              Residuos peligrosos (RP)
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Gestión integral de RP
            </h1>
            <p className="max-w-3xl text-base text-gray-600 sm:text-lg">
              Administra los documentos clave, bitácoras de ingreso y salida, y los registros de
              facturas y manifiestos de residuos peligrosos.
            </p>
          </div>
        </section>

        <div className="grid gap-6 lg:grid-cols-2">
          <Card className="border-emerald-100">
            <CardHeader>
              <CardTitle>Registro y plan de manejo</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {canManageDocs && (
                <form
                  className="grid gap-3 rounded-2xl border border-emerald-100 bg-emerald-50 p-4 md:grid-cols-2"
                  onSubmit={handleRegistroSubmit}
                >
                  <div className="space-y-2">
                    <Label>Nombre del documento</Label>
                    <Input
                      value={registroForm.title}
                      onChange={(event) =>
                        setRegistroForm((prev) => ({ ...prev, title: event.target.value }))
                      }
                      placeholder="Ej. Registro y plan de manejo RP"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Enlace al archivo</Label>
                    <Input
                      value={registroForm.href}
                      onChange={(event) =>
                        setRegistroForm((prev) => ({ ...prev, href: event.target.value }))
                      }
                      placeholder="https://..."
                    />
                  </div>
                  <div className="md:col-span-2">
                    <Button type="submit">Guardar enlace</Button>
                  </div>
                </form>
              )}          
              <div className="flex flex-wrap items-center justify-between gap-3">
                <div>
                  <p className="text-sm font-semibold text-gray-900">{registroDoc.title}</p>
                  <p className="text-xs text-gray-500">{registroDoc.description}</p>
                </div>
                {canManageDocs && (
                  <div className="flex flex-wrap gap-2">
                    <Button variant="outline" size="sm" onClick={handleEditRegistro}>
                      Editar
                    </Button>
                    <Button variant="destructive" size="sm" onClick={handleDeleteRegistro}>
                      <Trash2 className="mr-2 h-4 w-4" />
                      Eliminar
                    </Button>
                  </div>
                )}
              </div>
              {registroDoc.href ? (
                <div className="rounded-2xl border border-slate-200 bg-slate-50 p-4">
                  <Link
                    className="text-sm font-semibold text-emerald-600 hover:text-emerald-700"
                    href={registroDoc.href}
                    target="_blank"
                  >
                    Abrir registro y plan de manejo RP
                  </Link>
                </div>
              ) : (
                <div className="rounded-2xl border border-dashed border-slate-200 bg-white p-6 text-sm text-gray-500">
                  Documento no disponible. Solicita al responsable cargar un nuevo PDF.
                </div>
              )}
            </CardContent>
          </Card>

          <Card className="border-emerald-100">
            <CardHeader>
              <CardTitle>Proveedores / Permisos</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {permits.length === 0 ? (
                <div className="rounded-2xl border border-dashed border-slate-200 bg-white p-6 text-sm text-gray-500">
                  No hay permisos registrados actualmente.
                </div>
              ) : (
                <div className="space-y-3">
                  {permits.map((permit) => (
                    <div
                      key={permit.id}
                      className="flex flex-wrap items-center justify-between gap-3 rounded-2xl border border-slate-200 bg-white p-4"
                    >
                      <div className="flex items-start gap-3">
                        <div className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-100 text-emerald-600">
                          <FileText className="h-5 w-5" />
                        </div>
                        <div>
                          <p className="text-sm font-semibold text-gray-900">{permit.title}</p>
                          <p className="text-xs text-gray-500">{permit.description}</p>
                          <Link
                            className="text-xs font-semibold text-emerald-600 hover:text-emerald-700"
                            href={permit.href}
                            target="_blank"
                          >
                            Ver PDF
                          </Link>
                        </div>
                      </div>
                      {canManageDocs && (
                        <div className="flex flex-wrap gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleEditPermit(permit.id)}
                          >
                            Editar
                          </Button>
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => handleDeletePermit(permit.id)}
                          >
                            Eliminar
                          </Button>
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              )}
              {canManageDocs && (
                <form
                  className="grid gap-3 rounded-2xl border border-emerald-100 bg-emerald-50 p-4 md:grid-cols-2"
                  onSubmit={handlePermitSubmit}
                >
                  <div className="space-y-2">
                    <Label>Nombre del permiso</Label>
                    <Input
                      value={permitForm.title}
                      onChange={(event) =>
                        setPermitForm((prev) => ({ ...prev, title: event.target.value }))
                      }
                      placeholder="Ej. Permiso del proveedor externo"
                    />
                  </div>
                  <div className="space-y-2 md:col-span-2">
                    <Label>Descripción</Label>
                    <Input
                      value={permitForm.description}
                      onChange={(event) =>
                        setPermitForm((prev) => ({ ...prev, description: event.target.value }))
                      }
                      placeholder="Descripción breve del permiso"
                    />
                  </div>
                  <div className="space-y-2 md:col-span-2">
                    <Label>Enlace al archivo</Label>
                    <Input
                      value={permitForm.href}
                      onChange={(event) =>
                        setPermitForm((prev) => ({ ...prev, href: event.target.value }))
                      }
                      placeholder="https://..."
                    />
                  </div>
                  <div className="md:col-span-2">
                    <Button type="submit">Agregar permiso</Button>
                  </div>
                </form>
              )}
            </CardContent>
          </Card>
        </div>

        <Card className="border-emerald-100">
          <CardHeader>
            <CardTitle>Bitácora de ingreso</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            {canManageDocs && (
              <form className="grid gap-4 sm:grid-cols-3" onSubmit={handleIngresoSubmit}>
                <div className="space-y-2">
                  <Label>Residuo</Label>
                  <select
                    className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                    value={ingresoForm.residuo}
                    onChange={(event) =>
                      setIngresoForm((prev) => ({ ...prev, residuo: event.target.value }))
                    }
                  >
                    {residuosPeligrosos.map((residuo) => (
                      <option key={residuo} value={residuo}>
                        {residuo}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label>Fecha</Label>
                  <Input
                    type="date"
                    value={ingresoForm.fecha}
                    onChange={(event) =>
                      setIngresoForm((prev) => ({ ...prev, fecha: event.target.value }))
                    }
                  />
                    {canManageDocs && <th className="px-4 py-3">Acciones</th>}                  
                </div>
                <div className="space-y-2">
                  <Label>Cantidad en kg.</Label>
                  <Input
                    placeholder="Ej. 35"
                    value={ingresoForm.cantidadKg}
                    onChange={(event) =>
                      setIngresoForm((prev) => ({ ...prev, cantidadKg: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Código de peligrosidad (C, R, E, T, I, B, M)</Label>
                  <Input
                    placeholder="Ej. C"
                    value={ingresoForm.codigoPeligrosidad}
                    onChange={(event) =>
                      setIngresoForm((prev) => ({
                        ...prev,
                        codigoPeligrosidad: event.target.value,
                      }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Área o proceso de generación</Label>
                  <Input
                    placeholder="Ej. Línea de producción"
                    value={ingresoForm.areaProceso}
                    onChange={(event) =>
                      setIngresoForm((prev) => ({ ...prev, areaProceso: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Responsable</Label>
                  <Input
                    placeholder="Ej. Juan Pérez"
                    value={ingresoForm.responsable}
                    onChange={(event) =>
                      setIngresoForm((prev) => ({ ...prev, responsable: event.target.value }))
                    }
                  />
                </div>
                <div className="sm:col-span-3">
                  <Button type="submit">Registrar ingreso</Button>
                </div>
              </form>
            )}

            <div className="max-h-96 overflow-x-auto overflow-y-auto rounded-2xl border border-slate-200">
              <table className="min-w-full divide-y divide-slate-200 text-sm">
                <thead className="bg-slate-50 text-left text-xs font-semibold uppercase tracking-wide text-slate-500">
                  <tr>
                    <th className="px-4 py-3">Residuo</th>
                    <th className="px-4 py-3">Fecha</th>
                    <th className="px-4 py-3">Cantidad en kg.</th>
                    <th className="px-4 py-3">Código de peligrosidad</th>
                    <th className="px-4 py-3">Área o proceso de generación</th>
                    <th className="px-4 py-3">Responsable</th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-slate-100 bg-white">
                  {ingresoEntries.length === 0 ? (
                    <tr>
                      <td
                        className="px-4 py-4 text-sm text-slate-500"
                        colSpan={canManageDocs ? 7 : 6}
                      >
                        Aún no hay registros de ingreso.
                      </td>
                    </tr>
                  ) : (
                    ingresoEntries.map((entry) => (
                      <tr key={entry.id}>
                        <td className="px-4 py-3 font-semibold text-slate-700">
                          {entry.residuo}
                        </td>
                        <td className="px-4 py-3 text-slate-600">{entry.fecha}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.cantidadKg}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.codigoPeligrosidad}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.areaProceso}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.responsable}</td>
                        {canManageDocs && (
                          <td className="px-4 py-3">
                            <div className="flex gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => handleEditIngreso(entry.id)}
                              >
                                Editar
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => handleDeleteIngreso(entry.id)}
                              >
                                Eliminar
                              </Button>
                            </div>
                          </td>
                        )}                        
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>

        <Card className="border-emerald-100">
          <CardHeader>
            <CardTitle>Bitácora de salida</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            {canManageDocs && (
              <form className="grid gap-4 lg:grid-cols-3" onSubmit={handleSalidaSubmit}>
                <div className="space-y-2">
                  <Label>Residuo</Label>
                  <select
                    className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm"
                    value={salidaForm.residuo}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, residuo: event.target.value }))
                    }
                  >
                    {residuosPeligrosos.map((residuo) => (
                      <option key={residuo} value={residuo}>
                        {residuo}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label>Fecha</Label>
                  <Input
                    type="date"
                    value={salidaForm.fecha}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, fecha: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Cantidad</Label>
                  <Input
                    placeholder="Ej. 120 kg"
                    value={salidaForm.cantidad}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, cantidad: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Código de peligrosidad (C, R, E, T, I, B, M)</Label>
                  <Input
                    placeholder="Ej. C"
                    value={salidaForm.codigoPeligrosidad}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({
                        ...prev,
                        codigoPeligrosidad: event.target.value,
                      }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Número de manifiesto</Label>
                  <Input
                    placeholder="Ej. MAN-2024-045"
                    value={salidaForm.numeroManifiesto}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({
                        ...prev,
                        numeroManifiesto: event.target.value,
                      }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Número de registro ambiental</Label>
                  <Input
                    value={salidaForm.registroAmbiental}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, registroAmbiental: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Nombre responsable</Label>
                  <Input
                    value={salidaForm.responsable}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, responsable: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Nombre del transportista</Label>
                  <Input
                    value={salidaForm.transportista}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, transportista: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Número de autorización transporte</Label>
                  <Input
                    value={salidaForm.autorizacionTransporte}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, autorizacionTransporte: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Nombre responsable transporte</Label>
                  <Input
                    value={salidaForm.responsableTransporte}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, responsableTransporte: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Nombre de destino final</Label>
                  <Input
                    value={salidaForm.destinoFinal}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, destinoFinal: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Número de autorización destino final</Label>
                  <Input
                    value={salidaForm.autorizacionDestinoFinal}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, autorizacionDestinoFinal: event.target.value }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Nombre responsable destino final</Label>
                  <Input
                    value={salidaForm.responsableDestinoFinal}
                    onChange={(event) =>
                      setSalidaForm((prev) => ({ ...prev, responsableDestinoFinal: event.target.value }))
                    }
                  />
                </div>
                <div className="lg:col-span-3">
                  <Button type="submit">Registrar salida</Button>
                </div>
              </form>
            )}

            <div className="max-h-96 overflow-x-auto overflow-y-auto rounded-2xl border border-slate-200">
              <table className="min-w-full divide-y divide-slate-200 text-sm">
                <thead className="bg-slate-50 text-left text-xs font-semibold uppercase tracking-wide text-slate-500">
                  <tr>
                    <th className="px-4 py-3">Residuo</th>
                    <th className="px-4 py-3">Fecha</th>
                    <th className="px-4 py-3">Cantidad</th>
                    <th className="px-4 py-3">Código de peligrosidad</th>
                    <th className="px-4 py-3">Número de manifiesto</th>
                    <th className="px-4 py-3">Registro ambiental</th>
                    <th className="px-4 py-3">Responsable</th>
                    <th className="px-4 py-3">Transportista</th>
                    <th className="px-4 py-3">Autorización transporte</th>
                    <th className="px-4 py-3">Responsable transporte</th>
                    <th className="px-4 py-3">Destino final</th>
                    <th className="px-4 py-3">Autorización destino final</th>
                    <th className="px-4 py-3">Responsable destino final</th>
                    {canManageDocs && <th className="px-4 py-3">Acciones</th>}
                  </tr>
                </thead>
                <tbody className="divide-y divide-slate-100 bg-white">
                  {salidaEntries.length === 0 ? (
                    <tr>
                      <td
                        className="px-4 py-4 text-sm text-slate-500"
                        colSpan={canManageDocs ? 14 : 13}
                      >
                        Aún no hay registros de salida.
                      </td>
                    </tr>
                  ) : (
                    salidaEntries.map((entry) => (
                      <tr key={entry.id}>
                        <td className="px-4 py-3 font-semibold text-slate-700">
                          {entry.residuo}
                        </td>
                        <td className="px-4 py-3 text-slate-600">{entry.fecha}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.cantidad}</td>
                        <td className="px-4 py-3 text-slate-600">
                          {entry.codigoPeligrosidad}
                        </td>
                        <td className="px-4 py-3 text-slate-600">{entry.numeroManifiesto}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.registroAmbiental}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.responsable}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.transportista}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.autorizacionTransporte}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.responsableTransporte}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.destinoFinal}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.autorizacionDestinoFinal}</td>
                        <td className="px-4 py-3 text-slate-600">{entry.responsableDestinoFinal}</td>
                        {canManageDocs && (
                          <td className="px-4 py-3">
                            <div className="flex gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => handleEditSalida(entry.id)}
                              >
                                Editar
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => handleDeleteSalida(entry.id)}
                              >
                                Eliminar
                              </Button>
                            </div>
                          </td>
                        )}
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>

        <div className="grid gap-6 lg:grid-cols-2">
          <Card className="border-emerald-100">
            <CardHeader>
              <CardTitle>Facturas</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="rounded-2xl border border-emerald-100 bg-emerald-50 px-4 py-3 text-sm text-emerald-700">
                Los PDF se almacenarán en SharePoint. Consulta el registro de ingreso en:{" "}
                <Link
                  href="https://meiwamoldcom-my.sharepoint.com/:f:/g/personal/s-sasaki_meiwamold_com/IgDU7yCVhENlRYrsQgz1O2b_AWsvDezHj8MshpcPq3id34g?e=5Slu2u"
                  target="_blank"
                  className="font-semibold text-emerald-700 underline"
                >
                  carpeta de Facturas
                </Link>
                .
              </div>
              {canManageDocs && (
                <form className="space-y-3" onSubmit={handleFacturaSubmit}>
                  <div className="space-y-2">
                    <Label>Cargar factura (PDF)</Label>
                    <Input
                      type="file"
                      accept="application/pdf"
                      onChange={(event) => setFacturaFile(event.target.files?.[0] ?? null)}
                    />
                  </div>
                  <Button type="submit" disabled={!facturaFile}>
                    <FileUp className="mr-2 h-4 w-4" />
                    Registrar factura
                  </Button>
                </form>
              )}
              <div className="space-y-2">
                {facturas.length === 0 ? (
                  <p className="text-sm text-slate-500">No hay facturas registradas.</p>
                ) : (
                  facturas.map((file) => (
                    <div
                      key={file.id}
                      className="flex items-center justify-between rounded-xl border border-slate-200 bg-white px-4 py-3 text-sm"
                    >
                      <div>
                        <p className="font-semibold text-slate-700">{file.name}</p>
                        <p className="text-xs text-slate-500">
                          {new Date(file.fileDate).toLocaleDateString("es-MX")}
                        </p>
                      </div>
                      {canManageDocs && (
                        <div className="flex gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleEditFile(file.id, "factura")}
                          >
                            Editar
                          </Button>
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => handleDeleteFile(file.id, "factura")}
                          >
                            Eliminar
                          </Button>
                        </div>
                      )}
                    </div>
                  ))
                )}
              </div>
            </CardContent>
          </Card>

          <Card className="border-emerald-100">
            <CardHeader>
              <CardTitle>Manifiestos</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="rounded-2xl border border-emerald-100 bg-emerald-50 px-4 py-3 text-sm text-emerald-700">
                Los PDF se almacenarán en SharePoint. Consulta el registro de ingreso en:{" "}
                <Link
                  href="https://meiwamoldcom-my.sharepoint.com/:f:/g/personal/s-sasaki_meiwamold_com/IgDIwd_2qayFQYIRB6AbxI-iAXWjlUG6CuGnt-O6bImg3q0?e=iudP6j"
                  target="_blank"
                  className="font-semibold text-emerald-700 underline"
                >
                  carpeta de Manifiestos
                </Link>
                .
              </div>
              {canManageDocs && (
                <form className="space-y-3" onSubmit={handleManifiestoSubmit}>
                  <div className="space-y-2">
                    <Label>Cargar manifiesto (PDF)</Label>
                    <Input
                      type="file"
                      accept="application/pdf"
                      onChange={(event) => setManifiestoFile(event.target.files?.[0] ?? null)}
                    />
                  </div>
                  <Button type="submit" disabled={!manifiestoFile}>
                    <FileUp className="mr-2 h-4 w-4" />
                    Registrar manifiesto
                  </Button>
                </form>
              )}
              <div className="space-y-2">
                {manifiestos.length === 0 ? (
                  <p className="text-sm text-slate-500">No hay manifiestos registrados.</p>
                ) : (
                  manifiestos.map((file) => (
                    <div
                      key={file.id}
                      className="flex items-center justify-between rounded-xl border border-slate-200 bg-white px-4 py-3 text-sm"
                    >
                      <div>
                        <p className="font-semibold text-slate-700">{file.name}</p>
                        <p className="text-xs text-slate-500">
                          {new Date(file.fileDate).toLocaleDateString("es-MX")}
                        </p>
                      </div>
                      {canManageDocs && (
                        <div className="flex gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleEditFile(file.id, "manifiesto")}
                          >
                            Editar
                          </Button>
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => handleDeleteFile(file.id, "manifiesto")}
                          >
                            Eliminar
                          </Button>
                        </div>
                      )}
                    </div>
                  ))
                )}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </SGIDashboardLayout>
  );
}