"use client";

import { type FormEvent, useEffect, useMemo, useState } from "react";
import { type User } from "@supabase/supabase-js";
import { toast } from "sonner";

import {
  Droplet,
  FileText,
  Factory,
  Filter,
  LineChart,
  PackageSearch,
  Pencil,
  Plus,
  Trash2,
  Waves,
} from "lucide-react";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { supabase } from "@/lib/supabase";

type WaterRecord = {
  id: string;
  type: "Red" | "RO";
  date: string;
  month: string;
  year: number;
  quantityM3: number;
  quantityLiters: number;
  unitCost: number;
  totalCost: number;
  provider: string;
  invoiceUrl?: string;
};

type WaterFormState = {
  type: "Red" | "RO";
  date: string;
  quantityM3: string;
  unitCost: string;
  provider: string;
  invoiceUrl: string;
};

const englishColumns = {
  type: "type",
  date: "date",
  month: "month",
  year: "year",
  quantityM3: "quantity_m3",
  quantityLiters: "quantity_liters",
  unitCost: "unit_cost",
  totalCost: "total_cost",
  provider: "provider",
  invoiceUrl: "invoice_url",
  createdBy: "created_by",
} as const;

const spanishColumns = {
  type: "tipo",
  date: "fecha",
  month: "mes",
  year: "anio",
  quantityM3: "cantidad_m3",
  quantityLiters: "cantidad_litros",
  unitCost: "costo_unitario",
  totalCost: "costo_total",
  provider: "proveedor",
  invoiceUrl: "factura_url",
  createdBy: "created_by",
} as const;

type WaterColumnConfig = typeof englishColumns | typeof spanishColumns;

const numberFormatter = new Intl.NumberFormat("es-MX", {
  minimumFractionDigits: 0,
  maximumFractionDigits: 2,
});

const currencyFormatter = new Intl.NumberFormat("es-MX", {
  style: "currency",
  currency: "MXN",
  minimumFractionDigits: 2,
  maximumFractionDigits: 2,
});

const formatDate = (value: string) =>
  new Date(`${value}T00:00:00`).toLocaleDateString("es-MX", {
    day: "2-digit",
    month: "short",
    year: "numeric",
  });

const buildStats = (records: WaterRecord[], type: "Red" | "RO") => {
  const entries = records.filter((record) => record.type === type);
  const totalM3 = entries.reduce((acc, record) => acc + record.quantityM3, 0);
  const totalLiters = entries.reduce((acc, record) => acc + record.quantityLiters, 0);
  const totalCost = entries.reduce((acc, record) => acc + record.totalCost, 0);
  const averageUnitCost = entries.length
    ? entries.reduce((acc, record) => acc + record.unitCost, 0) / entries.length
    : 0;
  const topProvider = entries.reduce<Record<string, number>>((acc, record) => {
    acc[record.provider] = (acc[record.provider] ?? 0) + 1;
    return acc;
  }, {});
  const provider = Object.keys(topProvider).sort(
    (first, second) => (topProvider[second] ?? 0) - (topProvider[first] ?? 0),
  )[0];

  return {
    entries,
    totalM3,
    totalLiters,
    totalCost,
    averageUnitCost,
    provider,
  };
};

const toSpanishMonth = (dateValue: string) =>
  new Date(`${dateValue}T00:00:00`).toLocaleDateString("es-MX", {
    month: "long",
  });

const detectColumnConfig = (
  records: Record<string, unknown>[],
  fallback: WaterColumnConfig,
): WaterColumnConfig => {
  const sample = records[0];
  if (!sample) return fallback;
  if ("fecha" in sample || "tipo" in sample || "cantidad_m3" in sample) {
    return spanishColumns;
  }
  if ("date" in sample || "type" in sample || "quantity_m3" in sample) {
    return englishColumns;
  }
  return fallback;
};

const sortByDateDesc = (records: WaterRecord[]) =>
  [...records].sort((first, second) => {
    const firstTime = new Date(`${first.date}T00:00:00`).getTime();
    const secondTime = new Date(`${second.date}T00:00:00`).getTime();
    if (Number.isNaN(firstTime) && Number.isNaN(secondTime)) return 0;
    if (Number.isNaN(firstTime)) return 1;
    if (Number.isNaN(secondTime)) return -1;
    return secondTime - firstTime;
  });

  const StatCard = ({
  title,
  value,
  helper,
  icon: Icon,
  accent,
}: {
  title: string;
  value: string;
  helper: string;
  icon: typeof Droplet;
  accent: string;
}) => (
  <Card className="border-gray-200">
    <CardContent className="flex items-start justify-between gap-4 p-6">
      <div>
        <p className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
          {title}
        </p>
        <p className="mt-3 text-2xl font-semibold text-gray-900">{value}</p>
        <p className="mt-2 text-sm text-gray-500">{helper}</p>
      </div>
      <div className={`rounded-2xl p-3 ${accent}`}>
        <Icon className="h-5 w-5 text-white" />
      </div>
    </CardContent>
  </Card>
);

const spanishMonths = [
  "Enero",
  "Febrero",
  "Marzo",
  "Abril",
  "Mayo",
  "Junio",
  "Julio",
  "Agosto",
  "Septiembre",
  "Octubre",
  "Noviembre",
  "Diciembre",
];

const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "admin.mexico@meiwa.com.mx", "admin2.mexico@meiwa.com.mx"]);

export default function WaterPage() {
  const [waterRecords, setWaterRecords] = useState<WaterRecord[]>([]);
  const [formState, setFormState] = useState<WaterFormState>({
    type: "Red",
    date: "",
    quantityM3: "",
    unitCost: "",
    provider: "",
    invoiceUrl: "",
  });
  const [user, setUser] = useState<User | null>(null);
  const [columnConfig, setColumnConfig] = useState<WaterColumnConfig>(englishColumns);
  const [selectedYear, setSelectedYear] = useState<string>("Todo");
  const [selectedMonth, setSelectedMonth] = useState<string>("Todo");
  const canManageWater = allowedManagers.has(user?.email ?? "");

  const availableYears = useMemo(
    () =>
      Array.from(new Set(waterRecords.map((record) => record.year)))
        .filter((year) => Number.isFinite(year))
        .sort((a, b) => b - a),
    [waterRecords],
  );

  const availableMonths = useMemo(() => {
    const monthSet = new Set(waterRecords.map((record) => record.month).filter(Boolean));
    return spanishMonths.filter((month) => monthSet.has(month));
  }, [waterRecords]);

  const filteredRecords = useMemo(
    () =>
      waterRecords.filter((record) => {
        const matchesYear =
          selectedYear === "Todo" || record.year === Number(selectedYear);
        const matchesMonth =
          selectedMonth === "Todo" || record.month === selectedMonth;
        return matchesYear && matchesMonth;
      }),
    [selectedMonth, selectedYear, waterRecords],
  );

  const redStats = useMemo(
    () => buildStats(filteredRecords, "Red"),
    [filteredRecords],
  );
  const roStats = useMemo(() => buildStats(filteredRecords, "RO"), [filteredRecords]);

  const mapRecord = (record: Record<string, unknown>, config: WaterColumnConfig): WaterRecord => {
    const dateValue = record[config.date] as string;
    const quantityM3 = (record[config.quantityM3] as number) ?? 0;
    const unitCost = (record[config.unitCost] as number) ?? 0;
    return {
      id: record.id as string,
      type: (record[config.type] as string) === "RO" ? "RO" : "Red",
      date: dateValue,
      month: (record[config.month] as string) ?? toSpanishMonth(dateValue),
      year: (record[config.year] as number) ?? new Date(`${dateValue}T00:00:00`).getFullYear(),
      quantityM3,
      quantityLiters:
        (record[config.quantityLiters] as number) ?? quantityM3 * 1000,
      unitCost,
      totalCost:
        (record[config.totalCost] as number) ?? quantityM3 * unitCost,
      provider: (record[config.provider] as string) ?? "",
      invoiceUrl: (record[config.invoiceUrl] as string) ?? undefined,
    };
  };

  const buildPayload = (
    config: WaterColumnConfig,
    parsedDate: Date,
    options: { includeDerived?: boolean; includeCreatedBy?: boolean } = {},
  ) => {
    const payload: Record<string, unknown> = {
      [config.type]: formState.type,
      [config.date]: formState.date,
      [config.quantityM3]: Number(formState.quantityM3),
      [config.unitCost]: Number(formState.unitCost),
      [config.provider]: formState.provider.trim(),
      [config.invoiceUrl]: formState.invoiceUrl.trim() || null,
    };

    if (options.includeDerived ?? true) {
      payload[config.month] = toSpanishMonth(formState.date);
      payload[config.year] = parsedDate.getFullYear();
      payload[config.quantityLiters] = Number(formState.quantityM3) * 1000;
      payload[config.totalCost] =
        Number(formState.quantityM3) * Number(formState.unitCost);
    }

    if (options.includeCreatedBy ?? true) {
      payload[config.createdBy] = user?.id;
    }

    return payload;
  };

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      setUser(data.session?.user ?? null);
    };

    const loadRecords = async () => {
      const fetchRecords = async () => supabase.from("water_records").select("*");

      const { data, error } = await fetchRecords();

      if (error) {
        toast.error("No pudimos cargar los registros de agua.");
        return;
      }

      const configToUse = detectColumnConfig(
        (data ?? []) as Record<string, unknown>[],
        columnConfig,
      );
      if (configToUse !== columnConfig) {
        setColumnConfig(configToUse);
      }
      const mapped = (data ?? []).map((record) =>
        mapRecord(record as Record<string, unknown>, configToUse),
      );

      setWaterRecords(sortByDateDesc(mapped));
    };

    loadSession();
    loadRecords();

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
    });

    return () => subscription.unsubscribe();
  }, []);

  const handleAddRecord = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManageWater) return;
    const quantityM3 = Number(formState.quantityM3);
    const unitCost = Number(formState.unitCost);
    if (!Number.isFinite(quantityM3) || !Number.isFinite(unitCost)) return;
    if (!formState.date.trim() || !formState.provider.trim()) return;

    const parsedDate = new Date(`${formState.date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) return;

    if (!user) {
      toast.error("Necesitamos una sesión activa para guardar el registro.");
      return;
    }

    const fullPayload = buildPayload(columnConfig, parsedDate, {
      includeDerived: true,
      includeCreatedBy: true,
    });
    const basePayload = buildPayload(columnConfig, parsedDate, {
      includeDerived: false,
      includeCreatedBy: false,
    });

    const insertRecord = (payload: Record<string, unknown>) =>
      supabase.from("water_records").insert(payload).select().single();

    let { data, error } = await insertRecord(fullPayload);
    let configToUse = columnConfig;

    if (error) {
      const baseAttempt = await insertRecord(basePayload);
      if (!baseAttempt.error) {
        data = baseAttempt.data;
        error = null;
      } else if (columnConfig === englishColumns) {
        const spanishFullPayload = buildPayload(spanishColumns, parsedDate, {
          includeDerived: true,
          includeCreatedBy: true,
        });
        const spanishBasePayload = buildPayload(spanishColumns, parsedDate, {
          includeDerived: false,
          includeCreatedBy: false,
        });

        const spanishFull = await insertRecord(spanishFullPayload);
        if (!spanishFull.error) {
          data = spanishFull.data;
          error = null;
          configToUse = spanishColumns;
        } else {
          const spanishBase = await insertRecord(spanishBasePayload);
          data = spanishBase.data;
          error = spanishBase.error;
          if (!spanishBase.error) {
            configToUse = spanishColumns;
          }
        }
      }
    }

    if (error || !data) {
      toast.error("No pudimos guardar el registro.");
      return;
    }

    if (configToUse !== columnConfig) {
      setColumnConfig(configToUse);
    }
    setWaterRecords((prev) => [
      mapRecord(data as Record<string, unknown>, configToUse),
      ...prev,
    ]);
    setFormState({
      type: "Red",
      date: "",
      quantityM3: "",
      unitCost: "",
      provider: "",
      invoiceUrl: "",
    });
    toast.success("Registro guardado.");
  };

  const handleEditRecord = async (recordId: string) => {
    if (!canManageWater) return;
    const record = waterRecords.find((item) => item.id === recordId);
    if (!record) return;
    const typeInput = window.prompt("Tipo de registro (Red o RO)", record.type);
    if (!typeInput) return;
    const type = typeInput.trim().toLowerCase() === "ro" ? "RO" : "Red";
    const date = window.prompt("Fecha de registro (YYYY-MM-DD)", record.date);
    if (!date) return;
    const quantityM3Input = window.prompt("Cantidad en m³", String(record.quantityM3));
    if (!quantityM3Input) return;
    const unitCostInput = window.prompt("Costo unitario (MXN/m³)", String(record.unitCost));
    if (!unitCostInput) return;
    const provider = window.prompt("Proveedor", record.provider);
    if (!provider) return;
    const invoiceUrl =
      window.prompt("URL de factura (opcional)", record.invoiceUrl ?? "") ?? "";

    const quantityM3 = Number(quantityM3Input);
    const unitCost = Number(unitCostInput);
    if (!Number.isFinite(quantityM3) || !Number.isFinite(unitCost)) return;
    const parsedDate = new Date(`${date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) return;

    const fullPayload = {
      [columnConfig.type]: type,
      [columnConfig.date]: date,
      [columnConfig.month]: toSpanishMonth(date),
      [columnConfig.year]: parsedDate.getFullYear(),
      [columnConfig.quantityM3]: quantityM3,
      [columnConfig.quantityLiters]: quantityM3 * 1000,
      [columnConfig.unitCost]: unitCost,
      [columnConfig.totalCost]: quantityM3 * unitCost,
      [columnConfig.provider]: provider.trim(),
      [columnConfig.invoiceUrl]: invoiceUrl.trim() || null,
    };

    const updateRecord = (payload: Record<string, unknown>) =>
      supabase.from("water_records").update(payload).eq("id", recordId).select().single();

    let { data, error } = await updateRecord(fullPayload);
    let configToUse = columnConfig;

    const basePayload = {
      [columnConfig.type]: type,
      [columnConfig.date]: date,
      [columnConfig.quantityM3]: quantityM3,
      [columnConfig.unitCost]: unitCost,
      [columnConfig.provider]: provider.trim(),
      [columnConfig.invoiceUrl]: invoiceUrl.trim() || null,
    };

    if (error) {
      const baseAttempt = await updateRecord(basePayload);
      if (!baseAttempt.error) {
        data = baseAttempt.data;
        error = null;
      } else if (columnConfig === englishColumns) {
        const spanishFullPayload = {
          [spanishColumns.type]: type,
          [spanishColumns.date]: date,
          [spanishColumns.month]: toSpanishMonth(date),
          [spanishColumns.year]: parsedDate.getFullYear(),
          [spanishColumns.quantityM3]: quantityM3,
          [spanishColumns.quantityLiters]: quantityM3 * 1000,
          [spanishColumns.unitCost]: unitCost,
          [spanishColumns.totalCost]: quantityM3 * unitCost,
          [spanishColumns.provider]: provider.trim(),
          [spanishColumns.invoiceUrl]: invoiceUrl.trim() || null,
        };
        const spanishBasePayload = {
          [spanishColumns.type]: type,
          [spanishColumns.date]: date,
          [spanishColumns.quantityM3]: quantityM3,
          [spanishColumns.unitCost]: unitCost,
          [spanishColumns.provider]: provider.trim(),
          [spanishColumns.invoiceUrl]: invoiceUrl.trim() || null,
        };

        const spanishFull = await updateRecord(spanishFullPayload);
        if (!spanishFull.error) {
          data = spanishFull.data;
          error = null;
          configToUse = spanishColumns;
        } else {
          const spanishBase = await updateRecord(spanishBasePayload);
          data = spanishBase.data;
          error = spanishBase.error;
          if (!spanishBase.error) {
            configToUse = spanishColumns;
          }
        }
      }
    }

       
    if (error || !data) {
      toast.error("No pudimos actualizar el registro.");
      return;
    }

    if (configToUse !== columnConfig) {
      setColumnConfig(configToUse);
    }

    setWaterRecords((prev) =>
      prev.map((item) =>
        item.id === recordId
          ? mapRecord(data as Record<string, unknown>, configToUse)
          : item,
      ),
    );
    toast.success("Registro actualizado.");
  };

  const handleDeleteRecord = async (recordId: string) => {
    if (!canManageWater) return;
    const confirmed = window.confirm("¿Deseas eliminar este registro?");
    if (!confirmed) return;

    const { error } = await supabase.from("water_records").delete().eq("id", recordId);
    if (error) {
      toast.error("No pudimos eliminar el registro.");
      return;
    }

    setWaterRecords((prev) => prev.filter((item) => item.id !== recordId));
    toast.success("Registro eliminado.");
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-sky-50">
        <section className="relative overflow-hidden rounded-3xl border border-sky-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-sky-500/10 to-emerald-500/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-sky-600">
              Operación · Agua
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Registro y consumo de agua
            </h1>
            <p className="max-w-4xl text-base text-gray-600 xs:text-lg">
              Controla el consumo de agua de red y recolecciones de agua RO con indicadores por
              periodo, costos y proveedores. Los reportes se separan por tipo para facilitar el
              seguimiento ambiental.
            </p>
            <div className="mt-6 grid gap-4 rounded-2xl border border-slate-200 bg-white/80 p-4 sm:grid-cols-3">
              <div className="space-y-2">
                <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                  Filtro de año
                </Label>
                <select
                  className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                  value={selectedYear}
                  onChange={(event) => setSelectedYear(event.target.value)}
                >
                  <option value="Todo">Todo</option>
                  {availableYears.map((year) => (
                    <option key={year} value={String(year)}>
                      {year}
                    </option>
                  ))}
                </select>
              </div>
              <div className="space-y-2">
                <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                  Filtro de mes
                </Label>
                <select
                  className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                  value={selectedMonth}
                  onChange={(event) => setSelectedMonth(event.target.value)}
                >
                  <option value="Todo">Todo</option>
                  {availableMonths.map((month) => (
                    <option key={month} value={month}>
                      {month}
                    </option>
                  ))}
                </select>
              </div>
              <div className="rounded-2xl border border-slate-100 bg-slate-50 px-4 py-5">
                <p className="text-xs font-semibold uppercase text-slate-600">
                  Registros filtrados
                </p>
                <p className="mt-2 text-2xl font-semibold text-slate-900">
                  {filteredRecords.length} registros
                </p>
                <p className="mt-1 text-xs text-slate-500">
                  Actualiza los filtros para ver otras métricas.
                </p>
              </div>
            </div>
            <div className="mt-6 grid gap-4 sm:grid-cols-3">
              <div className="rounded-2xl border border-sky-100 bg-sky-50 px-4 py-5">
                <p className="text-xs font-semibold uppercase text-sky-600">Registros totales</p>
                <p className="mt-2 text-2xl font-semibold text-sky-900">
                  {filteredRecords.length} registros
                </p>
              </div>
              <div className="rounded-2xl border border-emerald-100 bg-emerald-50 px-4 py-5">
                <p className="text-xs font-semibold uppercase text-emerald-600">Agua de red</p>
                <p className="mt-2 text-2xl font-semibold text-emerald-900">
                  {numberFormatter.format(redStats.totalM3)} m³
                </p>
              </div>
              <div className="rounded-2xl border border-cyan-100 bg-cyan-50 px-4 py-5">
                <p className="text-xs font-semibold uppercase text-cyan-600">Agua RO</p>
                <p className="mt-2 text-2xl font-semibold text-cyan-900">
                  {numberFormatter.format(roStats.totalM3)} m³
                </p>
              </div>
            </div>
          </div>
        </section>

        <section className="grid gap-6 lg:grid-cols-2">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between gap-3">
              <div>
                <CardTitle className="text-xl text-gray-900">Estadísticas Agua de Red</CardTitle>
                <p className="text-sm text-gray-500">
                  Consumo proveniente de la red municipal.
                </p>
              </div>
              <div className="rounded-2xl bg-emerald-100 p-3 text-emerald-600">
                <Waves className="h-5 w-5" />
              </div>
            </CardHeader>
            <CardContent className="grid gap-4 sm:grid-cols-2">
              <StatCard
                title="Consumo total"
                value={`${numberFormatter.format(redStats.totalM3)} m³`}
                helper={`${numberFormatter.format(redStats.totalLiters)} litros`}
                icon={Droplet}
                accent="bg-emerald-500"
              />
              <StatCard
                title="Costo total"
                value={currencyFormatter.format(redStats.totalCost)}
                helper="Acumulado del periodo"
                icon={LineChart}
                accent="bg-emerald-400"
              />
              <StatCard
                title="Costo unitario promedio"
                value={`${currencyFormatter.format(redStats.averageUnitCost)}/m³`}
                helper="Promedio de facturación"
                icon={Filter}
                accent="bg-emerald-500"
              />
              <StatCard
                title="Proveedor principal"
                value={redStats.provider ?? "Sin proveedor"}
                helper={`${redStats.entries.length} registros`}
                icon={PackageSearch}
                accent="bg-emerald-400"
              />
            </CardContent>
          </Card>

          <Card className="border-sky-100">
            <CardHeader className="flex flex-row items-center justify-between gap-3">
              <div>
                <CardTitle className="text-xl text-gray-900">Estadísticas Agua RO</CardTitle>
                <p className="text-sm text-gray-500">
                  Recuperación y tratamiento por ósmosis inversa.
                </p>
              </div>
              <div className="rounded-2xl bg-sky-100 p-3 text-sky-600">
                <Droplet className="h-5 w-5" />
              </div>
            </CardHeader>
            <CardContent className="grid gap-4 sm:grid-cols-2">
              <StatCard
                title="Consumo total"
                value={`${numberFormatter.format(roStats.totalM3)} m³`}
                helper={`${numberFormatter.format(roStats.totalLiters)} litros`}
                icon={Droplet}
                accent="bg-sky-500"
              />
              <StatCard
                title="Costo total"
                value={currencyFormatter.format(roStats.totalCost)}
                helper="Acumulado del periodo"
                icon={LineChart}
                accent="bg-sky-400"
              />
              <StatCard
                title="Costo unitario promedio"
                value={`${currencyFormatter.format(roStats.averageUnitCost)}/m³`}
                helper="Promedio de facturación"
                icon={Filter}
                accent="bg-sky-500"
              />
              <StatCard
                title="Proveedor principal"
                value={roStats.provider ?? "Sin proveedor"}
                helper={`${roStats.entries.length} registros`}
                icon={Factory}
                accent="bg-sky-400"
              />
            </CardContent>
          </Card>
        </section>

        <section className="space-y-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h2 className="text-2xl font-semibold text-gray-900">Registro de consumo</h2>
              <p className="text-sm text-gray-500">
                Detalle mensual de consumos de agua de red y recolecciones de agua RO.
              </p>
            </div>
          </div>
          <Card className="border-sky-100">
            {canManageWater && (
              <>
                <CardHeader>
                  <CardTitle className="text-lg text-gray-900">Nuevo registro de agua</CardTitle>
                  <p className="text-sm text-gray-500">
                    Completa el formulario para capturar un nuevo consumo o recolección.
                  </p>
                </CardHeader>
                <CardContent>
                  <form
                    className="grid gap-4 rounded-2xl border border-sky-100 bg-sky-50 p-4 md:grid-cols-2"
                    onSubmit={handleAddRecord}
                  >
                    <div className="space-y-2">
                      <Label htmlFor="water-type">Tipo de registro</Label>
                      <select
                        id="water-type"
                        className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                        value={formState.type}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            type: event.target.value === "RO" ? "RO" : "Red",
                          }))
                        }
                      >
                        <option value="Red">Agua de red</option>
                        <option value="RO">Agua RO</option>
                      </select>
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="water-date">Fecha de registro</Label>
                      <Input
                        id="water-date"
                        type="date"
                        value={formState.date}
                        onChange={(event) =>
                          setFormState((prev) => ({ ...prev, date: event.target.value }))
                        }
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="water-quantity">Cantidad (m³)</Label>
                      <Input
                        id="water-quantity"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 120.5"
                        value={formState.quantityM3}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            quantityM3: event.target.value,
                          }))
                        }
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="water-unit-cost">Costo unitario (MXN/m³)</Label>
                      <Input
                        id="water-unit-cost"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 12.50"
                        value={formState.unitCost}
                        onChange={(event) =>
                          setFormState((prev) => ({ ...prev, unitCost: event.target.value }))
                        }
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="water-provider">Proveedor</Label>
                      <Input
                        id="water-provider"
                        placeholder="Ej. Aguas de Aguascalientes"
                        value={formState.provider}
                        onChange={(event) =>
                          setFormState((prev) => ({ ...prev, provider: event.target.value }))
                        }
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="water-invoice">Factura (URL opcional)</Label>
                      <Input
                        id="water-invoice"
                        type="url"
                        placeholder="https://..."
                        value={formState.invoiceUrl}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            invoiceUrl: event.target.value,
                          }))
                        }
                      />
                    </div>
                    <div className="md:col-span-2 flex flex-wrap items-center justify-between gap-3">
                      <div className="text-xs text-gray-500">
                        El total y los litros se calculan automáticamente al guardar.
                      </div>
                      <Button className="gap-2 bg-sky-600 hover:bg-sky-700" type="submit">
                        <Plus className="h-4 w-4" />
                        Guardar registro
                      </Button>
                    </div>
                  </form>
                </CardContent>
              </>
            )}
          </Card>
          <Card className="border-gray-200">
            <CardContent className="p-0">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Tipo</TableHead>
                    <TableHead>Fecha registro</TableHead>
                    <TableHead>Mes</TableHead>
                    <TableHead>Año</TableHead>
                    <TableHead>Cantidad (m³)</TableHead>
                    <TableHead>Cantidad (L)</TableHead>
                    <TableHead>Costo unitario</TableHead>
                    <TableHead>Costo total</TableHead>
                    <TableHead>Proveedor</TableHead>
                    <TableHead>Factura</TableHead>
                    {canManageWater && (
                      <TableHead className="text-right">Acciones</TableHead>
                    )}
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredRecords.map((record) => (
                    <TableRow key={record.id}>
                      <TableCell>
                        <Badge
                          className={
                            record.type === "Red"
                              ? "bg-emerald-100 text-emerald-700"
                              : "bg-sky-100 text-sky-700"
                          }
                          variant="secondary"
                        >
                          {record.type === "Red" ? "Agua de red" : "Agua RO"}
                        </Badge>
                      </TableCell>
                      <TableCell className="font-medium text-gray-900">
                        {formatDate(record.date)}
                      </TableCell>
                      <TableCell>{record.month}</TableCell>
                      <TableCell>{record.year}</TableCell>
                      <TableCell>{numberFormatter.format(record.quantityM3)}</TableCell>
                      <TableCell>{numberFormatter.format(record.quantityLiters)}</TableCell>
                      <TableCell>{currencyFormatter.format(record.unitCost)}</TableCell>
                      <TableCell>{currencyFormatter.format(record.totalCost)}</TableCell>
                      <TableCell>{record.provider}</TableCell>
                      <TableCell>
                        {record.invoiceUrl ? (
                          <a
                            className="inline-flex items-center gap-1 text-sm font-medium text-sky-700 hover:text-sky-800"
                            href={record.invoiceUrl}
                          >
                            <FileText className="h-4 w-4" />
                            Ver factura
                          </a>
                        ) : (
                          <span className="text-sm text-gray-400">Sin factura</span>
                        )}
                      </TableCell>
                      {canManageWater && (
                        <TableCell>
                          <div className="flex justify-end gap-2">
                            <Button
                              size="sm"
                              variant="outline"
                              className="gap-1"
                              onClick={() => handleEditRecord(record.id)}
                            >
                              <Pencil className="h-4 w-4" />
                              Editar
                            </Button>
                            <Button
                              size="sm"
                              variant="outline"
                              className="gap-1 text-rose-600"
                              onClick={() => handleDeleteRecord(record.id)}
                            >
                              <Trash2 className="h-4 w-4" />
                              Eliminar
                            </Button>
                          </div>
                        </TableCell>
                      )}                  
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </section>
      </div>
    </SGIDashboardLayout>
  );
}