"use client";

import { useCallback, useEffect, useMemo, useState, type FormEvent } from "react";
import Link from "next/link";
import { useSearchParams } from "next/navigation";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { ChartContainer, ChartTooltip, ChartTooltipContent } from "@/components/ui/chart";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import {
  Award,
  Eye,
  FileDown,
  Leaf,
  Pencil,
  Plus,
  RefreshCw,
  Save,
  ShieldCheck,
  Trash2,
} from "lucide-react";
import { CartesianGrid, Line, LineChart, XAxis, YAxis } from "recharts";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);

type IndicatorRecord = {
  id: string;
  indicator_type: string | null;
  year: number | null;
  policy_id: string | null;
  objective_id: string | null;
  concept: string | null;
  element: string | null;
  indicator: string | null;
  pic: string | null;
  info_updater: string | null;
  jan: number | null;
  feb: number | null;
  mar: number | null;
  apr: number | null;
  may: number | null;
  jun: number | null;
  jul: number | null;
  aug: number | null;
  sep: number | null;
  oct: number | null;
  nov: number | null;
  dec: number | null;
  annual_sum_target: number | null;
  annual_average_target: number | null;
  compliance: boolean | null;
  last_updated_at: string | null;
  last_updated_by: string | null;
};

type ObjectiveIndicatorLink = {
  id: string;
  objective_id: string;
  indicator_id: string;
  weight: number | null;
  polarity: "higher_better" | "lower_better" | null;
  target_value: number | null;
  target_type: string | null;
  aggregation_method: string | null;
  active: boolean | null;
};

type ObjectiveSummary = {
  id: string;
  objective_id: string | null;
  target_indicator_value: number | null;
  target_date: string | null;
};

type IndicatorFormState = {
  indicator_type: string;
  year: string;
  policy_id: string;
  objective_id: string;
  concept: string;
  element: string;
  indicator: string;
  pic: string;
  info_updater: string;
  annual_sum_target: string;
  annual_average_target: string;
};

type MonthKey =
  | "jan"
  | "feb"
  | "mar"
  | "apr"
  | "may"
  | "jun"
  | "jul"
  | "aug"
  | "sep"
  | "oct"
  | "nov"
  | "dec";

type MonthEditState = Partial<Record<MonthKey, string>>;

type IndicatorEdits = Record<string, MonthEditState>;

type ObjectiveOption = {
  objective_id: string;
  objective_title: string | null;
  objective_type: string | null;
  policy_excerpt_id: string | null;  
};

const indicatorTypes = [
  { value: "environment", label: "Ambiental" },
  { value: "quality", label: "Calidad" },
  { value: "safety", label: "Seguridad" },
];

const categories = [
  {
    id: "environment",
    label: "Medio Ambiente",
    description: "Gestión ambiental y objetivos de sostenibilidad.",
    accent: "from-emerald-500 via-emerald-400 to-lime-400",
    icon: Leaf,
  },
  {
    id: "quality",
    label: "Calidad",
    description: "Metas de calidad y mejora continua.",
    accent: "from-sky-500 via-indigo-500 to-violet-500",
    icon: Award,
  },
  {
    id: "safety",
    label: "Seguridad",
    description: "Objetivos de seguridad y prevención.",
    accent: "from-rose-500 via-orange-500 to-amber-400",
    icon: ShieldCheck,
  },
];

const monthOptions: { key: MonthKey; label: string; full: string }[] = [
  { key: "jan", label: "ENE", full: "Enero" },
  { key: "feb", label: "FEB", full: "Febrero" },
  { key: "mar", label: "MAR", full: "Marzo" },
  { key: "apr", label: "ABR", full: "Abril" },
  { key: "may", label: "MAY", full: "Mayo" },
  { key: "jun", label: "JUN", full: "Junio" },
  { key: "jul", label: "JUL", full: "Julio" },
  { key: "aug", label: "AGO", full: "Agosto" },
  { key: "sep", label: "SEP", full: "Septiembre" },
  { key: "oct", label: "OCT", full: "Octubre" },
  { key: "nov", label: "NOV", full: "Noviembre" },
  { key: "dec", label: "DIC", full: "Diciembre" },
];

const emptyFormState: IndicatorFormState = {
  indicator_type: "environment",
  year: "",
  policy_id: "",
  objective_id: "",
  concept: "",
  element: "",
  indicator: "",
  pic: "",
  info_updater: "",
  annual_sum_target: "",
  annual_average_target: "",
};

const toCsvValue = (value: string | number | null) => {
  if (value === null || value === undefined) return "";
  const stringValue = `${value}`;
  if (stringValue.includes(",") || stringValue.includes("\n") || stringValue.includes('"')) {
    return `"${stringValue.replace(/"/g, '""')}"`;
  }
  return stringValue;
};

const parseNumberInput = (value: string) => {
  if (value === "") return null;
  const parsed = Number(value);
  return Number.isNaN(parsed) ? null : parsed;
};

const formatNumber = (value: number | null | undefined) =>
  value === null || value === undefined ? "-" : value.toString();

const clampValue = (value: number, min: number, max: number) =>
  Math.min(max, Math.max(min, value));

const computeIndicatorCompliance = (
  value: number | null | undefined,
  link: ObjectiveIndicatorLink,
) => {
  if (value === null || value === undefined) return null;
  const target = link.target_value ?? null;
  if (!target || target === 0) return null;
  const isLowerBetter =
    link.polarity === "lower_better" || link.target_type?.includes("<=") === true;
  const ratio = isLowerBetter ? 1 - value / target : value / target;
  return clampValue(ratio, 0, 1) * 100;
};

const computeObjectiveMonthlyProgress = (
  links: ObjectiveIndicatorLink[],
  indicatorMap: Map<string, IndicatorRecord>,
) => {
  return monthOptions.map((month) => {
    const entries = links
      .map((link) => {
        const indicator = indicatorMap.get(link.indicator_id);
        if (!indicator) return null;
        const compliance = computeIndicatorCompliance(indicator[month.key], link);
        if (compliance === null) return null;
        return {
          compliance,
          weight: link.weight ?? 0,
        };
      })
      .filter((entry): entry is { compliance: number; weight: number } => entry !== null);

    if (!entries.length) return null;
    const totalWeight = entries.reduce((sum, entry) => sum + entry.weight, 0);
    if (totalWeight > 0) {
      return entries.reduce((sum, entry) => sum + entry.compliance * (entry.weight / totalWeight), 0);
    }
    const equalWeight = 1 / entries.length;
    return entries.reduce((sum, entry) => sum + entry.compliance * equalWeight, 0);
  });
};

const computeObjectiveStatus = (
  currentValue: number | null,
  targetValue: number | null,
  targetDate: string | null,
) => {
  if (currentValue !== null && targetValue !== null && currentValue >= targetValue) {
    return "Cumplido";
  }
  if (targetDate) {
    const target = new Date(targetDate);
    if (target < new Date()) {
      return "Retrasado";
    }
  }
  return "En progreso";
};

const computeNextReviewDate = (frequency: string | null, baseDate: Date) => {
  if (!frequency) return null;
  const next = new Date(baseDate);
  switch (frequency) {
    case "Mensual":
      next.setMonth(next.getMonth() + 1);
      break;
    case "Trimestral":
      next.setMonth(next.getMonth() + 3);
      break;
    case "Semestral":
      next.setMonth(next.getMonth() + 6);
      break;
    case "Anual":
      next.setFullYear(next.getFullYear() + 1);
      break;
    default:
      return null;
  }
  return next.toISOString().split("T")[0];
};

const computeAnnualSummary = (record: IndicatorRecord) => {
  const values = monthOptions
    .map((month) => record[month.key])
    .filter((value): value is number => value !== null && value !== undefined);
  if (!values.length) {
    return { sum: null, average: null };
  }
  const sum = values.reduce((total, current) => total + current, 0);
  const average = values.length ? sum / values.length : null;
  return { sum, average };
};

const formatAnnualTargetMatch = (
  summary: { sum: number | null; average: number | null },
  record: IndicatorRecord,
) => {
  if (
    summary.sum === null ||
    summary.average === null ||
    record.annual_sum_target == null ||
    record.annual_average_target == null
  ) {
    return null;
  }
  const averageRounded = Number(summary.average.toFixed(2));
  const sumMatches = summary.sum === record.annual_sum_target;
  const averageMatches = averageRounded === record.annual_average_target;
  return sumMatches && averageMatches;
};

const getEditableWindow = () => {
  const now = new Date();
  const day = now.getDate();
  const monthIndex = now.getMonth();
  const year = now.getFullYear();

  if (day > 5) {
    return {
      windowOpen: false,
      editableMonthIndex: null as number | null,
      editableYear: null as number | null,
    };
  }

  const editableMonthIndex = monthIndex === 0 ? 11 : monthIndex - 1;
  const editableYear = monthIndex === 0 ? year - 1 : year;

  return {
    windowOpen: true,
    editableMonthIndex,
    editableYear,
  };
};

export default function EnvironmentalIndicatorsPage() {
  const searchParams = useSearchParams();    
  const [user, setUser] = useState<User | null>(null);
  const [records, setRecords] = useState<IndicatorRecord[]>([]);
  const [loading, setLoading] = useState(true);
  const [activeType, setActiveType] = useState(categories[0].id);
  const [search, setSearch] = useState("");
  const [filters, setFilters] = useState({
    year: "all",
    policyId: "",
    objectiveId: "",
    pic: "all",
    infoUpdater: "all",
  });
  const [selectedId, setSelectedId] = useState<string | null>(null);
  const [detailOpen, setDetailOpen] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingRecord, setEditingRecord] = useState<IndicatorRecord | null>(null);
  const [formState, setFormState] = useState<IndicatorFormState>(emptyFormState);
  const [monthlyEdits, setMonthlyEdits] = useState<IndicatorEdits>({});
  const [isSaving, setIsSaving] = useState(false);
  const [objectives, setObjectives] = useState<ObjectiveOption[]>([]);
  const [objectivesLoading, setObjectivesLoading] = useState(false);

  const canManage = allowedManagers.has(user?.email ?? "");

  const { windowOpen, editableMonthIndex, editableYear } = getEditableWindow();

  const fetchRecords = useCallback(async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("indicators_yearly")
      .select("*")
      .order("year", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar los indicadores.");
      setLoading(false);
      return;
    }

    setRecords(data ?? []);
    setLoading(false);
  }, []);

  const fetchObjectives = useCallback(async () => {
    setObjectivesLoading(true);
    const { data, error } = await supabase
      .from("objectives")
      .select("objective_id, objective_title, objective_type, policy_excerpt_id")
      .order("objective_id", { ascending: true });

    if (error) {
      console.error(error);
      toast.error("No se pudieron cargar los objetivos.");
      setObjectives([]);
      setObjectivesLoading(false);
      return;
    }

    const options = (data ?? []).filter(
      (objective): objective is ObjectiveOption => Boolean(objective.objective_id),
    );
    setObjectives(options);
    setObjectivesLoading(false);
  }, []);

  useEffect(() => {
    const loadUser = async () => {
      const { data } = await supabase.auth.getUser();
      setUser(data.user ?? null);
    };

    void loadUser();
    void fetchRecords();
    void fetchObjectives();
  }, [fetchObjectives, fetchRecords]);

  useEffect(() => {
    const objectiveId = searchParams.get("objectiveId");
    if (objectiveId) {
      setFilters((prev) => ({ ...prev, objectiveId }));
    }
  }, [searchParams]);

  const selectedRecord = useMemo(
    () => records.find((record) => record.id === selectedId) ?? null,
    [records, selectedId],
  );
  const detailSummary = useMemo(
    () => (selectedRecord ? computeAnnualSummary(selectedRecord) : null),
    [selectedRecord],
  );
  const detailTargetMatch = useMemo(() => {
    if (!selectedRecord || !detailSummary) return null;
    return formatAnnualTargetMatch(detailSummary, selectedRecord);
  }, [detailSummary, selectedRecord]);

  const yearOptions = useMemo(() => {
    const years = new Set(
      records.map((record) => record.year).filter((year): year is number => year !== null),
    );
    return Array.from(years).sort((a, b) => b - a);
  }, [records]);

  const activeCategory = useMemo(
    () => categories.find((category) => category.id === activeType),
    [activeType],
  );

  const availableObjectiveOptions = useMemo(() => {
    return objectives.filter((objective) => objective.objective_type === formState.indicator_type);
  }, [formState.indicator_type, objectives]);

  useEffect(() => {
    if (!formState.objective_id) return;
    const exists = availableObjectiveOptions.some(
      (objective) => objective.objective_id === formState.objective_id,
    );
    if (!exists) {
      setFormState((prev) => ({ ...prev, objective_id: "", policy_id: "" }));
    }
  }, [availableObjectiveOptions, formState.objective_id]);

  useEffect(() => {
    if (!formState.objective_id) {
      if (formState.policy_id) {
        setFormState((prev) => ({ ...prev, policy_id: "" }));
      }
      return;
    }

    const selectedObjective = objectives.find(
      (objective) => objective.objective_id === formState.objective_id,
    );
    const policyId = selectedObjective?.policy_excerpt_id ?? "";

    if (policyId !== formState.policy_id) {
      setFormState((prev) => ({ ...prev, policy_id: policyId }));
    }
  }, [formState.objective_id, formState.policy_id, objectives]);

  const picOptions = useMemo(() => {
    const pics = new Set(
      records.map((record) => record.pic).filter((value): value is string => Boolean(value)),
    );
    return Array.from(pics).sort();
  }, [records]);

  const infoUpdaterOptions = useMemo(() => {
    const updaters = new Set(
      records
        .map((record) => record.info_updater)
        .filter((value): value is string => Boolean(value)),
    );
    return Array.from(updaters).sort();
  }, [records]);

  const filteredRecords = useMemo(() => {
    const searchValue = search.trim().toLowerCase();
    return records.filter((record) => {
      if (record.indicator_type !== activeType) return false;
      if (filters.year !== "all" && `${record.year ?? ""}` !== filters.year) return false;
      if (filters.policyId && !record.policy_id?.toLowerCase().includes(filters.policyId.toLowerCase())) {
        return false;
      }
      if (
        filters.objectiveId &&
        !record.objective_id?.toLowerCase().includes(filters.objectiveId.toLowerCase())
      ) {
        return false;
      }
      if (filters.pic !== "all" && record.pic !== filters.pic) return false;
      if (filters.infoUpdater !== "all" && record.info_updater !== filters.infoUpdater) {
        return false;
      }
      if (!searchValue) return true;
      const haystack = [record.concept, record.element, record.indicator]
        .filter(Boolean)
        .join(" ")
        .toLowerCase();
      return haystack.includes(searchValue);
    });
  }, [records, activeType, filters, search]);

  const editableMessage = useMemo(() => {
    if (!windowOpen || editableMonthIndex === null || editableYear === null) {
      return "Ventana cerrada: edición bloqueada (a partir del día 6).";
    }
    const monthName = monthOptions[editableMonthIndex]?.full ?? "";
    return `Ventana abierta: puedes actualizar ${monthName} ${editableYear} hasta el día 5.`;
  }, [windowOpen, editableMonthIndex, editableYear]);

  const isMonthEditable = useCallback(
    (record: IndicatorRecord, monthIndex: number) => {
      const recordYear = record.year ?? 0;
      if (recordYear <= 2025) return true;
      if (!windowOpen || editableMonthIndex === null || editableYear === null) return false;
      return recordYear === editableYear && monthIndex === editableMonthIndex;
    },
    [editableMonthIndex, editableYear, windowOpen],
  );

  const recalculateObjectives = useCallback(async (indicatorIds: string[]) => {
    if (!indicatorIds.length) return [];
    const { data: linkedIndicators, error: linksError } = await supabase
      .from("objective_indicator_links")
      .select("*")
      .in("indicator_id", indicatorIds)
      .or("active.eq.true,active.is.null");

    const linkedIndicatorData = (linkedIndicators ?? []) as ObjectiveIndicatorLink[];

    if (linksError || !linkedIndicatorData.length) {
      if (linksError) console.error(linksError);
      return [];
    }

    const objectiveIds = Array.from(
      new Set(linkedIndicatorData.map((link) => link.objective_id)),
    );

    const { data: links, error: allLinksError } = await supabase
      .from("objective_indicator_links")
      .select("*")
      .in("objective_id", objectiveIds)
      .or("active.eq.true,active.is.null");

    const linkData = (links ?? []) as ObjectiveIndicatorLink[];

    if (allLinksError || !linkData.length) {
      if (allLinksError) console.error(allLinksError);
      return [];
    }

    const indicatorIdsForObjectives = Array.from(
      new Set(linkData.map((link) => link.indicator_id)),
    );

    const { data: indicators, error: indicatorsError } = await supabase
      .from("indicators_yearly")
      .select("*")
      .in("id", indicatorIdsForObjectives);

    if (indicatorsError || !indicators?.length) {
      if (indicatorsError) console.error(indicatorsError);
      return [];
    }

    const { data: objectives, error: objectivesError } = await supabase
      .from("objectives")
      .select("id, objective_id, target_indicator_value, target_date, review_frequency")
      .in("id", objectiveIds);

    if (objectivesError || !objectives?.length) {
      if (objectivesError) console.error(objectivesError);
      return [];
    }

    const indicatorMap = new Map<string, IndicatorRecord>(
      indicators.map((indicator) => [indicator.id, indicator]),
    );

    const updates = objectives.map(async (objective) => {
      const linksForObjective = linkData.filter((link) => link.objective_id === objective.id);
      const monthlyProgress = computeObjectiveMonthlyProgress(
        linksForObjective,
        indicatorMap,
      );
      const progressValues = monthlyProgress.filter(
        (value): value is number => value !== null && value !== undefined,
      );
      const averageProgress =
        progressValues.length > 0
          ? progressValues.reduce((sum, value) => sum + value, 0) / progressValues.length
          : null;
      const roundedProgress =
        averageProgress === null ? null : Math.round(averageProgress * 100) / 100;
      const status = computeObjectiveStatus(
        roundedProgress,
        objective.target_indicator_value ?? null,
        objective.target_date ?? null,
      );
      const nextReviewDate = computeNextReviewDate(
        objective.review_frequency ?? null,
        new Date(),
      );

      await supabase
        .from("objectives")
        .update({
          current_indicator_value: roundedProgress,
          status,
          last_updated_at: new Date().toISOString(),
          next_review_date: nextReviewDate,
        })
        .eq("id", objective.id);

      return objective.objective_id ?? objective.id;
    });

    return Promise.all(updates);
  }, []);

  const handleMonthlyChange = (recordId: string, monthKey: MonthKey, value: string) => {
    setMonthlyEdits((prev) => ({
      ...prev,
      [recordId]: {
        ...prev[recordId],
        [monthKey]: value,
      },
    }));
  };

  const handleSaveMonthlyChanges = async () => {
    if (!Object.keys(monthlyEdits).length) return;
    setIsSaving(true);

    const updatedRecordIds = Object.keys(monthlyEdits);

    const updates = Object.entries(monthlyEdits).map(async ([recordId, edits]) => {
      const payload: Partial<IndicatorRecord> = {
        last_updated_at: new Date().toISOString(),
        last_updated_by: user?.email ?? null,
      };

      monthOptions.forEach((month) => {
        if (month.key in edits) {
          const value = edits[month.key];
          payload[month.key] = value === undefined ? null : parseNumberInput(value);
        }
      });

      const { error } = await supabase
        .from("indicators_yearly")
        .update(payload)
        .eq("id", recordId);

      if (error) {
        throw error;
      }
    });

    try {
      await Promise.all(updates);
      const recalculatedObjectives = await recalculateObjectives(updatedRecordIds);
      toast.success("Cambios guardados correctamente.");
      if (recalculatedObjectives.length) {
        toast.success(
          `Indicador actualizado. Objetivo${recalculatedObjectives.length > 1 ? "s" : ""} ${recalculatedObjectives.join(
            ", ",
          )} recalculado${recalculatedObjectives.length > 1 ? "s" : ""} y estado actualizado.`,
        );
      }
      setMonthlyEdits({});
      await fetchRecords();
    } catch (error) {
      toast.error("No se pudieron guardar los cambios.");
    } finally {
      setIsSaving(false);
    }
  };

  const handleOpenCreate = () => {
    setEditingRecord(null);
    setFormState({ ...emptyFormState, indicator_type: activeType });
    setDialogOpen(true);
  };

  const handleOpenEdit = () => {
    if (!selectedRecord) return;
    setEditingRecord(selectedRecord);
    setFormState({
      indicator_type: selectedRecord.indicator_type ?? activeType,
      year: selectedRecord.year?.toString() ?? "",
      policy_id: selectedRecord.policy_id ?? "",
      objective_id: selectedRecord.objective_id ?? "",
      concept: selectedRecord.concept ?? "",
      element: selectedRecord.element ?? "",
      indicator: selectedRecord.indicator ?? "",
      pic: selectedRecord.pic ?? "",
      info_updater: selectedRecord.info_updater ?? "",
      annual_sum_target: selectedRecord.annual_sum_target?.toString() ?? "",
      annual_average_target: selectedRecord.annual_average_target?.toString() ?? "",
    });
    setDialogOpen(true);
  };

  const handleSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setIsSaving(true);

    if (!formState.objective_id) {
      toast.error("Selecciona un objetivo existente.");
      setIsSaving(false);
      return;
    }

    const objectiveExists = objectives.some(
      (objective) => objective.objective_id === formState.objective_id,
    );

    if (!objectiveExists) {
      toast.error("El objetivo seleccionado no existe.");
      setIsSaving(false);
      return;
    }

    const payload = {
      indicator_type: formState.indicator_type,
      year: formState.year ? Number(formState.year) : null,
      policy_id: formState.policy_id || null,
      objective_id: formState.objective_id || null,
      concept: formState.concept || null,
      element: formState.element || null,
      indicator: formState.indicator || null,
      pic: formState.pic || null,
      info_updater: formState.info_updater || null,
      annual_sum_target: parseNumberInput(formState.annual_sum_target),
      annual_average_target: parseNumberInput(formState.annual_average_target),
      last_updated_at: new Date().toISOString(),
      last_updated_by: user?.email ?? null,
    };

    const { data, error } = editingRecord
      ? await supabase
          .from("indicators_yearly")
          .update(payload)
          .eq("id", editingRecord.id)
          .select()
          .single()
      : await supabase.from("indicators_yearly").insert(payload).select().single();

    if (error) {
      toast.error("No se pudo guardar el indicador.");
      setIsSaving(false);
      return;
    }

    toast.success(editingRecord ? "Indicador actualizado." : "Indicador creado.");
    if (data?.id) {
      await recalculateObjectives([data.id]);
    }
    setDialogOpen(false);
    setEditingRecord(null);
    setFormState(emptyFormState);
    await fetchRecords();
    setIsSaving(false);
  };

  const handleDelete = async () => {
    if (!selectedRecord) return;
    const confirmation = window.confirm("¿Seguro que deseas eliminar este indicador?");
    if (!confirmation) return;

    const { error } = await supabase.from("indicators_yearly").delete().eq("id", selectedRecord.id);

    if (error) {
      toast.error("No se pudo eliminar el indicador.");
      return;
    }

    toast.success("Indicador eliminado.");
    setSelectedId(null);
    await fetchRecords();
  };

  const handleExport = () => {
    if (!filteredRecords.length) {
      toast.info("No hay datos para exportar.");
      return;
    }

    const headers = [
      "Tipo",
      "Año",
      "ID Política",
      "ID Objetivo",
      "Concepto",
      "Elemento",
      "Indicador",
      "PIC",
      "Actualiza",
      "Sumatoria anual",
      "Promedio anual",
      "Objetivo sumatoria anual",
      "Objetivo promedio anual",
      "Cumplimiento",
      ...monthOptions.map((month) => month.label),
    ];

    const rows = filteredRecords.map((record) => {
      const summary = computeAnnualSummary(record);
      const targetMatch = formatAnnualTargetMatch(summary, record);
      return [
        record.indicator_type,
        record.year,
        record.policy_id,
        record.objective_id,
        record.concept,
        record.element,
        record.indicator,
        record.pic,
        record.info_updater,
        summary.sum,
        summary.average,
        record.annual_sum_target,
        record.annual_average_target,
        targetMatch === null ? null : targetMatch ? "Cumple" : "No cumple",
        ...monthOptions.map((month) => record[month.key]),
      ].map(toCsvValue);
    });

    const csvContent = [headers, ...rows].map((row) => row.join(",")).join("\n");
    const blob = new Blob([csvContent], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.download = `indicadores-${activeType}.csv`;
    link.click();
    URL.revokeObjectURL(url);
  };

  const hasMonthlyEdits = Object.keys(monthlyEdits).length > 0;

  const complianceStats = useMemo(() => {
    let compliant = 0;
    let nonCompliant = 0;
    let pending = 0;

    filteredRecords.forEach((record) => {
      const summary = computeAnnualSummary(record);
      const targetMatch = formatAnnualTargetMatch(summary, record);
      if (targetMatch === true) {
        compliant += 1;
      } else if (targetMatch === false) {
        nonCompliant += 1;
      } else {
        pending += 1;
      }
    });

    const total = filteredRecords.length;
    const complianceRate = total ? Math.round((compliant / total) * 100) : null;

    return {
      total,
      compliant,
      nonCompliant,
      pending,
      complianceRate,
    };
  }, [filteredRecords]);

  const monthlyTrendData = useMemo(() => {
    return monthOptions.map((month) => {
      const monthValues = filteredRecords
        .map((record) => record[month.key])
        .filter((value): value is number => value !== null && value !== undefined);
      const average = monthValues.length
        ? Number((monthValues.reduce((sum, value) => sum + value, 0) / monthValues.length).toFixed(2))
        : null;
      return {
        month: month.full,
        short: month.label,
        value: average,
      };
    });
  }, [filteredRecords]);

  const monthlyTrendHasData = useMemo(
    () => monthlyTrendData.some((item) => item.value !== null && item.value !== undefined),
    [monthlyTrendData],
  );

  const monthlyChartConfig = useMemo(
    () => ({
      value: {
        label: "Promedio mensual",
        color: "hsl(160 84% 39%)",
      },
    }),
    [],
  );

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-16">
        <section className="rounded-3xl border border-emerald-100 bg-white p-8 shadow-sm">
          <div className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div>
              <p className="text-xs font-semibold uppercase tracking-[0.4em] text-emerald-600">
                Planificación
              </p>
              <h1 className="mt-2 text-3xl font-bold text-gray-900">
                Indicadores ambientales, calidad y seguridad
              </h1>
              <p className="mt-2 max-w-3xl text-sm text-gray-600">
                Administra y actualiza los indicadores mensuales con control por ventana de edición
                y permisos por responsable.
              </p>
            </div>
            <Badge className="w-fit bg-emerald-100 text-emerald-700">{editableMessage}</Badge>
          </div>
        </section>

        <section className="flex flex-wrap gap-3">
          {categories.map((category) => {
            const Icon = category.icon;
            const isActive = activeType === category.id;

            return (
              <button
                key={category.id}
                type="button"
                onClick={() => setActiveType(category.id)}
                className={`flex items-center gap-2 rounded-full border px-4 py-2 text-sm font-semibold transition-all duration-200 ${
                  isActive
                    ? "border-transparent bg-gradient-to-r from-emerald-500 to-lime-500 text-white shadow-lg"
                    : "border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
                }`}
              >
                <Icon className={`h-4 w-4 ${isActive ? "text-white" : "text-emerald-500"}`} />
                {category.label}
              </button>
            );
          })}
        </section>

        <Card className="border-emerald-100">
          <CardHeader className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div>
              <CardTitle>Gestión de indicadores</CardTitle>
              <p className="text-sm text-gray-500">
                Selecciona el tipo y administra los indicadores del año correspondiente.
              </p>
            </div>
            <div className="flex flex-wrap items-center gap-2">
              {canManage && (
                <Button onClick={handleOpenCreate} className="gap-2">
                  <Plus className="h-4 w-4" /> Nuevo indicador
                </Button>
              )}
              {canManage && (
                <Button
                  variant="secondary"
                  className="gap-2"
                  onClick={handleOpenEdit}
                  disabled={!selectedRecord}
                >
                  <Pencil className="h-4 w-4" /> Editar indicador
                </Button>
              )}
              <Button
                variant="outline"
                className="gap-2"
                onClick={handleSaveMonthlyChanges}
                disabled={!hasMonthlyEdits || isSaving}
              >
                <Save className="h-4 w-4" /> Guardar cambios
              </Button>
              <Button
                variant="outline"
                className="gap-2"
                onClick={() => setDetailOpen(true)}
                disabled={!selectedRecord}
              >
                <Eye className="h-4 w-4" /> Ver detalle
              </Button>
              {canManage && (
                <Button
                  variant="destructive"
                  className="gap-2"
                  onClick={handleDelete}
                  disabled={!selectedRecord}
                >
                  <Trash2 className="h-4 w-4" /> Eliminar
                </Button>
              )}
              <Button variant="outline" className="gap-2" onClick={handleExport}>
                <FileDown className="h-4 w-4" /> Exportar
              </Button>
              <Button variant="ghost" className="gap-2" onClick={fetchRecords}>
                <RefreshCw className="h-4 w-4" /> Recargar
              </Button>
            </div>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="grid gap-4 rounded-2xl border border-emerald-100 bg-emerald-50 p-4 sm:grid-cols-2 xl:grid-cols-5">
              <div className="space-y-1 rounded-xl bg-white p-4 shadow-sm">
                <p className="text-xs uppercase tracking-wide text-emerald-700">Total</p>
                <p className="text-2xl font-semibold text-gray-900">{complianceStats.total}</p>
                <p className="text-xs text-gray-500">Indicadores filtrados</p>
              </div>
              <div className="space-y-1 rounded-xl bg-white p-4 shadow-sm">
                <p className="text-xs uppercase tracking-wide text-emerald-700">Cumplen</p>
                <p className="text-2xl font-semibold text-emerald-700">
                  {complianceStats.compliant}
                </p>
                <p className="text-xs text-gray-500">Meta anual alcanzada</p>
              </div>
              <div className="space-y-1 rounded-xl bg-white p-4 shadow-sm">
                <p className="text-xs uppercase tracking-wide text-emerald-700">No cumplen</p>
                <p className="text-2xl font-semibold text-rose-600">
                  {complianceStats.nonCompliant}
                </p>
                <p className="text-xs text-gray-500">Requieren acciones</p>
              </div>
              <div className="space-y-1 rounded-xl bg-white p-4 shadow-sm">
                <p className="text-xs uppercase tracking-wide text-emerald-700">Sin datos</p>
                <p className="text-2xl font-semibold text-gray-700">
                  {complianceStats.pending}
                </p>
                <p className="text-xs text-gray-500">Pendientes de cálculo</p>
              </div>
              <div className="space-y-1 rounded-xl bg-white p-4 shadow-sm">
                <p className="text-xs uppercase tracking-wide text-emerald-700">Cumplimiento</p>
                <p className="text-2xl font-semibold text-emerald-700">
                  {complianceStats.complianceRate === null
                    ? "-"
                    : `${complianceStats.complianceRate}%`}
                </p>
                <p className="text-xs text-gray-500">Sobre indicadores activos</p>
              </div>
            </div>
            <div className="rounded-2xl border border-emerald-100 bg-white p-4 shadow-sm">
              <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <p className="text-sm font-semibold text-gray-900">
                    Evolución mensual de indicadores ambientales
                  </p>
                  <p className="text-xs text-gray-500">
                    Promedio mensual de los valores capturados por indicador.
                  </p>
                </div>
                {activeCategory && (
                  <Badge className="w-fit bg-emerald-100 text-emerald-700">
                    {activeCategory.label}
                  </Badge>
                )}
              </div>
              <div className="mt-4">
                {monthlyTrendHasData ? (
                  <ChartContainer config={monthlyChartConfig} className="h-[280px] w-full">
                    <LineChart data={monthlyTrendData} margin={{ left: 8, right: 16, top: 12 }}>
                      <CartesianGrid strokeDasharray="4 4" />
                      <XAxis dataKey="short" />
                      <YAxis
                        tickLine={false}
                        axisLine={false}
                        tickFormatter={(value) => (value === null ? "" : `${value}`)}
                      />
                      <ChartTooltip
                        content={
                          <ChartTooltipContent
                            indicator="line"
                            labelFormatter={(_, payload) =>
                              payload?.[0]?.payload?.month ?? "Mes"
                            }
                          />
                        }
                      />
                      <Line
                        type="monotone"
                        dataKey="value"
                        stroke="var(--color-value)"
                        strokeWidth={2}
                        dot={{ r: 3, strokeWidth: 2 }}
                        activeDot={{ r: 5 }}
                        connectNulls
                      />
                    </LineChart>
                  </ChartContainer>
                ) : (
                  <div className="rounded-xl border border-dashed border-emerald-200 bg-emerald-50 p-6 text-center text-sm text-emerald-700">
                    No hay datos suficientes para graficar la evolución mensual.
                  </div>
                )}
              </div>
            </div>            
            <div className="grid gap-4 rounded-2xl border border-slate-100 bg-slate-50 p-4 md:grid-cols-2 xl:grid-cols-4">
              <div className="space-y-2">
                <Label>Año</Label>
                <Select
                  value={filters.year}
                  onValueChange={(value) => setFilters((prev) => ({ ...prev, year: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {yearOptions.map((year) => (
                      <SelectItem key={year} value={`${year}`}>
                        {year}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>ID Política</Label>
                <Input
                  value={filters.policyId}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, policyId: event.target.value }))
                  }
                  placeholder="Buscar por ID"
                />
              </div>
              <div className="space-y-2">
                <Label>ID Objetivo</Label>
                <Input
                  value={filters.objectiveId}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, objectiveId: event.target.value }))
                  }
                  placeholder="Buscar por ID"
                />
              </div>
              <div className="space-y-2">
                <Label>PIC</Label>
                <Select
                  value={filters.pic}
                  onValueChange={(value) => setFilters((prev) => ({ ...prev, pic: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {picOptions.map((pic) => (
                      <SelectItem key={pic} value={pic}>
                        {pic}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>¿Quién actualiza?</Label>
                <Select
                  value={filters.infoUpdater}
                  onValueChange={(value) => setFilters((prev) => ({ ...prev, infoUpdater: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {infoUpdaterOptions.map((updater) => (
                      <SelectItem key={updater} value={updater}>
                        {updater}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2 md:col-span-2 xl:col-span-3">
                <Label>Búsqueda (Concepto/Elemento/Indicador)</Label>
                <Input
                  value={search}
                  onChange={(event) => setSearch(event.target.value)}
                  placeholder="Buscar texto libre"
                />
              </div>
            </div>

            <div className="overflow-auto rounded-2xl border border-slate-200 bg-white">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-[120px]">Año</TableHead>
                    <TableHead className="min-w-[140px]">ID POLÍTICA</TableHead>
                    <TableHead className="min-w-[140px]">ID OBJETIVO</TableHead>
                    <TableHead className="min-w-[180px]">CONCEPTO</TableHead>
                    <TableHead className="min-w-[180px]">ELEMENTO</TableHead>
                    <TableHead className="min-w-[200px]">INDICADOR</TableHead>
                    <TableHead className="min-w-[160px]">PIC</TableHead>
                    <TableHead className="min-w-[160px]">¿QUIÉN ACTUALIZA?</TableHead>
                    <TableHead className="min-w-[150px]">SUMATORIA ANUAL</TableHead>
                    <TableHead className="min-w-[150px]">PROMEDIO ANUAL</TableHead>
                    <TableHead className="min-w-[180px]">OBJETIVO SUMATORIA ANUAL</TableHead>
                    <TableHead className="min-w-[180px]">OBJETIVO PROMEDIO ANUAL</TableHead>
                    <TableHead className="min-w-[140px]">CUMPLIMIENTO</TableHead>
                    {monthOptions.map((month, index) => (
                      <TableHead
                        key={month.key}
                        className={`min-w-[110px] ${
                          windowOpen && editableMonthIndex === index
                            ? "bg-emerald-50 text-emerald-700"
                            : ""
                        }`}
                      >
                        <div className="flex flex-col">
                          <span>{month.label}</span>
                          {windowOpen && editableMonthIndex === index && (
                            <Badge className="mt-1 w-fit bg-emerald-100 text-emerald-700">
                              Editable (1–5)
                            </Badge>
                          )}
                        </div>
                      </TableHead>
                    ))}
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {loading ? (
                    <TableRow>
                      <TableCell colSpan={monthOptions.length + 13}>
                        Cargando indicadores...
                      </TableCell>
                    </TableRow>
                  ) : filteredRecords.length ? (
                    filteredRecords.map((record) => {
                      const summary = computeAnnualSummary(record);
                      const targetMatch = formatAnnualTargetMatch(summary, record);
                      const isSelected = record.id === selectedId;
                      return (
                        <TableRow
                          key={record.id}
                          onClick={() => setSelectedId(record.id)}
                          className={isSelected ? "bg-emerald-50" : ""}
                        >
                          <TableCell>{record.year ?? "-"}</TableCell>
                          <TableCell>{record.policy_id ?? "-"}</TableCell>
                          <TableCell>
                            {record.objective_id ? (
                              <Link
                                href={`/meiwa-core/meiwa-sgi/planning/environmental-objectives?objectiveId=${encodeURIComponent(
                                  record.objective_id,
                                )}`}
                                className="font-semibold text-emerald-700 hover:underline"
                              >
                                {record.objective_id}
                              </Link>
                            ) : (
                              "-"
                            )}
                          </TableCell>
                          <TableCell>{record.concept ?? "-"}</TableCell>
                          <TableCell>{record.element ?? "-"}</TableCell>
                          <TableCell>{record.indicator ?? "-"}</TableCell>
                          <TableCell>{record.pic ?? "-"}</TableCell>
                          <TableCell>{record.info_updater ?? "-"}</TableCell>
                          <TableCell>{formatNumber(summary.sum)}</TableCell>
                          <TableCell>
                            {summary.average === null
                              ? "-"
                              : summary.average.toFixed(2)}
                          </TableCell>
                          <TableCell>{formatNumber(record.annual_sum_target)}</TableCell>
                          <TableCell>
                            {record.annual_average_target == null
                              ? "-"
                              : record.annual_average_target.toFixed(2)}
                          </TableCell>
                          <TableCell>
                            {targetMatch === null ? (
                              "-"
                            ) : targetMatch ? (
                              <Badge className="bg-emerald-100 text-emerald-700">
                                Cumple
                              </Badge>
                            ) : (
                              <Badge className="bg-rose-100 text-rose-700">No cumple</Badge>
                            )}
                          </TableCell>
                          {monthOptions.map((month, index) => {
                            const editable = isMonthEditable(record, index);
                            const editValue = monthlyEdits[record.id]?.[month.key];
                            const displayValue = editValue ??
                              (record[month.key] === null || record[month.key] === undefined
                                ? ""
                                : record[month.key]?.toString());
                            return (
                              <TableCell
                                key={`${record.id}-${month.key}`}
                                className={editable ? "bg-emerald-50" : ""}
                                title={
                                  editable
                                    ? "Editable (1–5)."
                                    : "Años 2026+: solo se edita mes anterior del 1 al 5."
                                }
                              >
                                {editable ? (
                                  <Input
                                    type="number"
                                    step="any"
                                    value={displayValue}
                                    onChange={(event) =>
                                      handleMonthlyChange(record.id, month.key, event.target.value)
                                    }
                                    className="h-8 w-24"
                                  />
                                ) : (
                                  <span>{formatNumber(record[month.key])}</span>
                                )}
                              </TableCell>
                            );
                          })}
                        </TableRow>
                      );
                    })
                  ) : (
                    <TableRow>
                      <TableCell colSpan={monthOptions.length + 13}>
                        No hay indicadores registrados para esta vista.
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>
              {editingRecord ? "Editar indicador" : "Nuevo indicador"}
            </DialogTitle>
          </DialogHeader>
          <form className="space-y-4" onSubmit={handleSubmit}>
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Tipo</Label>
                <Select
                  value={formState.indicator_type}
                  onValueChange={(value) =>
                    setFormState((prev) => ({ ...prev, indicator_type: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona" />
                  </SelectTrigger>
                  <SelectContent>
                    {indicatorTypes.map((type) => (
                      <SelectItem key={type.value} value={type.value}>
                        {type.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Año</Label>
                <Input
                  type="number"
                  value={formState.year}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, year: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>ID Política</Label>
                <Input
                  value={formState.policy_id}
                  readOnly
                  className="bg-slate-50"
                />
              </div>
              <div className="space-y-2">
                <Label>Objetivo existente</Label>
                <Select
                  value={formState.objective_id}
                  onValueChange={(value) =>
                    setFormState((prev) => ({ ...prev, objective_id: value }))
                  }
                  disabled={objectivesLoading || availableObjectiveOptions.length === 0}
                >
                  <SelectTrigger>
                    <SelectValue
                      placeholder={
                        objectivesLoading
                          ? "Cargando objetivos..."
                          : availableObjectiveOptions.length
                            ? "Selecciona un objetivo"
                            : "Sin objetivos disponibles"
                      }
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {availableObjectiveOptions.map((objective) => (
                      <SelectItem
                        key={objective.objective_id}
                        value={objective.objective_id}
                      >
                        {objective.objective_id}
                        {objective.objective_title ? ` - ${objective.objective_title}` : ""}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Concepto</Label>
                <Input
                  value={formState.concept}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, concept: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Elemento</Label>
                <Input
                  value={formState.element}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, element: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Indicador</Label>
                <Input
                  value={formState.indicator}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, indicator: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>PIC</Label>
                <Input
                  value={formState.pic}
                  onChange={(event) => setFormState((prev) => ({ ...prev, pic: event.target.value }))}
                />
              </div>
              <div className="space-y-2">
                <Label>Objetivo sumatoria anual</Label>
                <Input
                  type="number"
                  step="any"
                  value={formState.annual_sum_target}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, annual_sum_target: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Objetivo promedio anual</Label>
                <Input
                  type="number"
                  step="any"
                  value={formState.annual_average_target}
                  onChange={(event) =>
                    setFormState((prev) => ({
                      ...prev,
                      annual_average_target: event.target.value,
                    }))
                  }
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>¿Quién actualiza la información?</Label>
                <Input
                  value={formState.info_updater}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, info_updater: event.target.value }))
                  }
                />
              </div>
            </div>
            <div className="flex justify-end gap-2">
              <Button type="button" variant="secondary" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit" disabled={isSaving}>
                {editingRecord ? "Guardar cambios" : "Crear indicador"}
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={detailOpen} onOpenChange={setDetailOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Detalle del indicador</DialogTitle>
          </DialogHeader>
          {selectedRecord ? (
            <div className="space-y-4 text-sm">
              <div className="grid gap-3 sm:grid-cols-2">
                <div>
                  <span className="text-xs uppercase text-gray-400">Tipo</span>
                  <p className="font-semibold text-gray-900">
                    {indicatorTypes.find((type) => type.value === selectedRecord.indicator_type)
                      ?.label ?? "-"}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">Año</span>
                  <p className="font-semibold text-gray-900">{selectedRecord.year ?? "-"}</p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">ID Política</span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.policy_id ?? "-"}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">ID Objetivo</span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.objective_id ? (
                      <Link
                        href={`/meiwa-core/meiwa-sgi/planning/environmental-objectives?objectiveId=${encodeURIComponent(
                          selectedRecord.objective_id,
                        )}`}
                        className="text-emerald-700 hover:underline"
                      >
                        {selectedRecord.objective_id}
                      </Link>
                    ) : (
                      "-"
                    )}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">Concepto</span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.concept ?? "-"}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">Elemento</span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.element ?? "-"}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">Indicador</span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.indicator ?? "-"}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">PIC</span>
                  <p className="font-semibold text-gray-900">{selectedRecord.pic ?? "-"}</p>
                </div>
                <div className="sm:col-span-2">
                  <span className="text-xs uppercase text-gray-400">¿Quién actualiza?</span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.info_updater ?? "-"}
                  </p>
                </div>
              </div>
              <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-3">
                <div>
                  <span className="text-xs uppercase text-gray-400">Sumatoria anual</span>
                  <p className="font-semibold text-gray-900">
                    {detailSummary?.sum === null ? "-" : detailSummary?.sum}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">Promedio anual</span>
                  <p className="font-semibold text-gray-900">
                    {detailSummary?.average === null
                      ? "-"
                      : detailSummary?.average.toFixed(2)}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">
                    Objetivo sumatoria anual
                  </span>
                  <p className="font-semibold text-gray-900">
                    {formatNumber(selectedRecord.annual_sum_target)}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">
                    Objetivo promedio anual
                  </span>
                  <p className="font-semibold text-gray-900">
                    {selectedRecord.annual_average_target == null
                      ? "-"
                      : selectedRecord.annual_average_target.toFixed(2)}
                  </p>
                </div>
                <div>
                  <span className="text-xs uppercase text-gray-400">Cumplimiento</span>
                  <p className="font-semibold text-gray-900">
                    {detailTargetMatch === null ? "-" : detailTargetMatch ? "Cumple" : "No cumple"}
                  </p>
                </div>
              </div>
              <div className="grid gap-3 sm:grid-cols-3">
                {monthOptions.map((month) => (
                  <div key={month.key} className="rounded-lg border border-slate-200 p-3">
                    <p className="text-xs uppercase text-gray-400">{month.full}</p>
                    <p className="mt-1 text-base font-semibold text-gray-900">
                      {formatNumber(selectedRecord[month.key])}
                    </p>
                  </div>
                ))}
              </div>
              <div className="rounded-lg border border-emerald-100 bg-emerald-50 p-3">
                <p className="text-xs uppercase text-emerald-600">Actualizado por</p>
                <p className="mt-1 text-sm font-semibold text-emerald-900">
                  {selectedRecord.last_updated_by ?? "-"}
                </p>
                <p className="text-xs text-emerald-700">
                  {selectedRecord.last_updated_at
                    ? new Date(selectedRecord.last_updated_at).toLocaleString("es-MX")
                    : "Sin actualizaciones"}
                </p>
              </div>
            </div>
          ) : (
            <p className="text-sm text-gray-500">Selecciona un indicador para ver el detalle.</p>
          )}
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}