"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Calendar,
  CheckCircle2,
  Eye,
  FileText,
  Filter,
  Pencil,
  Plus,
  Search,
  Trash2,
  XCircle,
} from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

const pagePath = "/meiwa-core/meiwa-sgi/planning/legal-requirements";
const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);
const actionSourceLabel = "Requisitos legales";

const buildActionPlanHref = (item: LegalRequirement) => {
  const params = new URLSearchParams({
    source: actionSourceLabel,
    description: item.title ?? "",
    comments: item.description ?? "",
    actionPlan: item.required_action ?? "",
    owner: item.followup_responsible ?? "",
    targetDate: item.commitment_date ?? "",
  });
  return `/meiwa-core/meiwa-sgi/planning/action-plan?${params.toString()}`;
};

const requirementTypeOptions = [
  { value: "legal", label: "Legal" },
  { value: "other", label: "Otro requisito" },
];

const frequencyOptions = [
  { value: "monthly", label: "Mensual", months: 1 },
  { value: "quarterly", label: "Trimestral", months: 3 },
  { value: "semiannual", label: "Semestral", months: 6 },
  { value: "annual", label: "Anual", months: 12 },
  { value: "eventual", label: "Eventual", months: null },
  { value: "other", label: "Otra", months: null },
];

const complianceOptions = [
  { value: "complies", label: "Cumple", color: "bg-emerald-100 text-emerald-800" },
  { value: "non_complies", label: "No cumple", color: "bg-rose-100 text-rose-800" },
  { value: "not_evaluated", label: "No evaluado", color: "bg-slate-100 text-slate-700" },
];

const actionStatusOptions = [
  { value: "open", label: "Abierta" },
  { value: "in_progress", label: "En proceso" },
  { value: "closed", label: "Cerrada" },
];

const sortOptions = [
  { value: "requirement_id", label: "ID del requisito" },
  { value: "title", label: "Requerimiento" },
  { value: "next_update_date", label: "Siguiente actualización" },
  { value: "last_updated_at", label: "Última actualización" },
];

const emptyFormState = {
  requirement_id: "",
  type: "legal",
  article: "",
  title: "",
  description: "",
  environmental_aspect: "",
  verification_frequency: "monthly",
  compliance_status: "not_evaluated",
  evidence_url: "",
  retention_time: "",
  responsible_area: "",
  followup_responsible: "",
  required_action: "",
  commitment_date: "",
  action_status: "",
  closure_date: "",
  next_update_date: "",
  last_updated_at: "",
};

type LegalRequirement = {
  id: string;
  requirement_id: string;
  type: string;
  article: string;
  title: string;
  description: string;
  environmental_aspect: string | null;
  verification_frequency: string;
  compliance_status: string;
  evidence_url: string | null;
  retention_time: string | null;
  responsible_area: string;
  followup_responsible: string;
  required_action: string | null;
  commitment_date: string | null;
  action_status: string | null;
  closure_date: string | null;
  last_updated_at: string | null;
  next_update_date: string | null;
};

type FormState = typeof emptyFormState;

type FilterState = {
  type: string;
  compliance: string;
  actionStatus: string;
  area: string;
  followup: string;
  aspect: string;
  nextUpdateFrom: string;
  nextUpdateTo: string;
  commitmentFrom: string;
  commitmentTo: string;
};

const defaultFilters: FilterState = {
  type: "all",
  compliance: "all",
  actionStatus: "all",
  area: "all",
  followup: "all",
  aspect: "all",
  nextUpdateFrom: "",
  nextUpdateTo: "",
  commitmentFrom: "",
  commitmentTo: "",
};

const formatDateLabel = (value?: string | null) => {
  if (!value) return "-";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "-";
  return date.toLocaleDateString("es-MX");
};

const toDateInputValue = (value: Date) => value.toISOString().split("T")[0];

const addMonths = (date: Date, months: number) => {
  const updated = new Date(date);
  updated.setMonth(updated.getMonth() + months);
  return updated;
};

const addDays = (date: Date, days: number) => {
  const updated = new Date(date);
  updated.setDate(updated.getDate() + days);
  return updated;
};

const parseRetentionTime = (value: string) => {
  if (!value.trim()) return null;
  const normalized = value.toLowerCase().trim();
  const match = normalized.match(/(\d+)\s*(\w+)?/);
  if (!match) return null;
  const amount = Number.parseInt(match[1], 10);
  if (Number.isNaN(amount) || amount <= 0) return null;

  const unit = match[2] ?? "";
  if (unit.startsWith("año") || unit.startsWith("ano")) {
    return { amount, unit: "years" as const };
  }
  if (unit.startsWith("mes")) {
    return { amount, unit: "months" as const };
  }
  if (unit.startsWith("dia") || unit.startsWith("dí") || unit.startsWith("d")) {
    return { amount, unit: "days" as const };
  }

  return { amount, unit: "years" as const };
};

const toDateInputValueFromString = (value?: string | null) => {
  if (!value) return "";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "";
  return toDateInputValue(date);
};

const getNextRequirementId = (requirements: LegalRequirement[]) => {
  const maxId = requirements.reduce((max, item) => {
    const match = item.requirement_id.match(/(\d+)(?!.*\d)/);
    if (!match) return max;
    const value = Number.parseInt(match[1], 10);
    return Number.isNaN(value) ? max : Math.max(max, value);
  }, 0);

  const nextValue = maxId + 1;
  return `REQ-${String(nextValue).padStart(3, "0")}`;
};

const getFrequencyConfig = (frequency: string) =>
  frequencyOptions.find((option) => option.value === frequency);

const isManualFrequency = (frequency: string) => {
  const config = getFrequencyConfig(frequency);
  return !config || config.months === null;
};

const getNextUpdateDate = (frequency: string, baseDate: Date) => {
  const config = getFrequencyConfig(frequency);
  if (!config || config.months === null) return "";
  return toDateInputValue(addMonths(baseDate, config.months));
};

const isValidUrl = (value: string) => {
  if (!value) return true;
  try {
    new URL(value);
    return true;
  } catch {
    return false;
  }
};

const resolveBadge = (status: string) =>
  complianceOptions.find((option) => option.value === status) ?? complianceOptions[2];

const resolveActionLabel = (status?: string | null) =>
  actionStatusOptions.find((option) => option.value === status)?.label ?? "-";

const normalize = (value: string) => value.toLowerCase().trim();

const startOfDay = (date: Date) => new Date(date.getFullYear(), date.getMonth(), date.getDate());

const daysUntil = (date: Date, base: Date) =>
  Math.ceil((date.getTime() - base.getTime()) / (1000 * 60 * 60 * 24));

export default function LegalRequirementsPage() {
  const router = useRouter();
  const [requirements, setRequirements] = useState<LegalRequirement[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [detailOpen, setDetailOpen] = useState(false);
  const [editingRequirement, setEditingRequirement] = useState<LegalRequirement | null>(null);
  const [detailRequirement, setDetailRequirement] = useState<LegalRequirement | null>(null);
  const [formState, setFormState] = useState<FormState>(emptyFormState);
  const [formErrors, setFormErrors] = useState<Record<string, string>>({});
  const [deleteTarget, setDeleteTarget] = useState<LegalRequirement | null>(null);
  const [search, setSearch] = useState("");
  const [filters, setFilters] = useState<FilterState>(defaultFilters);
  const [sortKey, setSortKey] = useState("last_updated_at");
  const [sortDirection, setSortDirection] = useState<"asc" | "desc">("desc");
  const [canManage, setCanManage] = useState(false);

  const fetchRequirements = useCallback(async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("legal_requirements")
      .select("*")
      .order("last_updated_at", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar los requisitos legales.");
      console.error(error);
      setLoading(false);
      return;
    }

    setRequirements((data as LegalRequirement[]) ?? []);
    setLoading(false);
  }, []);

  const resolvePermissions = useCallback(
    async (currentUser: User | null) => {
      if (!currentUser) {
        setCanManage(false);
        return;
      }
      if (allowedManagers.has(currentUser.email ?? "")) {
        setCanManage(true);
        return;
      }
      const { data: pageData, error: pageError } = await supabase
        .from("sgi_pages")
        .select("id")
        .eq("path", pagePath)
        .maybeSingle();

      if (pageError) {
        console.error(pageError);
        setCanManage(false);
        return;
      }

      if (!pageData?.id) {
        setCanManage(true);
        return;
      }

      const { data: permissionData, error: permissionError } = await supabase
        .from("sgi_admin_permissions")
        .select("id")
        .eq("page_id", pageData.id)
        .eq("user_id", currentUser.id)
        .maybeSingle();

      if (permissionError) {
        console.error(permissionError);
        setCanManage(false);
        return;
      }

      setCanManage(Boolean(permissionData?.id));
    },
    []
  );

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      if (!data.session?.user) {
        router.push("/");
        return;
      }
      void resolvePermissions(data.session.user);
    };

    void loadSession();
  }, [router, resolvePermissions]);

  useEffect(() => {
    void fetchRequirements();
  }, [fetchRequirements]);

  const filterOptions = useMemo(() => {
    const unique = (items: Array<string | null | undefined>) =>
      Array.from(new Set(items.filter(Boolean) as string[])).sort();

    return {
      areas: unique(requirements.map((item) => item.responsible_area)),
      followups: unique(requirements.map((item) => item.followup_responsible)),
      aspects: unique(requirements.map((item) => item.environmental_aspect)),
    };
  }, [requirements]);

  const filteredRequirements = useMemo(() => {
    const today = startOfDay(new Date());
    const nextUpdateFrom = filters.nextUpdateFrom
      ? startOfDay(new Date(filters.nextUpdateFrom))
      : null;
    const nextUpdateTo = filters.nextUpdateTo
      ? startOfDay(new Date(filters.nextUpdateTo))
      : null;
    const commitmentFrom = filters.commitmentFrom
      ? startOfDay(new Date(filters.commitmentFrom))
      : null;
    const commitmentTo = filters.commitmentTo
      ? startOfDay(new Date(filters.commitmentTo))
      : null;

    return requirements
      .filter((item) => {
        if (filters.type !== "all" && item.type !== filters.type) return false;
        if (filters.compliance !== "all" && item.compliance_status !== filters.compliance)
          return false;
        if (filters.actionStatus !== "all" && item.action_status !== filters.actionStatus)
          return false;
        if (filters.area !== "all" && item.responsible_area !== filters.area) return false;
        if (filters.followup !== "all" && item.followup_responsible !== filters.followup)
          return false;
        if (filters.aspect !== "all" && item.environmental_aspect !== filters.aspect)
          return false;

        if (nextUpdateFrom || nextUpdateTo) {
          if (!item.next_update_date) return false;
          const nextUpdate = startOfDay(new Date(item.next_update_date));
          if (nextUpdateFrom && nextUpdate < nextUpdateFrom) return false;
          if (nextUpdateTo && nextUpdate > nextUpdateTo) return false;
        }

        if (commitmentFrom || commitmentTo) {
          if (!item.commitment_date) return false;
          const commitment = startOfDay(new Date(item.commitment_date));
          if (commitmentFrom && commitment < commitmentFrom) return false;
          if (commitmentTo && commitment > commitmentTo) return false;
        }

        if (!search) return true;
        const query = normalize(search);
        return [
          item.requirement_id,
          item.title,
          item.article,
          item.responsible_area,
          item.followup_responsible,
          item.environmental_aspect ?? "",
        ]
          .join(" ")
          .toLowerCase()
          .includes(query);
      })
      .sort((a, b) => {
        const direction = sortDirection === "asc" ? 1 : -1;
        const aValue = (a as Record<string, string | null>)[sortKey];
        const bValue = (b as Record<string, string | null>)[sortKey];

        if (aValue === null || aValue === undefined) return 1 * direction;
        if (bValue === null || bValue === undefined) return -1 * direction;

        if (sortKey.includes("date") || sortKey.includes("updated")) {
          const aDate = new Date(aValue);
          const bDate = new Date(bValue);
          return (aDate.getTime() - bDate.getTime()) * direction;
        }

        return aValue.localeCompare(bValue) * direction;
      });
  }, [requirements, filters, search, sortKey, sortDirection]);

  const kpis = useMemo(() => {
    const today = startOfDay(new Date());
    const total = filteredRequirements.length;
    const complianceCounts = {
      complies: filteredRequirements.filter((item) => item.compliance_status === "complies").length,
      nonComplies: filteredRequirements.filter((item) => item.compliance_status === "non_complies")
        .length,
      notEvaluated: filteredRequirements.filter((item) => item.compliance_status === "not_evaluated")
        .length,
    };

    const actionCounts = {
      open: filteredRequirements.filter((item) => item.action_status === "open").length,
      inProgress: filteredRequirements.filter((item) => item.action_status === "in_progress").length,
      closed: filteredRequirements.filter((item) => item.action_status === "closed").length,
      overdue: filteredRequirements.filter((item) => {
        if (item.compliance_status !== "non_complies") return false;
        if (!item.commitment_date) return false;
        const commitment = startOfDay(new Date(item.commitment_date));
        return commitment < today && item.action_status !== "closed";
      }).length,
    };

    const updateStatus = {
      overdue: filteredRequirements.filter((item) => {
        if (!item.next_update_date) return false;
        const nextUpdate = startOfDay(new Date(item.next_update_date));
        return nextUpdate < today;
      }).length,
      upcoming: filteredRequirements.filter((item) => {
        if (!item.next_update_date) return false;
        const nextUpdate = startOfDay(new Date(item.next_update_date));
        const days = daysUntil(nextUpdate, today);
        return days >= 0 && days <= 15;
      }).length,
      latest: filteredRequirements
        .map((item) => item.last_updated_at)
        .filter(Boolean)
        .sort((a, b) => new Date(b as string).getTime() - new Date(a as string).getTime())[0],
    };

    const percent = (value: number) => (total ? Math.round((value / total) * 100) : 0);

    return {
      total,
      complianceCounts,
      actionCounts,
      updateStatus,
      percent,
    };
  }, [filteredRequirements]);

  const resetForm = () => {
    setFormState(emptyFormState);
    setFormErrors({});
    setEditingRequirement(null);
  };

  const openCreateDialog = () => {
    resetForm();
    setFormState((prev) => ({
      ...prev,
      requirement_id: getNextRequirementId(requirements),
      last_updated_at: toDateInputValue(new Date()),
    }));
    setDialogOpen(true);
  };

  const openEditDialog = (item: LegalRequirement) => {
    setEditingRequirement(item);
    setFormState({
      requirement_id: item.requirement_id,
      type: item.type,
      article: item.article,
      title: item.title,
      description: item.description,
      environmental_aspect: item.environmental_aspect ?? "",
      verification_frequency: item.verification_frequency,
      compliance_status: item.compliance_status,
      evidence_url: item.evidence_url ?? "",
      retention_time: item.retention_time ?? "",
      responsible_area: item.responsible_area,
      followup_responsible: item.followup_responsible,
      required_action: item.required_action ?? "",
      commitment_date: item.commitment_date ?? "",
      action_status: item.action_status ?? "",
      closure_date: item.closure_date ?? "",
      next_update_date: item.next_update_date ?? "",
      last_updated_at: toDateInputValueFromString(item.last_updated_at),
    });
    setFormErrors({});
    setDialogOpen(true);
  };

  const openDetailDialog = (item: LegalRequirement) => {
    setDetailRequirement(item);
    setDetailOpen(true);
  };

  const validateForm = () => {
    const errors: Record<string, string> = {};

    if (!formState.requirement_id.trim()) errors.requirement_id = "El ID es obligatorio.";
    if (!formState.type) errors.type = "El tipo es obligatorio.";
    if (!formState.article.trim()) errors.article = "El artículo/numeral es obligatorio.";
    if (!formState.title.trim()) errors.title = "El requerimiento es obligatorio.";
    if (!formState.verification_frequency)
      errors.verification_frequency = "La frecuencia es obligatoria.";
    if (!formState.compliance_status) errors.compliance_status = "El estatus es obligatorio.";
    if (!formState.responsible_area.trim())
      errors.responsible_area = "El área responsable es obligatoria.";
    if (!formState.followup_responsible.trim())
      errors.followup_responsible = "El responsable es obligatorio.";
    if (!formState.last_updated_at)
      errors.last_updated_at = "La última actualización es obligatoria.";

    if (formState.evidence_url && !isValidUrl(formState.evidence_url)) {
      errors.evidence_url = "Ingresa una URL válida.";
    }

    if (formState.compliance_status === "non_complies") {
      if (!formState.required_action.trim())
        errors.required_action = "La acción requerida es obligatoria.";
      if (!formState.commitment_date)
        errors.commitment_date = "La fecha compromiso es obligatoria.";
      if (!formState.action_status)
        errors.action_status = "El estatus de acción es obligatorio.";

      if (formState.action_status === "closed" && !formState.closure_date) {
        errors.closure_date = "La fecha de cierre es obligatoria.";
      }
    }

    setFormErrors(errors);
    return Object.keys(errors).length === 0;
  };

  const handleSave = async () => {
    if (!validateForm()) return;

    const now = new Date();
    const lastUpdatedBase = formState.last_updated_at
      ? new Date(`${formState.last_updated_at}T00:00:00`)
      : now;
    const lastUpdated = Number.isNaN(lastUpdatedBase.getTime())
      ? now.toISOString()
      : lastUpdatedBase.toISOString();
    const retention = parseRetentionTime(formState.retention_time);
    const nextUpdateDate = retention
      ? toDateInputValue(
          retention.unit === "days"
            ? addDays(lastUpdatedBase, retention.amount)
            : addMonths(
                lastUpdatedBase,
                retention.unit === "years" ? retention.amount * 12 : retention.amount
              )
        )
      : "";

    const payload = {
      requirement_id: formState.requirement_id.trim(),
      type: formState.type,
      article: formState.article.trim(),
      title: formState.title.trim(),
      description: formState.description.trim(),
      environmental_aspect: formState.environmental_aspect.trim() || null,
      verification_frequency: formState.verification_frequency,
      compliance_status: formState.compliance_status,
      evidence_url: formState.evidence_url.trim() || null,
      retention_time: formState.retention_time.trim() || null,
      responsible_area: formState.responsible_area.trim(),
      followup_responsible: formState.followup_responsible.trim(),
      required_action:
        formState.compliance_status === "non_complies" && formState.required_action.trim()
          ? formState.required_action.trim()
          : null,
      commitment_date:
        formState.compliance_status === "non_complies" && formState.commitment_date
          ? formState.commitment_date
          : null,
      action_status:
        formState.compliance_status === "non_complies" && formState.action_status
          ? formState.action_status
          : null,
      closure_date:
        formState.compliance_status === "non_complies" && formState.action_status === "closed"
          ? formState.closure_date
          : null,
      last_updated_at: lastUpdated,
      next_update_date: nextUpdateDate || null,
    };

    if (editingRequirement) {
      const { error } = await supabase
        .from("legal_requirements")
        .update(payload)
        .eq("id", editingRequirement.id);

      if (error) {
        toast.error("No se pudo actualizar el requisito.");
        console.error(error);
        return;
      }

      toast.success("Requisito actualizado.");
    } else {
      const { error } = await supabase.from("legal_requirements").insert(payload);
      if (error) {
        toast.error("No se pudo crear el requisito.");
        console.error(error);
        return;
      }

      toast.success("Requisito creado.");
    }

    setDialogOpen(false);
    resetForm();
    await fetchRequirements();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase.from("legal_requirements").delete().eq("id", deleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el requisito.");
      console.error(error);
      return;
    }

    toast.success("Requisito eliminado.");
    setDeleteTarget(null);
    await fetchRequirements();
  };

  const updateFormField = (field: keyof FormState, value: string) => {
    setFormState((prev) => ({
      ...prev,
      [field]: value,
    }));
  };

  const computedNextUpdateDate = useMemo(() => {
    const retention = parseRetentionTime(formState.retention_time);
    if (!retention || !formState.last_updated_at) {
      return formState.next_update_date;
    }
    const baseDate = new Date(`${formState.last_updated_at}T00:00:00`);
    if (Number.isNaN(baseDate.getTime())) return formState.next_update_date;
    const nextDate =
      retention.unit === "days"
        ? addDays(baseDate, retention.amount)
        : addMonths(
            baseDate,
            retention.unit === "years" ? retention.amount * 12 : retention.amount
          );
    return toDateInputValue(nextDate);
  }, [formState.last_updated_at, formState.next_update_date, formState.retention_time]);

  const getComplianceStatusMeta = (item: LegalRequirement) => {
    const today = startOfDay(new Date());
    const nextUpdate = item.next_update_date ? startOfDay(new Date(item.next_update_date)) : null;
    const commitmentDate = item.commitment_date ? startOfDay(new Date(item.commitment_date)) : null;

    const overdueUpdate = Boolean(nextUpdate && nextUpdate < today);
    const upcomingUpdate = Boolean(
      nextUpdate && daysUntil(nextUpdate, today) >= 0 && daysUntil(nextUpdate, today) <= 15
    );
    const overdueAction = Boolean(
      item.compliance_status === "non_complies" &&
        commitmentDate &&
        commitmentDate < today &&
        item.action_status !== "closed"
    );

    return { overdueUpdate, upcomingUpdate, overdueAction };
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-gradient-to-br from-slate-50 via-white to-emerald-50 px-4 py-6 sm:px-8">
        <section className="rounded-3xl border border-emerald-100 bg-white px-6 py-8 shadow-sm">
          <div className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div className="space-y-3">
              <p className="text-xs font-semibold uppercase tracking-[0.45em] text-emerald-600">
                Planeación SGI
              </p>
              <h1 className="text-3xl font-bold text-gray-900">Matriz de requisitos legales</h1>
              <p className="max-w-3xl text-sm text-gray-600 sm:text-base">
                Administra los requisitos ISO 14001 y el cumplimiento legal. Actualiza el estatus,
                gestiona acciones correctivas y asegura el seguimiento continuo.
              </p>
            </div>
            <div className="flex flex-wrap gap-3">
              <Button
                onClick={openCreateDialog}
                className="bg-emerald-600 text-white hover:bg-emerald-700"
                disabled={!canManage}
              >
                <Plus className="mr-2 h-4 w-4" />
                Nuevo requisito
              </Button>
              {!canManage && (
                <Badge variant="outline" className="border-amber-200 text-amber-600">
                  Modo lectura
                </Badge>
              )}
            </div>
          </div>
        </section>

        <section className="grid gap-4 lg:grid-cols-3">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-base text-gray-700">Cumplimiento</CardTitle>
              <CheckCircle2 className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent className="space-y-2">
              <p className="text-2xl font-semibold text-gray-900">
                {kpis.complianceCounts.complies} / {kpis.total}
              </p>
              <p className="text-sm text-gray-500">
                {kpis.percent(kpis.complianceCounts.complies)}% cumple
              </p>
              <div className="flex flex-wrap gap-2 text-xs text-gray-500">
                <span>No cumple: {kpis.complianceCounts.nonComplies}</span>
                <span>No evaluado: {kpis.complianceCounts.notEvaluated}</span>
              </div>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-base text-gray-700">Acciones</CardTitle>
              <XCircle className="h-4 w-4 text-rose-500" />
            </CardHeader>
            <CardContent className="space-y-2">
              <p className="text-2xl font-semibold text-gray-900">
                {kpis.actionCounts.open + kpis.actionCounts.inProgress} activas
              </p>
              <div className="flex flex-wrap gap-2 text-xs text-gray-500">
                <span>Abiertas: {kpis.actionCounts.open}</span>
                <span>En proceso: {kpis.actionCounts.inProgress}</span>
                <span>Cerradas: {kpis.actionCounts.closed}</span>
                <span>Vencidas: {kpis.actionCounts.overdue}</span>
              </div>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-base text-gray-700">Seguimiento</CardTitle>
              <Calendar className="h-4 w-4 text-sky-500" />
            </CardHeader>
            <CardContent className="space-y-2">
              <p className="text-2xl font-semibold text-gray-900">{kpis.updateStatus.overdue}</p>
              <p className="text-sm text-gray-500">Requisitos vencidos</p>
              <div className="flex flex-wrap gap-2 text-xs text-gray-500">
                <span>Próximos 15 días: {kpis.updateStatus.upcoming}</span>
                <span>
                  Última actualización: {formatDateLabel(kpis.updateStatus.latest ?? "")}
                </span>
              </div>
            </CardContent>
          </Card>
        </section>

        <section className="space-y-4 rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <div className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div className="flex flex-1 items-center gap-2 rounded-full border border-emerald-100 bg-emerald-50/40 px-4 py-2 text-sm text-gray-600">
              <Search className="h-4 w-4 text-emerald-500" />
              <Input
                value={search}
                onChange={(event) => setSearch(event.target.value)}
                placeholder="Buscar por ID, requerimiento, artículo, área, responsable o aspecto"
                className="border-0 bg-transparent focus-visible:ring-0"
              />
            </div>
            <div className="flex flex-wrap items-center gap-2">
              <Filter className="h-4 w-4 text-emerald-500" />
              <Select value={sortKey} onValueChange={setSortKey}>
                <SelectTrigger className="w-[200px]">
                  <SelectValue placeholder="Ordenar por" />
                </SelectTrigger>
                <SelectContent>
                  {sortOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={sortDirection} onValueChange={(value) => setSortDirection(value as "asc" | "desc")}>
                <SelectTrigger className="w-[140px]">
                  <SelectValue placeholder="Orden" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="asc">Ascendente</SelectItem>
                  <SelectItem value="desc">Descendente</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-3">
            <Select
              value={filters.type}
              onValueChange={(value) => setFilters((prev) => ({ ...prev, type: value }))}
            >
              <SelectTrigger>
                <SelectValue placeholder="Tipo" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todos los tipos</SelectItem>
                {requirementTypeOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select
              value={filters.compliance}
              onValueChange={(value) => setFilters((prev) => ({ ...prev, compliance: value }))}
            >
              <SelectTrigger>
                <SelectValue placeholder="Estatus" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todos los estatus</SelectItem>
                {complianceOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select
              value={filters.actionStatus}
              onValueChange={(value) => setFilters((prev) => ({ ...prev, actionStatus: value }))}
            >
              <SelectTrigger>
                <SelectValue placeholder="Estatus de acción" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todas las acciones</SelectItem>
                {actionStatusOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select
              value={filters.area}
              onValueChange={(value) => setFilters((prev) => ({ ...prev, area: value }))}
            >
              <SelectTrigger>
                <SelectValue placeholder="Área responsable" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todas las áreas</SelectItem>
                {filterOptions.areas.map((option) => (
                  <SelectItem key={option} value={option}>
                    {option}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select
              value={filters.followup}
              onValueChange={(value) => setFilters((prev) => ({ ...prev, followup: value }))}
            >
              <SelectTrigger>
                <SelectValue placeholder="Responsable de seguimiento" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todos los responsables</SelectItem>
                {filterOptions.followups.map((option) => (
                  <SelectItem key={option} value={option}>
                    {option}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select
              value={filters.aspect}
              onValueChange={(value) => setFilters((prev) => ({ ...prev, aspect: value }))}
            >
              <SelectTrigger>
                <SelectValue placeholder="Aspecto ambiental" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todos los aspectos</SelectItem>
                {filterOptions.aspects.map((option) => (
                  <SelectItem key={option} value={option}>
                    {option}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <div className="grid gap-2 rounded-2xl border border-emerald-100 p-3">
              <p className="text-xs font-semibold uppercase text-emerald-600">
                Rango siguiente actualización
              </p>
              <div className="grid gap-2 sm:grid-cols-2">
                <Input
                  type="date"
                  value={filters.nextUpdateFrom}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, nextUpdateFrom: event.target.value }))
                  }
                />
                <Input
                  type="date"
                  value={filters.nextUpdateTo}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, nextUpdateTo: event.target.value }))
                  }
                />
              </div>
            </div>

            <div className="grid gap-2 rounded-2xl border border-emerald-100 p-3">
              <p className="text-xs font-semibold uppercase text-emerald-600">
                Rango fecha compromiso
              </p>
              <div className="grid gap-2 sm:grid-cols-2">
                <Input
                  type="date"
                  value={filters.commitmentFrom}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, commitmentFrom: event.target.value }))
                  }
                />
                <Input
                  type="date"
                  value={filters.commitmentTo}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, commitmentTo: event.target.value }))
                  }
                />
              </div>
            </div>
          </div>

          <div className="flex flex-wrap items-center justify-between gap-3 text-sm text-gray-500">
            <span>Mostrando {filteredRequirements.length} requisitos</span>
            <Button
              variant="ghost"
              onClick={() => setFilters(defaultFilters)}
              className="text-emerald-600 hover:text-emerald-700"
            >
              Limpiar filtros
            </Button>
          </div>
        </section>

        <section className="rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          {loading ? (
            <p className="text-sm text-gray-500">Cargando requisitos...</p>
          ) : (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>ID</TableHead>
                    <TableHead>Tipo</TableHead>
                    <TableHead>Artículo</TableHead>
                    <TableHead>Requerimiento</TableHead>
                    <TableHead>Aspecto ambiental</TableHead>
                    <TableHead>Frecuencia</TableHead>
                    <TableHead>Estatus</TableHead>
                    <TableHead>Área responsable</TableHead>
                    <TableHead>Responsable</TableHead>
                    <TableHead>Siguiente actualización</TableHead>
                    <TableHead>Acción requerida</TableHead>
                    <TableHead>Fecha compromiso</TableHead>
                    <TableHead>Crear acción</TableHead>                    
                    <TableHead>Acciones</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredRequirements.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={14} className="text-center text-sm text-gray-500">
                        No hay requisitos con los filtros seleccionados.
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredRequirements.map((item) => {
                      const statusBadge = resolveBadge(item.compliance_status);
                      const { overdueUpdate, upcomingUpdate, overdueAction } =
                        getComplianceStatusMeta(item);

                      return (
                        <TableRow key={item.id}>
                          <TableCell className="font-medium text-gray-900">
                            {item.requirement_id}
                          </TableCell>
                          <TableCell>
                            {
                              requirementTypeOptions.find((option) => option.value === item.type)
                                ?.label
                            }
                          </TableCell>
                          <TableCell>{item.article}</TableCell>
                          <TableCell className="max-w-[220px] truncate">{item.title}</TableCell>
                          <TableCell>{item.environmental_aspect ?? "-"}</TableCell>
                          <TableCell>
                            {
                              frequencyOptions.find(
                                (option) => option.value === item.verification_frequency
                              )?.label
                            }
                          </TableCell>
                          <TableCell>
                            <div className="flex flex-col gap-1">
                              <Badge className={`${statusBadge.color} w-fit`}>{statusBadge.label}</Badge>
                              {overdueUpdate && (
                                <Badge variant="outline" className="border-rose-200 text-rose-600">
                                  Vencido
                                </Badge>
                              )}
                              {!overdueUpdate && upcomingUpdate && (
                                <Badge variant="outline" className="border-amber-200 text-amber-600">
                                  Próximo a vencer
                                </Badge>
                              )}
                              {overdueAction && (
                                <Badge variant="outline" className="border-rose-200 text-rose-600">
                                  Acción vencida
                                </Badge>
                              )}
                            </div>
                          </TableCell>
                          <TableCell>{item.responsible_area}</TableCell>
                          <TableCell>{item.followup_responsible}</TableCell>
                          <TableCell>{formatDateLabel(item.next_update_date)}</TableCell>
                          <TableCell className="max-w-[220px] truncate">
                            {item.required_action ?? "-"}
                          </TableCell>
                          <TableCell>{formatDateLabel(item.commitment_date)}</TableCell>
                          <TableCell>
                            <Button
                              asChild
                              size="sm"
                              className="bg-sky-500 text-white hover:bg-sky-600"
                            >
                              <Link href={buildActionPlanHref(item)}>
                                Crear acción
                              </Link>
                            </Button>
                          </TableCell>                          
                          <TableCell>
                            <div className="flex flex-wrap gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => openDetailDialog(item)}
                              >
                                <Eye className="mr-1 h-4 w-4" /> Ver
                              </Button>
                              {item.evidence_url && (
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => window.open(item.evidence_url ?? "", "_blank")}
                                >
                                  <FileText className="mr-1 h-4 w-4" /> Evidencia
                                </Button>
                              )}
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => openEditDialog(item)}
                                disabled={!canManage}
                              >
                                <Pencil className="mr-1 h-4 w-4" /> Editar
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => setDeleteTarget(item)}
                                disabled={!canManage}
                              >
                                <Trash2 className="mr-1 h-4 w-4" /> Eliminar
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      );
                    })
                  )}
                </TableBody>
              </Table>
            </div>
          )}
        </section>

        <Dialog open={dialogOpen} onOpenChange={(open) => (open ? setDialogOpen(open) : setDialogOpen(false))}>
          <DialogContent className="max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>
                {editingRequirement ? "Editar requisito" : "Nuevo requisito"}
              </DialogTitle>
            </DialogHeader>
            <div className="grid gap-4">
              <div className="grid gap-2">
                <Label>ID del requisito *</Label>
                <Input
                  value={formState.requirement_id}
                  placeholder="REQ-001"
                  disabled
                />
                <p className="text-xs text-gray-500">
                  Se genera automáticamente con la secuencia de requisitos.
                </p>
                {formErrors.requirement_id && (
                  <p className="text-xs text-rose-600">{formErrors.requirement_id}</p>
                )}
              </div>

              <div className="grid gap-2 md:grid-cols-2">
                <div className="grid gap-2">
                  <Label>Tipo *</Label>
                  <Select
                    value={formState.type}
                    onValueChange={(value) => updateFormField("type", value)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Selecciona" />
                    </SelectTrigger>
                    <SelectContent>
                      {requirementTypeOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {formErrors.type && <p className="text-xs text-rose-600">{formErrors.type}</p>}
                </div>
                <div className="grid gap-2">
                  <Label>Artículo / numeral *</Label>
                  <Input
                    value={formState.article}
                    onChange={(event) => updateFormField("article", event.target.value)}
                    placeholder="Ej. NOM-001"
                  />
                  {formErrors.article && (
                    <p className="text-xs text-rose-600">{formErrors.article}</p>
                  )}
                </div>
              </div>

              <div className="grid gap-2">
                <Label>Requerimiento *</Label>
                <Input
                  value={formState.title}
                  onChange={(event) => updateFormField("title", event.target.value)}
                  placeholder="Nombre del requisito"
                />
                {formErrors.title && <p className="text-xs text-rose-600">{formErrors.title}</p>}
              </div>

              <div className="grid gap-2">
                <Label>Descripción</Label>
                <Textarea
                  value={formState.description}
                  onChange={(event) => updateFormField("description", event.target.value)}
                  placeholder="Descripción detallada del requisito"
                />
              </div>

              <div className="grid gap-2 md:grid-cols-2">
                <div className="grid gap-2">
                  <Label>Aspecto ambiental asociado</Label>
                  <Input
                    value={formState.environmental_aspect}
                    onChange={(event) =>
                      updateFormField("environmental_aspect", event.target.value)
                    }
                  />
                </div>
                <div className="grid gap-2">
                  <Label>Frecuencia de verificación *</Label>
                  <Select
                    value={formState.verification_frequency}
                    onValueChange={(value) => updateFormField("verification_frequency", value)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Selecciona" />
                    </SelectTrigger>
                    <SelectContent>
                      {frequencyOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {formErrors.verification_frequency && (
                    <p className="text-xs text-rose-600">{formErrors.verification_frequency}</p>
                  )}
                </div>
              </div>

              <div className="grid gap-2 md:grid-cols-2">
                <div className="grid gap-2">
                  <Label>Estatus *</Label>
                  <Select
                    value={formState.compliance_status}
                    onValueChange={(value) => updateFormField("compliance_status", value)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Selecciona" />
                    </SelectTrigger>
                    <SelectContent>
                      {complianceOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {formErrors.compliance_status && (
                    <p className="text-xs text-rose-600">{formErrors.compliance_status}</p>
                  )}
                </div>
                <div className="grid gap-2">
                  <Label>Evidencia (URL)</Label>
                  <Input
                    value={formState.evidence_url}
                    onChange={(event) => updateFormField("evidence_url", event.target.value)}
                    placeholder="https://"
                  />
                  {formErrors.evidence_url && (
                    <p className="text-xs text-rose-600">{formErrors.evidence_url}</p>
                  )}
                </div>
              </div>

              <div className="grid gap-2 md:grid-cols-2">
                <div className="grid gap-2">
                  <Label>Tiempo de retención (años)</Label>
                  <Input
                    value={formState.retention_time}
                    onChange={(event) => updateFormField("retention_time", event.target.value)}
                    placeholder="Ej. 5 años"
                  />
                </div>
                <div className="grid gap-2">
                  <Label>Área responsable *</Label>
                  <Input
                    value={formState.responsible_area}
                    onChange={(event) => updateFormField("responsible_area", event.target.value)}
                  />
                  {formErrors.responsible_area && (
                    <p className="text-xs text-rose-600">{formErrors.responsible_area}</p>
                  )}
                </div>
              </div>

              <div className="grid gap-2">
                <Label>Responsable de seguimiento *</Label>
                <Input
                  value={formState.followup_responsible}
                  onChange={(event) => updateFormField("followup_responsible", event.target.value)}
                />
                {formErrors.followup_responsible && (
                  <p className="text-xs text-rose-600">{formErrors.followup_responsible}</p>
                )}
              </div>

              {formState.compliance_status === "non_complies" && (
                <div className="rounded-2xl border border-rose-100 bg-rose-50/40 p-4">
                  <h3 className="text-sm font-semibold text-rose-700">
                    Gestión de acción correctiva
                  </h3>
                  <div className="mt-4 grid gap-4">
                    <div className="grid gap-2">
                      <Label>Acción requerida *</Label>
                      <Textarea
                        value={formState.required_action}
                        onChange={(event) => updateFormField("required_action", event.target.value)}
                      />
                      {formErrors.required_action && (
                        <p className="text-xs text-rose-600">{formErrors.required_action}</p>
                      )}
                    </div>
                    <div className="grid gap-2 md:grid-cols-2">
                      <div className="grid gap-2">
                        <Label>Fecha compromiso *</Label>
                        <Input
                          type="date"
                          value={formState.commitment_date}
                          onChange={(event) =>
                            updateFormField("commitment_date", event.target.value)
                          }
                        />
                        {formErrors.commitment_date && (
                          <p className="text-xs text-rose-600">{formErrors.commitment_date}</p>
                        )}
                      </div>
                      <div className="grid gap-2">
                        <Label>Estatus de acción *</Label>
                        <Select
                          value={formState.action_status}
                          onValueChange={(value) => updateFormField("action_status", value)}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Selecciona" />
                          </SelectTrigger>
                          <SelectContent>
                            {actionStatusOptions.map((option) => (
                              <SelectItem key={option.value} value={option.value}>
                                {option.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {formErrors.action_status && (
                          <p className="text-xs text-rose-600">{formErrors.action_status}</p>
                        )}
                      </div>
                    </div>

                    {formState.action_status === "closed" && (
                      <div className="grid gap-2">
                        <Label>Fecha de cierre *</Label>
                        <Input
                          type="date"
                          value={formState.closure_date}
                          onChange={(event) => updateFormField("closure_date", event.target.value)}
                        />
                        {formErrors.closure_date && (
                          <p className="text-xs text-rose-600">{formErrors.closure_date}</p>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              )}

              <div className="grid gap-2 md:grid-cols-2">
                <div className="grid gap-2">
                  <Label>Siguiente fecha de actualización</Label>
                  <Input
                    type="date"
                    value={computedNextUpdateDate}
                    onChange={(event) => updateFormField("next_update_date", event.target.value)}
                    disabled
                  />
                  <p className="text-xs text-gray-500">
                    Se calcula automáticamente según el tiempo de retención y la última
                    actualización.
                  </p>
                </div>
                <div className="grid gap-2">
                  <Label>Última actualización</Label>
                  <Input
                    type="date"
                    value={formState.last_updated_at}
                    onChange={(event) => updateFormField("last_updated_at", event.target.value)}
                  />
                  {formErrors.last_updated_at && (
                    <p className="text-xs text-rose-600">{formErrors.last_updated_at}</p>
                  )}
                </div>
              </div>
            </div>

            <div className="flex flex-wrap justify-end gap-2">
              <Button variant="outline" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button onClick={handleSave} disabled={!canManage}>
                {editingRequirement ? "Guardar cambios" : "Crear requisito"}
              </Button>
            </div>
          </DialogContent>
        </Dialog>

        <Dialog open={detailOpen} onOpenChange={setDetailOpen}>
          <DialogContent className="max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Detalle del requisito</DialogTitle>
            </DialogHeader>
            {detailRequirement && (
              <div className="grid gap-4 text-sm text-gray-600">
                <div className="grid gap-2 rounded-2xl border border-emerald-100 p-4">
                  <div className="flex items-center justify-between">
                    <p className="text-xs font-semibold uppercase text-emerald-600">
                      {detailRequirement.requirement_id}
                    </p>
                    <Badge className={resolveBadge(detailRequirement.compliance_status).color}>
                      {resolveBadge(detailRequirement.compliance_status).label}
                    </Badge>
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    {detailRequirement.title}
                  </h3>
                  <p>{detailRequirement.description || "Sin descripción"}</p>
                </div>

                <div className="grid gap-3 md:grid-cols-2">
                  <div>
                    <p className="text-xs uppercase text-gray-400">Tipo</p>
                    <p>
                      {
                        requirementTypeOptions.find(
                          (option) => option.value === detailRequirement.type
                        )?.label
                      }
                    </p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Artículo</p>
                    <p>{detailRequirement.article}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Aspecto ambiental</p>
                    <p>{detailRequirement.environmental_aspect ?? "-"}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Frecuencia</p>
                    <p>
                      {
                        frequencyOptions.find(
                          (option) => option.value === detailRequirement.verification_frequency
                        )?.label
                      }
                    </p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Área responsable</p>
                    <p>{detailRequirement.responsible_area}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Responsable seguimiento</p>
                    <p>{detailRequirement.followup_responsible}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Evidencia</p>
                    <p>{detailRequirement.evidence_url ?? "-"}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Retención</p>
                    <p>{detailRequirement.retention_time ?? "-"}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Siguiente actualización</p>
                    <p>{formatDateLabel(detailRequirement.next_update_date)}</p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-gray-400">Última actualización</p>
                    <p>{formatDateLabel(detailRequirement.last_updated_at)}</p>
                  </div>
                </div>

                {detailRequirement.compliance_status === "non_complies" && (
                  <div className="rounded-2xl border border-rose-100 bg-rose-50/40 p-4">
                    <h4 className="text-sm font-semibold text-rose-700">Acción correctiva</h4>
                    <p className="mt-2">{detailRequirement.required_action ?? "-"}</p>
                    <div className="mt-3 grid gap-2 text-sm">
                      <p>
                        <span className="font-semibold">Fecha compromiso:</span>{" "}
                        {formatDateLabel(detailRequirement.commitment_date)}
                      </p>
                      <p>
                        <span className="font-semibold">Estatus de acción:</span>{" "}
                        {resolveActionLabel(detailRequirement.action_status)}
                      </p>
                      <p>
                        <span className="font-semibold">Fecha de cierre:</span>{" "}
                        {formatDateLabel(detailRequirement.closure_date)}
                      </p>
                    </div>
                  </div>
                )}
              </div>
            )}
          </DialogContent>
        </Dialog>

        <AlertDialog open={Boolean(deleteTarget)} onOpenChange={() => setDeleteTarget(null)}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>¿Eliminar requisito?</AlertDialogTitle>
            </AlertDialogHeader>
            <p className="text-sm text-gray-600">
              Esta acción es irreversible. Se eliminará el requisito y toda su información asociada.
            </p>
            <AlertDialogFooter>
              <AlertDialogCancel onClick={() => setDeleteTarget(null)}>Cancelar</AlertDialogCancel>
              <AlertDialogAction onClick={handleDelete}>Eliminar</AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </div>
    </SGIDashboardLayout>
  );
}