"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import {
  Eye,
  Pencil,
  Plus,
  Search,
  Trash2,
} from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

const pagePath = "/meiwa-core/meiwa-sgi/planning/risk-and-opportunities";
const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);
const actionSourceLabel = "Riesgos y oportunidades";

const buildActionPlanHref = (record: RiskOpportunity) => {
  const params = new URLSearchParams({
    source: actionSourceLabel,
    description: record.title ?? "",
    comments: record.description ?? "",
    rootCause: record.current_controls ?? "",
    actionPlan: record.action_plan ?? "",
    owner: record.action_owner ?? "",
    targetDate: record.commitment_date ?? "",
  });
  return `/meiwa-core/meiwa-sgi/planning/action-plan?${params.toString()}`;
};

const typeOptions = [
  { value: "risk", label: "Riesgo" },
  { value: "opportunity", label: "Oportunidad" },
];

const sourceOptions = [
  { value: "aspect", label: "Aspecto" },
  { value: "legal", label: "Legal" },
  { value: "4_1", label: "4.1" },
  { value: "4_2", label: "4.2" },
  { value: "emergency", label: "Emergencia" },
];

const actionStatusOptions = [
  { value: "open", label: "Abierta" },
  { value: "in_progress", label: "En proceso" },
  { value: "closed", label: "Cerrada" },
];

const classificationOptions = [
  { value: "low", label: "Bajo" },
  { value: "medium", label: "Medio" },
  { value: "high", label: "Alto" },
  { value: "critical", label: "Crítico" },
];

const probabilityScale = [
  { value: 1, label: "Raro", detail: "No ha ocurrido / muy improbable. ≤ 1 vez cada 5 años" },
  { value: 2, label: "Poco probable", detail: "~ 1 vez cada 2–5 años" },
  { value: 3, label: "Ocasional", detail: "~ 1 vez por año" },
  { value: 4, label: "Probable", detail: "~ 1 vez por trimestre a mensual" },
  { value: 5, label: "Frecuente", detail: "Semanal o diario" },
];

const impactScale = [
  { value: 1, label: "Insignificante", detail: "Mínima o nula, contención/limpieza inmediata, sin residuos especiales" },
  { value: 2, label: "Menor", detail: "Localizada/controlable, residuos manejables, sin notificación" },
  { value: 3, label: "Moderado", detail: "Relevante dentro del sitio, correctivas formales, posible RP/RME adicional" },
  { value: 4, label: "Mayor", detail: "Seria, posible salida del área, drenaje/suelo, incumplimiento probable" },
  { value: 5, label: "Crítico", detail: "Daño severo, fuera del sitio, autoridad/sanción alta, incidente mayor" },
];

const emptyFormState = {
  type: "risk",
  title: "",
  description: "",
  source: "aspect",
  environmental_aspect: "",
  linked_to_legal: "no",
  current_controls: "",
  p_initial: "",
  i_initial: "",
  action_plan: "",
  action_owner: "",
  commitment_date: "",
  action_status: "",
  closure_date: "",
  evidence_urls: "",
  p_residual: "",
  i_residual: "",
  next_review_date: "",
};

type RiskOpportunity = {
  id: string;
  type: string;
  title: string;
  description: string | null;
  source: string;
  environmental_aspect: string | null;
  linked_to_legal: boolean | null;
  current_controls: string;
  p_initial: number | null;
  i_initial: number | null;
  action_plan: string | null;
  action_owner: string | null;
  commitment_date: string | null;
  action_status: string | null;
  closure_date: string | null;
  evidence_urls: string | string[] | null;
  p_residual: number | null;
  i_residual: number | null;
  last_updated_at: string | null;
  next_review_date: string | null;
};

type FormState = typeof emptyFormState;

type FilterState = {
  type: string;
  source: string;
  residualClass: string;
  acceptable: string;
  linkedLegal: string;
  actionStatus: string;
  responsible: string;
  commitmentFrom: string;
  commitmentTo: string;
  closureFrom: string;
  closureTo: string;
};

const defaultFilters: FilterState = {
  type: "all",
  source: "all",
  residualClass: "all",
  acceptable: "all",
  linkedLegal: "all",
  actionStatus: "all",
  responsible: "all",
  commitmentFrom: "",
  commitmentTo: "",
  closureFrom: "",
  closureTo: "",
};

const formatDateLabel = (value?: string | null) => {
  if (!value) return "-";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "-";
  return date.toLocaleDateString("es-MX");
};

const parseEvidence = (value: string | string[] | null | undefined) => {
  if (!value) return [];
  if (Array.isArray(value)) return value.filter(Boolean);
  return value
    .split(/\n|,/)
    .map((entry) => entry.trim())
    .filter(Boolean);
};

const serializeEvidence = (value: string) =>
  value
    .split(/\n|,/)
    .map((entry) => entry.trim())
    .filter(Boolean)
    .join("\n");

const getLevel = (p?: number | null, i?: number | null) => {
  if (!p || !i) return null;
  return p * i;
};

const getClassification = (level?: number | null, impact?: number | null) => {
  if (!level) return null;
  let classification = "low";
  if (level >= 20) classification = "critical";
  else if (level >= 13) classification = "high";
  else if (level >= 6) classification = "medium";

  if (impact === 5 && (classification === "low" || classification === "medium")) {
    classification = "high";
  }

  return classification;
};

const classificationLabel = (classification?: string | null) => {
  switch (classification) {
    case "low":
      return "Bajo";
    case "medium":
      return "Medio";
    case "high":
      return "Alto";
    case "critical":
      return "Crítico";
    default:
      return "-";
  }
};

const classificationBadge = (classification?: string | null) => {
  switch (classification) {
    case "low":
      return "bg-emerald-100 text-emerald-800";
    case "medium":
      return "bg-amber-100 text-amber-800";
    case "high":
      return "bg-orange-100 text-orange-800";
    case "critical":
      return "bg-rose-100 text-rose-800";
    default:
      return "bg-slate-100 text-slate-700";
  }
};

const getAcceptable = (level?: number | null) => {
  if (!level) return null;
  return level <= 12;
};

const dateInRange = (value: string | null, from?: string, to?: string) => {
  if (!value) return false;
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return false;
  if (from) {
    const fromDate = new Date(from);
    if (date < fromDate) return false;
  }
  if (to) {
    const toDate = new Date(to);
    if (date > toDate) return false;
  }
  return true;
};

export default function RiskOpportunitiesPage() {
  const router = useRouter();
  const [records, setRecords] = useState<RiskOpportunity[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [detailOpen, setDetailOpen] = useState(false);
  const [editingRecord, setEditingRecord] = useState<RiskOpportunity | null>(null);
  const [detailRecord, setDetailRecord] = useState<RiskOpportunity | null>(null);
  const [formState, setFormState] = useState<FormState>(emptyFormState);
  const [formErrors, setFormErrors] = useState<Record<string, string>>({});
  const [deleteTarget, setDeleteTarget] = useState<RiskOpportunity | null>(null);
  const [search, setSearch] = useState("");
  const [filters, setFilters] = useState<FilterState>(defaultFilters);
  const [canManage, setCanManage] = useState(false);

  const fetchRecords = useCallback(async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("env_risk_opportunities")
      .select("*")
      .order("last_updated_at", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar los registros.");
      console.error(error);
      setLoading(false);
      return;
    }

    setRecords((data as RiskOpportunity[]) ?? []);
    setLoading(false);
  }, []);

  const resolvePermissions = useCallback(async (currentUser: User | null) => {
    if (!currentUser) {
      setCanManage(false);
      return;
    }
    if (allowedManagers.has(currentUser.email ?? "")) {
      setCanManage(true);
      return;
    }

    const { data: pageData, error: pageError } = await supabase
      .from("sgi_pages")
      .select("id")
      .eq("path", pagePath)
      .maybeSingle();

    if (pageError) {
      console.error(pageError);
      setCanManage(false);
      return;
    }

    if (!pageData?.id) {
      setCanManage(true);
      return;
    }

    const { data: permissionData, error: permissionError } = await supabase
      .from("sgi_admin_permissions")
      .select("id")
      .eq("page_id", pageData.id)
      .eq("user_id", currentUser.id)
      .maybeSingle();

    if (permissionError) {
      console.error(permissionError);
      setCanManage(false);
      return;
    }

    setCanManage(Boolean(permissionData?.id));
  }, []);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      if (!data.session?.user) {
        router.push("/");
        return;
      }
      void resolvePermissions(data.session.user);
    };

    void loadSession();
  }, [router, resolvePermissions]);

  useEffect(() => {
    void fetchRecords();
  }, [fetchRecords]);

  const filteredRecords = useMemo(() => {
    const query = search.trim().toLowerCase();
    return records.filter((record) => {
      if (filters.type !== "all" && record.type !== filters.type) return false;
      if (filters.source !== "all" && record.source !== filters.source) return false;
      if (filters.actionStatus !== "all" && record.action_status !== filters.actionStatus) return false;
      if (filters.linkedLegal !== "all") {
        const expected = filters.linkedLegal === "yes";
        if (Boolean(record.linked_to_legal) !== expected) return false;
      }
      if (filters.responsible !== "all") {
        if ((record.action_owner ?? "-") !== filters.responsible) return false;
      }
      if (filters.commitmentFrom || filters.commitmentTo) {
        if (!dateInRange(record.commitment_date, filters.commitmentFrom, filters.commitmentTo)) return false;
      }
      if (filters.closureFrom || filters.closureTo) {
        if (!dateInRange(record.closure_date, filters.closureFrom, filters.closureTo)) return false;
      }

      const residualLevel = getLevel(record.p_residual, record.i_residual);
      const residualClass = getClassification(residualLevel, record.i_residual);
      const acceptable = getAcceptable(residualLevel);

      if (filters.residualClass !== "all") {
        if (!residualClass || residualClass !== filters.residualClass) return false;
      }
      if (filters.acceptable !== "all") {
        const expected = filters.acceptable === "yes";
        if (acceptable === null || acceptable !== expected) return false;
      }

      if (query) {
        const evidenceText = parseEvidence(record.evidence_urls).join(" ").toLowerCase();
        const searchTarget = [
          record.title,
          record.description,
          record.current_controls,
          record.action_owner,
          record.source,
          evidenceText,
        ]
          .filter(Boolean)
          .join(" ")
          .toLowerCase();
        if (!searchTarget.includes(query)) return false;
      }

      return true;
    });
  }, [records, search, filters]);

  const kpiData = useMemo(() => {
    const total = filteredRecords.length;
    const initialCounts: Record<string, number> = {
      low: 0,
      medium: 0,
      high: 0,
      critical: 0,
    };
    const residualCounts: Record<string, number> = {
      low: 0,
      medium: 0,
      high: 0,
      critical: 0,
    };
    let acceptableCount = 0;
    let nonAcceptableCount = 0;
    let openActions = 0;
    let inProgressActions = 0;
    let closedActions = 0;
    let overdueActions = 0;
    let legalPriority = 0;

    const today = new Date();

    filteredRecords.forEach((record) => {
      const initialLevel = getLevel(record.p_initial, record.i_initial);
      const initialClass = getClassification(initialLevel, record.i_initial);
      if (initialClass) initialCounts[initialClass] += 1;

      const residualLevel = getLevel(record.p_residual, record.i_residual);
      const residualClass = getClassification(residualLevel, record.i_residual);
      if (residualClass) residualCounts[residualClass] += 1;

      const acceptable = getAcceptable(residualLevel);
      if (acceptable === true) acceptableCount += 1;
      if (acceptable === false) nonAcceptableCount += 1;

      if (record.action_status === "open") openActions += 1;
      if (record.action_status === "in_progress") inProgressActions += 1;
      if (record.action_status === "closed") closedActions += 1;

      if (record.action_status !== "closed" && record.commitment_date) {
        const commitmentDate = new Date(record.commitment_date);
        if (!Number.isNaN(commitmentDate.getTime()) && commitmentDate < today) {
          overdueActions += 1;
        }
      }

      if (record.linked_to_legal) legalPriority += 1;
    });

    return {
      total,
      initialCounts,
      residualCounts,
      acceptableCount,
      nonAcceptableCount,
      openActions,
      inProgressActions,
      closedActions,
      overdueActions,
      legalPriority,
    };
  }, [filteredRecords]);

  const resetForm = () => {
    setFormState(emptyFormState);
    setFormErrors({});
    setEditingRecord(null);
  };

  const openCreateDialog = () => {
    resetForm();
    setDialogOpen(true);
  };

  const openEditDialog = (record: RiskOpportunity) => {
    setEditingRecord(record);
    setFormState({
      type: record.type ?? "risk",
      title: record.title ?? "",
      description: record.description ?? "",
      source: record.source ?? "aspect",
      environmental_aspect: record.environmental_aspect ?? "",
      linked_to_legal: record.linked_to_legal ? "yes" : "no",
      current_controls: record.current_controls ?? "",
      p_initial: record.p_initial ? String(record.p_initial) : "",
      i_initial: record.i_initial ? String(record.i_initial) : "",
      action_plan: record.action_plan ?? "",
      action_owner: record.action_owner ?? "",
      commitment_date: record.commitment_date ?? "",
      action_status: record.action_status ?? "",
      closure_date: record.closure_date ?? "",
      evidence_urls: parseEvidence(record.evidence_urls).join("\n"),
      p_residual: record.p_residual ? String(record.p_residual) : "",
      i_residual: record.i_residual ? String(record.i_residual) : "",
      next_review_date: record.next_review_date ?? "",
    });
    setFormErrors({});
    setDialogOpen(true);
  };

  const openDetailDialog = (record: RiskOpportunity) => {
    setDetailRecord(record);
    setDetailOpen(true);
  };

  const updateForm = (field: keyof FormState, value: string) => {
    setFormState((prev) => ({
      ...prev,
      [field]: value,
    }));
  };

  const validateForm = () => {
    const errors: Record<string, string> = {};
    if (!formState.type) errors.type = "Selecciona el tipo.";
    if (!formState.title.trim()) errors.title = "Indica un título.";
    if (!formState.source) errors.source = "Selecciona la fuente.";
    if (!formState.current_controls.trim()) errors.current_controls = "Describe los controles actuales.";
    if (!formState.p_initial) errors.p_initial = "Selecciona P inicial.";
    if (!formState.i_initial) errors.i_initial = "Selecciona I inicial.";

    const pInitial = Number(formState.p_initial);
    const iInitial = Number(formState.i_initial);
    const initialLevel = getLevel(pInitial, iInitial) ?? 0;
    const actionRequired = initialLevel >= 13 || formState.linked_to_legal === "yes";

    if (actionRequired) {
      if (!formState.action_plan.trim()) errors.action_plan = "La acción es obligatoria.";
      if (!formState.action_owner.trim()) errors.action_owner = "Asigna un responsable.";
      if (!formState.commitment_date) errors.commitment_date = "Define una fecha compromiso.";
      if (!formState.action_status) errors.action_status = "Selecciona un estatus.";
    }

    const evidenceInput = serializeEvidence(formState.evidence_urls);
    const actionClosed = formState.action_status === "closed";
    const residualFilled = formState.p_residual || formState.i_residual;
    const residualEnabled = actionClosed && Boolean(evidenceInput);

    if (residualFilled) {
      if (!residualEnabled) {
        errors.p_residual = "La residual solo se registra con acción cerrada y evidencia.";
      } else {
        if (!formState.p_residual) errors.p_residual = "Selecciona P residual.";
        if (!formState.i_residual) errors.i_residual = "Selecciona I residual.";
      }
    }

    if (formState.p_residual && formState.i_residual) {
      const pResidual = Number(formState.p_residual);
      const iResidual = Number(formState.i_residual);
      if (pResidual > pInitial || iResidual > iInitial) {
        errors.p_residual = "La evaluación residual debe ser menor o igual a la inicial.";
        errors.i_residual = "La evaluación residual debe ser menor o igual a la inicial.";
      }
    }

    setFormErrors(errors);
    return Object.keys(errors).length === 0;
  };

  const handleSave = async () => {
    if (!validateForm()) return;

    const payload = {
      type: formState.type,
      title: formState.title.trim(),
      description: formState.description.trim() || null,
      source: formState.source,
      environmental_aspect: formState.environmental_aspect.trim() || null,
      linked_to_legal: formState.linked_to_legal === "yes",
      current_controls: formState.current_controls.trim(),
      p_initial: formState.p_initial ? Number(formState.p_initial) : null,
      i_initial: formState.i_initial ? Number(formState.i_initial) : null,
      action_plan: formState.action_plan.trim() || null,
      action_owner: formState.action_owner.trim() || null,
      commitment_date: formState.commitment_date || null,
      action_status: formState.action_status || null,
      closure_date: formState.closure_date || null,
      evidence_urls: serializeEvidence(formState.evidence_urls) || null,
      p_residual: formState.p_residual ? Number(formState.p_residual) : null,
      i_residual: formState.i_residual ? Number(formState.i_residual) : null,
      next_review_date: formState.next_review_date || null,
      last_updated_at: new Date().toISOString(),
    };

    if (editingRecord) {
      const { error } = await supabase
        .from("env_risk_opportunities")
        .update(payload)
        .eq("id", editingRecord.id);

      if (error) {
        toast.error("No se pudo actualizar el registro.");
        console.error(error);
        return;
      }
      toast.success("Registro actualizado correctamente.");
    } else {
      const { error } = await supabase.from("env_risk_opportunities").insert(payload);
      if (error) {
        toast.error("No se pudo crear el registro.");
        console.error(error);
        return;
      }
      toast.success("Registro creado correctamente.");
    }

    setDialogOpen(false);
    resetForm();
    void fetchRecords();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase.from("env_risk_opportunities").delete().eq("id", deleteTarget.id);
    if (error) {
      toast.error("No se pudo eliminar el registro.");
      console.error(error);
      return;
    }
    toast.success("Registro eliminado.");
    setDeleteTarget(null);
    void fetchRecords();
  };

  const residualEnabled = formState.action_status === "closed" && Boolean(serializeEvidence(formState.evidence_urls));
  const pInitial = formState.p_initial ? Number(formState.p_initial) : null;
  const iInitial = formState.i_initial ? Number(formState.i_initial) : null;
  const initialLevel = getLevel(pInitial, iInitial);
  const initialClass = getClassification(initialLevel, iInitial);
  const pResidual = formState.p_residual ? Number(formState.p_residual) : null;
  const iResidual = formState.i_residual ? Number(formState.i_residual) : null;
  const residualLevel = getLevel(pResidual, iResidual);
  const residualClass = getClassification(residualLevel, iResidual);
  const acceptableResidual = getAcceptable(residualLevel);

  const responsibleOptions = useMemo(() => {
    const owners = new Set<string>();
    records.forEach((record) => {
      if (record.action_owner) owners.add(record.action_owner);
    });
    return Array.from(owners);
  }, [records]);

  return (
    <SGIDashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col gap-4 md:flex-row md:items-center md:justify-between">
          <div>
            <h1 className="text-2xl font-semibold text-slate-900">Riesgos y oportunidades ambientales</h1>
            <p className="text-sm text-slate-600">
              Administra evaluaciones iniciales y residuales con controles, evidencia y trazabilidad.
            </p>
          </div>
          {canManage && (
            <Button onClick={openCreateDialog} className="gap-2">
              <Plus className="h-4 w-4" />
              Nuevo registro
            </Button>
          )}
        </div>

        <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
          <Card>
            <CardHeader>
              <CardTitle>Total de registros</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-slate-900">{kpiData.total}</CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Distribución inicial</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-slate-700">
              {classificationOptions.map((option) => (
                <div key={option.value} className="flex items-center justify-between">
                  <span>{option.label}</span>
                  <span>
                    {kpiData.initialCounts[option.value]} ({kpiData.total ? Math.round((kpiData.initialCounts[option.value] / kpiData.total) * 100) : 0}%)
                  </span>
                </div>
              ))}
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Distribución residual</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-slate-700">
              {classificationOptions.map((option) => (
                <div key={option.value} className="flex items-center justify-between">
                  <span>{option.label}</span>
                  <span>
                    {kpiData.residualCounts[option.value]} ({kpiData.total ? Math.round((kpiData.residualCounts[option.value] / kpiData.total) * 100) : 0}%)
                  </span>
                </div>
              ))}
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Residual aceptable</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-slate-700">
              <div className="flex items-center justify-between">
                <span>Aceptables</span>
                <span>
                  {kpiData.acceptableCount} ({kpiData.total ? Math.round((kpiData.acceptableCount / kpiData.total) * 100) : 0}%)
                </span>
              </div>
              <div className="flex items-center justify-between">
                <span>No aceptables</span>
                <span>
                  {kpiData.nonAcceptableCount} ({kpiData.total ? Math.round((kpiData.nonAcceptableCount / kpiData.total) * 100) : 0}%)
                </span>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Acciones</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-slate-700">
              <div className="flex items-center justify-between">
                <span>Abiertas</span>
                <span>{kpiData.openActions}</span>
              </div>
              <div className="flex items-center justify-between">
                <span>En proceso</span>
                <span>{kpiData.inProgressActions}</span>
              </div>
              <div className="flex items-center justify-between">
                <span>Cerradas</span>
                <span>{kpiData.closedActions}</span>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Acciones vencidas</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-rose-600">{kpiData.overdueActions}</CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Prioridad legal</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-amber-600">{kpiData.legalPriority}</CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filtros y búsqueda</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex flex-col gap-3 md:flex-row md:items-center">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-slate-400" />
                <Input
                  value={search}
                  onChange={(event) => setSearch(event.target.value)}
                  placeholder="Buscar por título, descripción, controles, responsable, fuente o evidencia"
                  className="pl-9"
                />
              </div>
              <Button variant="outline" onClick={() => setFilters(defaultFilters)}>
                Limpiar filtros
              </Button>
            </div>
            <div className="grid gap-3 md:grid-cols-2 lg:grid-cols-3">
              <div className="space-y-1">
                <Label>Tipo</Label>
                <Select value={filters.type} onValueChange={(value) => setFilters((prev) => ({ ...prev, type: value }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {typeOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Fuente</Label>
                <Select value={filters.source} onValueChange={(value) => setFilters((prev) => ({ ...prev, source: value }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todas</SelectItem>
                    {sourceOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Clasificación residual</Label>
                <Select value={filters.residualClass} onValueChange={(value) => setFilters((prev) => ({ ...prev, residualClass: value }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todas</SelectItem>
                    {classificationOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Aceptable</Label>
                <Select value={filters.acceptable} onValueChange={(value) => setFilters((prev) => ({ ...prev, acceptable: value }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    <SelectItem value="yes">Sí</SelectItem>
                    <SelectItem value="no">No</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Ligado a requisito legal</Label>
                <Select value={filters.linkedLegal} onValueChange={(value) => setFilters((prev) => ({ ...prev, linkedLegal: value }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    <SelectItem value="yes">Sí</SelectItem>
                    <SelectItem value="no">No</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Estatus de acción</Label>
                <Select value={filters.actionStatus} onValueChange={(value) => setFilters((prev) => ({ ...prev, actionStatus: value }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {actionStatusOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Responsable</Label>
                <Select
                  value={filters.responsible}
                  onValueChange={(value) => setFilters((prev) => ({ ...prev, responsible: value }))}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {responsibleOptions.map((owner) => (
                      <SelectItem key={owner} value={owner}>
                        {owner}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-1">
                <Label>Fecha compromiso (desde)</Label>
                <Input
                  type="date"
                  value={filters.commitmentFrom}
                  onChange={(event) => setFilters((prev) => ({ ...prev, commitmentFrom: event.target.value }))}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha compromiso (hasta)</Label>
                <Input
                  type="date"
                  value={filters.commitmentTo}
                  onChange={(event) => setFilters((prev) => ({ ...prev, commitmentTo: event.target.value }))}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha cierre (desde)</Label>
                <Input
                  type="date"
                  value={filters.closureFrom}
                  onChange={(event) => setFilters((prev) => ({ ...prev, closureFrom: event.target.value }))}
                />
              </div>
              <div className="space-y-1">
                <Label>Fecha cierre (hasta)</Label>
                <Input
                  type="date"
                  value={filters.closureTo}
                  onChange={(event) => setFilters((prev) => ({ ...prev, closureTo: event.target.value }))}
                />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Registro de riesgos y oportunidades</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Tipo</TableHead>
                    <TableHead>Título</TableHead>
                    <TableHead>Fuente</TableHead>
                    <TableHead>Controles actuales</TableHead>
                    <TableHead>Inicial</TableHead>
                    <TableHead>Acción</TableHead>
                    <TableHead>Residual</TableHead>
                    <TableHead>Aceptable</TableHead>
                    <TableHead>Evidencia</TableHead>
                    <TableHead>Crear acción</TableHead>                    
                    <TableHead>Acciones</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {loading ? (
                    <TableRow>
                      <TableCell colSpan={11} className="text-center text-sm text-slate-500">
                        Cargando registros...
                      </TableCell>
                    </TableRow>
                  ) : filteredRecords.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={11} className="text-center text-sm text-slate-500">
                        No hay registros con los filtros seleccionados.
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredRecords.map((record) => {
                      const initialLevelRow = getLevel(record.p_initial, record.i_initial);
                      const initialClassRow = getClassification(initialLevelRow, record.i_initial);
                      const residualLevelRow = getLevel(record.p_residual, record.i_residual);
                      const residualClassRow = getClassification(residualLevelRow, record.i_residual);
                      const acceptableRow = getAcceptable(residualLevelRow);
                      const evidenceList = parseEvidence(record.evidence_urls);

                      return (
                        <TableRow key={record.id}>
                          <TableCell>{record.type === "risk" ? "Riesgo" : "Oportunidad"}</TableCell>
                          <TableCell className="font-medium text-slate-900">{record.title}</TableCell>
                          <TableCell>{sourceOptions.find((option) => option.value === record.source)?.label ?? record.source}</TableCell>
                          <TableCell className="min-w-[200px]">{record.current_controls}</TableCell>
                          <TableCell className="space-y-1">
                            <div className="text-xs text-slate-500">P {record.p_initial ?? "-"} / I {record.i_initial ?? "-"}</div>
                            <div className="text-sm font-semibold">Nivel: {initialLevelRow ?? "-"}</div>
                            <Badge className={classificationBadge(initialClassRow)}>{classificationLabel(initialClassRow)}</Badge>
                          </TableCell>
                          <TableCell className="space-y-1">
                            <div className="text-sm text-slate-700">{record.action_plan ?? "Sin acción"}</div>
                            <div className="text-xs text-slate-500">
                              {actionStatusOptions.find((option) => option.value === record.action_status)?.label ?? "Sin estatus"} · {formatDateLabel(record.commitment_date)}
                            </div>
                          </TableCell>
                          <TableCell className="space-y-1">
                            <div className="text-xs text-slate-500">P {record.p_residual ?? "-"} / I {record.i_residual ?? "-"}</div>
                            <div className="text-sm font-semibold">Nivel: {residualLevelRow ?? "-"}</div>
                            <Badge className={classificationBadge(residualClassRow)}>{classificationLabel(residualClassRow)}</Badge>
                          </TableCell>
                          <TableCell>
                            {acceptableRow === null ? "-" : (
                              <Badge className={acceptableRow ? "bg-emerald-100 text-emerald-800" : "bg-rose-100 text-rose-800"}>
                                {acceptableRow ? "Sí" : "No"}
                              </Badge>
                            )}
                          </TableCell>
                          <TableCell className="space-y-1">
                            {evidenceList.length > 0 ? (
                              evidenceList.map((item) => (
                                <a key={item} href={item} className="block text-xs text-blue-600 underline" target="_blank" rel="noreferrer">
                                  Ver evidencia
                                </a>
                              ))
                            ) : (
                              <span className="text-sm text-slate-500">Sin evidencia</span>
                            )}
                          </TableCell>
                          <TableCell>
                            <Button
                              asChild
                              size="sm"
                              className="bg-sky-500 text-white hover:bg-sky-600"
                            >
                              <Link href={buildActionPlanHref(record)}>
                                Crear acción
                              </Link>
                            </Button>
                          </TableCell>                          
                          <TableCell>
                            <div className="flex items-center gap-2">
                              <Button variant="ghost" size="icon" onClick={() => openDetailDialog(record)}>
                                <Eye className="h-4 w-4" />
                              </Button>
                              {canManage && (
                                <>
                                  <Button variant="ghost" size="icon" onClick={() => openEditDialog(record)}>
                                    <Pencil className="h-4 w-4" />
                                  </Button>
                                  <Button variant="ghost" size="icon" onClick={() => setDeleteTarget(record)}>
                                    <Trash2 className="h-4 w-4" />
                                  </Button>
                                </>
                              )}
                            </div>
                          </TableCell>
                        </TableRow>
                      );
                    })
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>

        <Dialog open={dialogOpen} onOpenChange={(open) => {
          if (!open) resetForm();
          setDialogOpen(open);
        }}>
          <DialogContent className="max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>{editingRecord ? "Editar registro" : "Nuevo registro"}</DialogTitle>
            </DialogHeader>
            <div className="space-y-6">
              <div className="grid gap-4 md:grid-cols-2">
                <div className="space-y-1">
                  <Label>Tipo *</Label>
                  <Select value={formState.type} onValueChange={(value) => updateForm("type", value)}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {typeOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {formErrors.type && <p className="text-xs text-rose-500">{formErrors.type}</p>}
                </div>
                <div className="space-y-1">
                  <Label>Fuente *</Label>
                  <Select value={formState.source} onValueChange={(value) => updateForm("source", value)}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {sourceOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {formErrors.source && <p className="text-xs text-rose-500">{formErrors.source}</p>}
                </div>
              </div>
              <div className="space-y-1">
                <Label>Título *</Label>
                <Input value={formState.title} onChange={(event) => updateForm("title", event.target.value)} />
                {formErrors.title && <p className="text-xs text-rose-500">{formErrors.title}</p>}
              </div>
              <div className="space-y-1">
                <Label>Descripción</Label>
                <Textarea value={formState.description} onChange={(event) => updateForm("description", event.target.value)} />
              </div>
              <div className="grid gap-4 md:grid-cols-2">
                <div className="space-y-1">
                  <Label>Aspecto ambiental</Label>
                  <Input value={formState.environmental_aspect} onChange={(event) => updateForm("environmental_aspect", event.target.value)} />
                </div>
                <div className="space-y-1">
                  <Label>Ligado a requisito legal</Label>
                  <Select value={formState.linked_to_legal} onValueChange={(value) => updateForm("linked_to_legal", value)}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="no">No</SelectItem>
                      <SelectItem value="yes">Sí</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
              <div className="space-y-1">
                <Label>Controles actuales *</Label>
                <Textarea value={formState.current_controls} onChange={(event) => updateForm("current_controls", event.target.value)} />
                {formErrors.current_controls && <p className="text-xs text-rose-500">{formErrors.current_controls}</p>}
              </div>

              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Evaluación inicial</CardTitle>
                </CardHeader>
                <CardContent className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-1">
                    <Label className="flex items-center gap-2">
                      P inicial *
                      <TooltipProvider>
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Button type="button" size="icon" variant="ghost">
                              ?
                            </Button>
                          </TooltipTrigger>
                          <TooltipContent className="max-w-xs">
                            {probabilityScale.map((item) => (
                              <p key={item.value} className="text-xs">
                                <strong>{item.value}. {item.label}</strong>: {item.detail}
                              </p>
                            ))}
                          </TooltipContent>
                        </Tooltip>
                      </TooltipProvider>
                    </Label>
                    <Select value={formState.p_initial} onValueChange={(value) => updateForm("p_initial", value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Selecciona" />
                      </SelectTrigger>
                      <SelectContent>
                        {[1, 2, 3, 4, 5].map((value) => (
                          <SelectItem key={value} value={String(value)}>
                            {value}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {formErrors.p_initial && <p className="text-xs text-rose-500">{formErrors.p_initial}</p>}
                  </div>
                  <div className="space-y-1">
                    <Label className="flex items-center gap-2">
                      I inicial *
                      <TooltipProvider>
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Button type="button" size="icon" variant="ghost">
                              ?
                            </Button>
                          </TooltipTrigger>
                          <TooltipContent className="max-w-xs">
                            {impactScale.map((item) => (
                              <p key={item.value} className="text-xs">
                                <strong>{item.value}. {item.label}</strong>: {item.detail}
                              </p>
                            ))}
                          </TooltipContent>
                        </Tooltip>
                      </TooltipProvider>
                    </Label>
                    <Select value={formState.i_initial} onValueChange={(value) => updateForm("i_initial", value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Selecciona" />
                      </SelectTrigger>
                      <SelectContent>
                        {[1, 2, 3, 4, 5].map((value) => (
                          <SelectItem key={value} value={String(value)}>
                            {value}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {formErrors.i_initial && <p className="text-xs text-rose-500">{formErrors.i_initial}</p>}
                  </div>
                  <div className="space-y-1">
                    <Label>Nivel inicial</Label>
                    <Input value={initialLevel ?? "-"} readOnly />
                  </div>
                  <div className="space-y-1">
                    <Label>Clasificación inicial</Label>
                    <Badge className={classificationBadge(initialClass)}>{classificationLabel(initialClass)}</Badge>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Acción / control adicional</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-1">
                    <Label>Acción / control adicional</Label>
                    <Textarea value={formState.action_plan} onChange={(event) => updateForm("action_plan", event.target.value)} />
                    {formErrors.action_plan && <p className="text-xs text-rose-500">{formErrors.action_plan}</p>}
                  </div>
                  <div className="grid gap-4 md:grid-cols-2">
                    <div className="space-y-1">
                      <Label>Responsable</Label>
                      <Input value={formState.action_owner} onChange={(event) => updateForm("action_owner", event.target.value)} />
                      {formErrors.action_owner && <p className="text-xs text-rose-500">{formErrors.action_owner}</p>}
                    </div>
                    <div className="space-y-1">
                      <Label>Fecha compromiso</Label>
                      <Input type="date" value={formState.commitment_date} onChange={(event) => updateForm("commitment_date", event.target.value)} />
                      {formErrors.commitment_date && <p className="text-xs text-rose-500">{formErrors.commitment_date}</p>}
                    </div>
                  </div>
                  <div className="grid gap-4 md:grid-cols-2">
                    <div className="space-y-1">
                      <Label>Estatus de acción</Label>
                      <Select value={formState.action_status} onValueChange={(value) => updateForm("action_status", value)}>
                        <SelectTrigger>
                          <SelectValue placeholder="Selecciona" />
                        </SelectTrigger>
                        <SelectContent>
                          {actionStatusOptions.map((option) => (
                            <SelectItem key={option.value} value={option.value}>
                              {option.label}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {formErrors.action_status && <p className="text-xs text-rose-500">{formErrors.action_status}</p>}
                    </div>
                    <div className="space-y-1">
                      <Label>Fecha de cierre</Label>
                      <Input type="date" value={formState.closure_date} onChange={(event) => updateForm("closure_date", event.target.value)} />
                    </div>
                  </div>
                  <div className="space-y-1">
                    <Label>Evidencia (uno por línea o separado por coma)</Label>
                    <Textarea value={formState.evidence_urls} onChange={(event) => updateForm("evidence_urls", event.target.value)} />
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Evaluación residual</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <p className="text-xs text-slate-500">
                    Solo se habilita cuando la acción esté cerrada y exista evidencia. Nivel residual ≤ nivel inicial.
                  </p>
                  <div className="grid gap-4 md:grid-cols-2">
                    <div className="space-y-1">
                      <Label>P residual</Label>
                      <Select
                        value={formState.p_residual}
                        onValueChange={(value) => updateForm("p_residual", value)}
                        disabled={!residualEnabled}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder={residualEnabled ? "Selecciona" : "Bloqueado"} />
                        </SelectTrigger>
                        <SelectContent>
                          {[1, 2, 3, 4, 5].map((value) => (
                            <SelectItem key={value} value={String(value)}>
                              {value}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {formErrors.p_residual && <p className="text-xs text-rose-500">{formErrors.p_residual}</p>}
                    </div>
                    <div className="space-y-1">
                      <Label>I residual</Label>
                      <Select
                        value={formState.i_residual}
                        onValueChange={(value) => updateForm("i_residual", value)}
                        disabled={!residualEnabled}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder={residualEnabled ? "Selecciona" : "Bloqueado"} />
                        </SelectTrigger>
                        <SelectContent>
                          {[1, 2, 3, 4, 5].map((value) => (
                            <SelectItem key={value} value={String(value)}>
                              {value}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {formErrors.i_residual && <p className="text-xs text-rose-500">{formErrors.i_residual}</p>}
                    </div>
                    <div className="space-y-1">
                      <Label>Nivel residual</Label>
                      <Input value={residualLevel ?? "-"} readOnly />
                    </div>
                    <div className="space-y-1">
                      <Label>Clasificación residual</Label>
                      <Badge className={classificationBadge(residualClass)}>{classificationLabel(residualClass)}</Badge>
                    </div>
                    <div className="space-y-1">
                      <Label>Aceptable</Label>
                      <Input value={acceptableResidual === null ? "-" : acceptableResidual ? "Sí" : "No"} readOnly />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <div className="space-y-1">
                <Label>Siguiente fecha de revisión</Label>
                <Input type="date" value={formState.next_review_date} onChange={(event) => updateForm("next_review_date", event.target.value)} />
              </div>

              <div className="flex flex-col gap-2 md:flex-row md:justify-end">
                <Button variant="outline" onClick={() => setDialogOpen(false)}>
                  Cancelar
                </Button>
                <Button onClick={handleSave}>Guardar</Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        <Dialog open={detailOpen} onOpenChange={setDetailOpen}>
          <DialogContent className="max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Detalle del registro</DialogTitle>
            </DialogHeader>
            {detailRecord && (
              <div className="space-y-4 text-sm text-slate-700">
                <div>
                  <h2 className="text-base font-semibold text-slate-900">{detailRecord.title}</h2>
                  <p>{detailRecord.description ?? "Sin descripción"}</p>
                </div>
                <div className="grid gap-3 md:grid-cols-2">
                  <div>
                    <span className="font-medium">Tipo:</span> {detailRecord.type === "risk" ? "Riesgo" : "Oportunidad"}
                  </div>
                  <div>
                    <span className="font-medium">Fuente:</span> {sourceOptions.find((option) => option.value === detailRecord.source)?.label ?? detailRecord.source}
                  </div>
                  <div>
                    <span className="font-medium">Aspecto:</span> {detailRecord.environmental_aspect ?? "-"}
                  </div>
                  <div>
                    <span className="font-medium">Legal:</span> {detailRecord.linked_to_legal ? "Sí" : "No"}
                  </div>
                </div>
                <div>
                  <span className="font-medium">Controles actuales:</span>
                  <p>{detailRecord.current_controls}</p>
                </div>
                <div className="grid gap-3 md:grid-cols-2">
                  <div>
                    <span className="font-medium">Evaluación inicial:</span>
                    <p>
                      P {detailRecord.p_initial ?? "-"} · I {detailRecord.i_initial ?? "-"} · Nivel {getLevel(detailRecord.p_initial, detailRecord.i_initial) ?? "-"} · {classificationLabel(getClassification(getLevel(detailRecord.p_initial, detailRecord.i_initial), detailRecord.i_initial))}
                    </p>
                  </div>
                  <div>
                    <span className="font-medium">Evaluación residual:</span>
                    <p>
                      P {detailRecord.p_residual ?? "-"} · I {detailRecord.i_residual ?? "-"} · Nivel {getLevel(detailRecord.p_residual, detailRecord.i_residual) ?? "-"} · {classificationLabel(getClassification(getLevel(detailRecord.p_residual, detailRecord.i_residual), detailRecord.i_residual))}
                    </p>
                  </div>
                </div>
                <div>
                  <span className="font-medium">Acción:</span>
                  <p>{detailRecord.action_plan ?? "Sin acción"}</p>
                  <p className="text-xs text-slate-500">
                    Responsable: {detailRecord.action_owner ?? "-"} · Estatus: {actionStatusOptions.find((option) => option.value === detailRecord.action_status)?.label ?? "-"} · Compromiso: {formatDateLabel(detailRecord.commitment_date)}
                  </p>
                </div>
                <div>
                  <span className="font-medium">Evidencias:</span>
                  <ul className="list-inside list-disc">
                    {parseEvidence(detailRecord.evidence_urls).length > 0 ? (
                      parseEvidence(detailRecord.evidence_urls).map((item) => (
                        <li key={item}>
                          <a href={item} className="text-blue-600 underline" target="_blank" rel="noreferrer">
                            {item}
                          </a>
                        </li>
                      ))
                    ) : (
                      <li>Sin evidencia</li>
                    )}
                  </ul>
                </div>
                <div className="grid gap-3 md:grid-cols-2">
                  <div>
                    <span className="font-medium">Última actualización:</span> {formatDateLabel(detailRecord.last_updated_at)}
                  </div>
                  <div>
                    <span className="font-medium">Siguiente revisión:</span> {formatDateLabel(detailRecord.next_review_date)}
                  </div>
                </div>
              </div>
            )}
          </DialogContent>
        </Dialog>

        <AlertDialog open={Boolean(deleteTarget)} onOpenChange={() => setDeleteTarget(null)}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>¿Eliminar registro?</AlertDialogTitle>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancelar</AlertDialogCancel>
              <AlertDialogAction className="bg-rose-600 hover:bg-rose-700" onClick={handleDelete}>
                Eliminar
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </div>
    </SGIDashboardLayout>
  );
}