"use client";

import Link from "next/link";
import { useEffect, useMemo, useState } from "react";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Tabs, TabsList, TabsTrigger } from "@/components/ui/tabs";

const languageOptions = [
  { id: "es", label: "Español" },
  { id: "ja", label: "日本語" },
  { id: "en", label: "English" },  
] as const;

type LanguageId = (typeof languageOptions)[number]["id"];

type PolicyContent = {
  title: string;
  intro: string;
  items: string[];
  closing?: string;
};

type Policy = {
  id: string;
  label: string;
  accent: string;
  summary: string;
  languages: Partial<Record<LanguageId, PolicyContent>>;
};

const policies: Policy[] = [
  {
    id: "ambiental",
    label: "Política Ambiental",
    accent: "from-emerald-500 via-emerald-400 to-lime-400",
    summary: "Compromiso con la protección ambiental y la sostenibilidad.",
    languages: {
      es: {
        title: "Política Ambiental",
        intro:
          "En Meiwa Mold Mexico, como empresa dedicada a la fabricación de moldes de acero para inyección de aluminio, reafirmamos nuestro compromiso con la protección del medio ambiente, integrando prácticas sostenibles en todas nuestras operaciones, productos y servicios. Nuestro Sistema de Gestión Ambiental, alineado con los requisitos de la norma ISO 14001:2015, guía nuestra actuación con base en los siguientes principios:",
        items: [
          "Garantizamos el estricto cumplimiento de la legislación ambiental vigente, reglamentos aplicables y otros compromisos adquiridos, desde la adquisición de materias primas hasta la entrega del producto final.",
          "Establecemos, monitoreamos y actualizamos periódicamente objetivos y metas ambientales, promoviendo la mejora continua de nuestro desempeño ambiental y del Sistema de Gestión Ambiental.",
          "Actuamos de manera proactiva para prevenir la contaminación, reducir emisiones, minimizar residuos, fomentar la reutilización y reciclaje, y promover el uso eficiente de la energía y recursos naturales.",
          "Diseñamos y fabricamos moldes con altos estándares de calidad y responsabilidad ambiental, promoviendo soluciones innovadoras que contribuyan al desarrollo sostenible y al bienestar social.",
          "Identificamos, evaluamos y gestionamos los aspectos e impactos ambientales de nuestras actividades, considerando el contexto organizacional y los riesgos y oportunidades ambientales.",
          "Impulsamos la sensibilización, capacitación continua y participación de todos nuestros colaboradores y partes interesadas, fortaleciendo una cultura organizacional orientada a la sostenibilidad.",
          "Esta política se comunica a todos los colaboradores y partes interesadas, y está disponible públicamente como reflejo de nuestro compromiso con el medio ambiente.",
        ],
        closing:
          "En Meiwa Mold Mexico, trabajamos todos los días con la visión de garantizar un futuro sostenible, integrando el respeto por el medio ambiente en cada decisión y proceso de nuestra organización.",
      },
      ja: {
        title: "環境方針",
        intro:
          "Meiwa Mold Mexicoでは、アルミニウム射出成形用の鋼製金型の製造を行う企業として、すべての業務、製品、サービスにおいて持続可能な取り組みを取り入れ、環境保護への取り組みを改めて表明します。当社の環境マネジメントシステムはISO 14001:2015規格の要件に準拠しており、以下の原則に基づいて運用されています。",
        items: [
          "原材料の調達から最終製品の納品まで、現行の環境法規制、適用される規則、およびその他の取り決めた義務を厳格に遵守します。",
          "環境目標を設定・監視・定期的に更新し、環境パフォーマンスおよび環境マネジメントシステムの継続的改善を推進します。",
          "汚染の予防、排出の削減、廃棄物の最小化、再利用およびリサイクルの促進、エネルギーおよび天然資源の効率的利用を積極的に行います。",
          "高品質で環境に配慮した金型を設計・製造し、持続可能な発展と社会的福祉に貢献する革新的なソリューションを推進します。",
          "組織の状況や環境に関連するリスクと機会を考慮し、当社の活動による環境側面および影響を特定、評価、管理します。",
          "全従業員および関係者の意識向上、継続的な教育・訓練、積極的な参加を促進し、持続可能性を重視した組織文化を強化します。",
          "この方針は、すべての従業員および関係者に周知されており、当社の環境への取り組みを反映したものとして公開されています。",
        ],
        closing:
          "Meiwa Mold Mexicoでは、持続可能な未来を確保するというビジョンのもと、日々の業務に取り組んでいます。私たちは、あらゆる意思決定やプロセスにおいて、環境への配慮を組み込んでいます。",
      },
    },
  },
  {
    id: "calidad",
    label: "Política de Calidad",
    accent: "from-sky-500 via-indigo-500 to-violet-500",
    summary: "Excelencia y mejora continua en todos los procesos.",
    languages: {
      es: {
        title: "Política de Calidad",
        intro:
          "En Meiwa Mold Mexico, como empresa dedicada a la fabricación de moldes de acero para inyección de aluminio, reconocemos y asumimos nuestra responsabilidad social y profesional dentro del sector. Nuestra misión es proveer productos de alta calidad en los que nuestros clientes puedan confiar.",
        items: [
          "Con este propósito, establecemos como objetivos clave de calidad: “Cero defectos que lleguen al cliente” y “Erradicación de los factores tipo B que provocan defectos”.",
          "Nos comprometemos con la prevención y no recurrencia de los defectos, así como con la eliminación de los 3M (Muri: sobrecarga de personas, equipos o procesos. Muda: desperdicios o actividades que no agregan valor. Mura: irregularidades o variabilidad en procesos y resultados.) en el área de producción.",
          "Promovemos la estandarización de procesos y la aplicación rigurosa de la metodología 5S —Seiri (Clasificar), Seiton (Ordenar), Seiso (Limpiar), Seiketsu (Estandarizar) y Shitsuke (Disciplina)— con el fin de estabilizar y mejorar continuamente la calidad en todas nuestras operaciones.",
          "Nuestra empresa opera y mantiene de forma efectiva el Sistema de Gestión de Calidad, promoviendo su mejora continua para asegurar el cumplimiento de los requisitos del cliente y de la normativa legal aplicable, con el objetivo final de maximizar la satisfacción del cliente.",
          "La presente política de calidad se comunica a todo el personal y partes interesadas, y se aplica en las actividades diarias.",
          "También servirá como base para la definición de los objetivos de calidad internos y el desarrollo de actividades de mejora, manteniéndose siempre actualizada mediante revisiones periódicas que aseguren su vigencia y eficacia.",
        ],
        closing:
          "En Meiwa Mold Mexico, trabajamos cada día con la visión de construir una cultura de calidad sólida, integrando la mejora continua, la precisión y el compromiso con la excelencia en cada proceso, decisión y producto que entregamos.",
      },
    },
  },
  {
    id: "seguridad",
    label: "Política de Seguridad",
    accent: "from-amber-500 via-orange-500 to-rose-500",
    summary: "Prevención de riesgos y bienestar para todo el equipo.",
    languages: {
      es: {
        title: "Política de Seguridad",
        intro:
          "Estamos desarrollando la política de seguridad para asegurar que cada colaborador cuente con lineamientos claros de prevención, protección y respuesta en el entorno laboral.",
        items: [
          "Mantener la seguridad como prioridad en cada operación y proceso.",
          "Fortalecer la cultura preventiva mediante capacitación continua.",
          "Promover la mejora continua de las condiciones de trabajo.",
        ],
        closing:
          "Próximamente publicaremos la versión oficial en español y japonés.",
      },
    },
  },
];

const languageFallback: PolicyContent = {
  title: "Contenido en preparación",
  intro:
    "Esta política estará disponible próximamente en el idioma seleccionado. Por favor, selecciona otro idioma o vuelve más tarde.",
  items: [],
};

export default function PoliciesPage() {
  const [activePolicy, setActivePolicy] = useState(policies[0].id);
  const [activeLanguage, setActiveLanguage] = useState<LanguageId>("es");

  useEffect(() => {
    const policy = policies.find((item) => item.id === activePolicy);
    if (!policy) {
      return;
    }
    if (!policy.languages[activeLanguage]) {
      const nextLanguage = (Object.keys(policy.languages)[0] ?? "es") as LanguageId;
      setActiveLanguage(nextLanguage);
    }
  }, [activePolicy, activeLanguage]);

  const activePolicyData = useMemo(
    () => policies.find((item) => item.id === activePolicy),
    [activePolicy]
  );

  const activeContent = useMemo(() => {
    if (!activePolicyData) {
      return languageFallback;
    }
    return activePolicyData.languages[activeLanguage] ?? languageFallback;
  }, [activeLanguage, activePolicyData]);

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-emerald-50">
        <div className="relative overflow-hidden">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-sky-500/10" />
          <div className="relative max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pt-16 pb-10">
            <div className="max-w-3xl space-y-4">
              <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
                Meiwa Mold Mexico
              </p>
              <h1 className="text-4xl sm:text-5xl font-bold text-gray-900">
                Políticas Meiwa Mold Mexico
              </h1>
              <p className="text-lg text-gray-600">
                Selecciona la política que deseas consultar y el idioma para visualizarla.
              </p>
              <div className="flex flex-wrap items-center gap-3">
                <Button asChild variant="outline">
                  <Link href="/meiwa-core/meiwa-sgi/leadership">Volver a liderazgo</Link>
                </Button>
                <Button asChild>
                  <Link
                    href="https://surveys.meiwa.com.mx/"
                    target="_blank"
                    rel="noreferrer"
                  >
                    Ver en línea
                  </Link>
                </Button>
              </div>         
            </div>
          </div>
        </div>

        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pb-20">
          <div className="space-y-6">
            <Tabs value={activePolicy} onValueChange={setActivePolicy}>
              <TabsList className="grid w-full max-w-2xl grid-cols-1 gap-2 sm:grid-cols-3">
                {policies.map((policy) => (
                  <TabsTrigger
                    key={policy.id}
                    value={policy.id}
                    className="rounded-full text-sm font-semibold"
                  >
                    {policy.label}
                  </TabsTrigger>
                ))}
              </TabsList>
            </Tabs>

            <div className="flex flex-wrap items-center gap-3">
              {languageOptions.map((option) => {
                const isActive = option.id === activeLanguage;
                return (
                  <button
                    key={option.id}
                    type="button"
                    onClick={() => setActiveLanguage(option.id)}
                    className={`rounded-full border px-4 py-2 text-sm font-semibold transition-all duration-200 ${
                      isActive
                        ? "border-transparent bg-gradient-to-r from-emerald-500 to-lime-500 text-white shadow-lg"
                        : "border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
                    }`}
                  >
                    {option.label}
                  </button>
                );
              })}
            </div>

            <div className="grid gap-6 lg:grid-cols-[280px_1fr]">
              <aside className="space-y-6">
                <div className="rounded-3xl border border-emerald-100 bg-white p-6 shadow-lg">
                  <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-600">
                    Resumen
                  </p>
                  <p className="mt-3 text-2xl font-semibold text-gray-900">
                    {activePolicyData?.label}
                  </p>
                  <p className="mt-2 text-sm text-gray-500">
                    {activePolicyData?.summary}
                  </p>
                  <div className="mt-6 space-y-3">
                    <div className="rounded-2xl bg-emerald-50 px-4 py-3">
                      <p className="text-xs font-semibold uppercase text-emerald-600">Idioma</p>
                      <p className="text-xl font-semibold text-emerald-900">
                        {languageOptions.find((option) => option.id === activeLanguage)?.label}
                      </p>
                    </div>
                    <div className="rounded-2xl bg-gray-50 px-4 py-3">
                      <p className="text-xs font-semibold uppercase text-gray-500">Disponibilidad</p>
                      <p className="mt-1 text-sm font-semibold text-gray-800">
                        {activePolicyData?.languages[activeLanguage]
                          ? "Contenido activo"
                          : "Próximamente"}
                      </p>
                    </div>
                  </div>
                </div>

                <div
                  className={`rounded-3xl bg-gradient-to-br ${
                    activePolicyData?.accent ?? "from-emerald-500 to-lime-400"
                  } p-6 text-white shadow-lg`}
                >
                  <p className="text-xs font-semibold uppercase tracking-[0.3em] text-white/80">
                    Actualización
                  </p>
                  <p className="mt-2 text-2xl font-semibold">2025</p>
                  <p className="mt-2 text-sm text-white/80">
                    Las políticas se revisan periódicamente para mantener su vigencia.
                  </p>
                </div>
              </aside>

              <section className="space-y-6">
                <article className="rounded-3xl border border-gray-200 bg-white p-8 shadow-sm">
                  <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-600">
                    {activePolicyData?.label}
                  </p>
                  <h2 className="mt-3 text-3xl font-semibold text-gray-900">
                    {activeContent.title}
                  </h2>
                  <p className="mt-4 text-base leading-relaxed text-gray-700">
                    {activeContent.intro}
                  </p>
                  {activeContent.items.length > 0 && (
                    <ol className="mt-6 list-decimal space-y-3 pl-5 text-sm text-gray-700">
                      {activeContent.items.map((item, index) => (
                        <li key={`${activeContent.title}-${index}`}>{item}</li>
                      ))}
                    </ol>
                  )}
                  {activeContent.closing && (
                    <p className="mt-6 text-base leading-relaxed text-gray-700">
                      {activeContent.closing}
                    </p>
                  )}
                </article>
              </section>
            </div>
          </div>
        </div>
      </div>
    </SGIDashboardLayout>
  );
}