"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { type User } from "@supabase/supabase-js";
import { CheckCircle2, ChevronLeft, Lock } from "lucide-react";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { supabase } from "@/lib/supabase";

const categoryLabels: Record<string, string> = {
  environment: "Medio Ambiente",
  quality: "Calidad",
  safety: "Seguridad",
  general: "General",
};

const categoryBadgeClasses: Record<string, string> = {
  environment: "bg-emerald-100 text-emerald-700",
  quality: "bg-sky-100 text-sky-700",
  safety: "bg-rose-100 text-rose-700",
  general: "bg-slate-100 text-slate-700",
};

type QuizQuestion = {
  question: string;
  options: string[];
  correctIndex: number;
  points: number;
};

export default function AwarenessModulePage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const moduleId = searchParams.get("moduleId");
  const [user, setUser] = useState<User | null>(null);
  const [moduleData, setModuleData] = useState<any | null>(null);
  const [lessons, setLessons] = useState<any[]>([]);
  const [quizQuestions, setQuizQuestions] = useState<QuizQuestion[]>([]);
  const [progressRecord, setProgressRecord] = useState<any | null>(null);
  const [viewedLessonIds, setViewedLessonIds] = useState<Set<string>>(new Set());
  const [answers, setAnswers] = useState<Record<number, number>>({});
  const [loading, setLoading] = useState(true);
  const [savingQuiz, setSavingQuiz] = useState(false);

  const MAX_MODULE_ATTEMPTS = 2;

  const loadModule = useCallback(async () => {
    if (!moduleId) {
      toast.error("Selecciona un módulo válido.");
      setLoading(false);
      return;
    }

    setLoading(true);
    const { data: { session } } = await supabase.auth.getSession();

    if (!session?.user) {
      router.push("/");
      return;
    }

    setUser(session.user);

    const [moduleResponse, lessonsResponse, quizResponse, progressResponse] = await Promise.all([
      supabase.from("awareness_modules").select("*").eq("id", moduleId).single(),
      supabase
        .from("awareness_lessons")
        .select("*")
        .eq("module_id", moduleId)
        .eq("is_active", true)
        .order("order_index", { ascending: true }),
      supabase.from("awareness_quizzes").select("*").eq("module_id", moduleId).maybeSingle(),
      supabase
        .from("awareness_user_module_progress")
        .select("*")
        .eq("user_id", session.user.id)
        .eq("module_id", moduleId)
        .maybeSingle(),
    ]);

    if (moduleResponse.error) {
      toast.error("No se pudo cargar el módulo.");
      console.error(moduleResponse.error);
      setLoading(false);
      return;
    }

    setModuleData(moduleResponse.data);

    if (lessonsResponse.error) {
      toast.error("No se pudieron cargar las lecciones.");
      console.error(lessonsResponse.error);
    }

    setLessons(lessonsResponse.data ?? []);

    if (quizResponse.error) {
      toast.error("No se pudo cargar el cuestionario.");
      console.error(quizResponse.error);
    }

    const quizData = quizResponse.data?.quiz_json as { questions?: QuizQuestion[] } | undefined;
    setQuizQuestions(quizData?.questions ?? []);

    if (progressResponse.error) {
      toast.error("No se pudo cargar el progreso.");
      console.error(progressResponse.error);
    }

    setProgressRecord(progressResponse.data ?? null);

    const lessonIds = (lessonsResponse.data ?? []).map((lesson) => lesson.id);
    if (lessonIds.length) {
      const viewsResponse = await supabase
        .from("awareness_user_lesson_views")
        .select("lesson_id")
        .eq("user_id", session.user.id)
        .in("lesson_id", lessonIds);

      if (viewsResponse.error) {
        toast.error("No se pudieron cargar las vistas de lecciones.");
        console.error(viewsResponse.error);
      }

      setViewedLessonIds(new Set((viewsResponse.data ?? []).map((view) => view.lesson_id)));
    }

    setLoading(false);
  }, [moduleId, router]);

  useEffect(() => {
    void loadModule();
  }, [loadModule]);

  const totalLessons = lessons.length;
  const viewedLessons = viewedLessonIds.size;
  const progressPercent = totalLessons > 0 ? (viewedLessons / totalLessons) * 100 : 0;
  const quizUnlocked = totalLessons > 0 && viewedLessons === totalLessons;

  const totalQuizScore = useMemo(() => {
    return quizQuestions.reduce((total, question) => total + (question.points ?? 0), 0);
  }, [quizQuestions]);

  const markLessonViewed = async (lessonId: string) => {
    if (!user || viewedLessonIds.has(lessonId) || !moduleId) return;

    const viewedAt = new Date().toISOString();

    const { error: viewError } = await supabase.from("awareness_user_lesson_views").insert({
      user_id: user.id,
      lesson_id: lessonId,
      viewed_at: viewedAt,
    });

    if (viewError) {
      toast.error("No se pudo marcar la lección como vista.");
      console.error(viewError);
      return;
    }

    const nextViewed = new Set(viewedLessonIds);
    nextViewed.add(lessonId);
    setViewedLessonIds(nextViewed);

    const nextViewedCount = nextViewed.size;
    const nextProgressPercent = totalLessons > 0 ? (nextViewedCount / totalLessons) * 100 : 0;

    if (progressRecord) {
      const { error } = await supabase
        .from("awareness_user_module_progress")
        .update({
          status: progressRecord.status === "completed" ? "completed" : "in_progress",
          progress_percent: nextProgressPercent,
          last_updated_at: viewedAt,
        })
        .eq("id", progressRecord.id);

      if (error) {
        toast.error("No se pudo actualizar el progreso.");
        console.error(error);
        return;
      }

      setProgressRecord((prev: any) =>
        prev
          ? {
              ...prev,
              status: prev.status === "completed" ? "completed" : "in_progress",
              progress_percent: nextProgressPercent,
              last_updated_at: viewedAt,
            }
          : prev,
      );
    } else {
      const { data, error } = await supabase
        .from("awareness_user_module_progress")
        .insert({
          user_id: user.id,
          module_id: moduleId,
          status: "in_progress",
          start_date: viewedAt,
          progress_percent: nextProgressPercent,
          attempts_count: 0,
          points_obtained: 0,
          last_updated_at: viewedAt,
        })
        .select("*")
        .single();

      if (error) {
        toast.error("No se pudo registrar el progreso.");
        console.error(error);
        return;
      }

      setProgressRecord(data);
    }
  };

  const submitQuiz = async () => {
    if (!user || !moduleData || !moduleId) return;

    if (!quizUnlocked) {
      toast.error("Completa todas las lecciones antes de enviar el cuestionario.");
      return;
    }

    if ((progressRecord?.attempts_count ?? 0) >= MAX_MODULE_ATTEMPTS) {
      toast.error("Has alcanzado el máximo de 2 intentos para este módulo.");
      return;
    }

    setSavingQuiz(true);

    const earnedPoints = quizQuestions.reduce((total, question, index) => {
      const selected = answers[index];
      if (selected === question.correctIndex) {
        return total + (question.points ?? 0);
      }
      return total;
    }, 0);

    const finalScore = totalQuizScore > 0 ? Math.round((earnedPoints / totalQuizScore) * 100) : 0;
    const pointsObtained = Math.round((moduleData.points_available ?? 0) * finalScore / 100);
    const attemptCount = (progressRecord?.attempts_count ?? 0) + 1;
    const completedAt = new Date().toISOString();

    if (progressRecord) {
      const { error } = await supabase
        .from("awareness_user_module_progress")
        .update({
          status: "completed",
          end_date: completedAt,
          final_score: finalScore,
          attempts_count: attemptCount,
          points_obtained: pointsObtained,
          progress_percent: 100,
          last_updated_at: completedAt,
        })
        .eq("id", progressRecord.id);

      if (error) {
        toast.error("No se pudo guardar la calificación.");
        console.error(error);
        setSavingQuiz(false);
        return;
      }

      setProgressRecord((prev: any) =>
        prev
          ? {
              ...prev,
              status: "completed",
              end_date: completedAt,
              final_score: finalScore,
              attempts_count: attemptCount,
              points_obtained: pointsObtained,
              progress_percent: 100,
              last_updated_at: completedAt,
            }
          : prev,
      );
    } else {
      const { data, error } = await supabase
        .from("awareness_user_module_progress")
        .insert({
          user_id: user.id,
          module_id: moduleId,
          status: "completed",
          start_date: completedAt,
          end_date: completedAt,
          progress_percent: 100,
          final_score: finalScore,
          attempts_count: attemptCount,
          points_obtained: pointsObtained,
          last_updated_at: completedAt,
        })
        .select("*")
        .single();

      if (error) {
        toast.error("No se pudo guardar la calificación.");
        console.error(error);
        setSavingQuiz(false);
        return;
      }

      setProgressRecord(data);
    }

    toast.success("Cuestionario enviado correctamente.");
    setSavingQuiz(false);
  };

  if (loading) {
    return (
      <SGIDashboardLayout>
        <Card>
          <CardContent className="py-10 text-center text-sm text-muted-foreground">
            Cargando módulo...
          </CardContent>
        </Card>
      </SGIDashboardLayout>
    );
  }

  if (!moduleData || !moduleId) {
    return (
      <SGIDashboardLayout>
        <Card>
          <CardContent className="py-10 text-center text-sm text-muted-foreground">
            No se encontró el módulo solicitado.
          </CardContent>
        </Card>
      </SGIDashboardLayout>
    );
  }

  return (
    <SGIDashboardLayout>
      <div className="space-y-6">
        <Button variant="ghost" className="gap-2" onClick={() => router.push("/meiwa-core/meiwa-sgi/support/awareness")}>
          <ChevronLeft className="h-4 w-4" />
          Volver a Awareness
        </Button>

        <Card>
          <CardHeader className="space-y-3">
            <div className="flex flex-wrap items-center justify-between gap-3">
              <Badge className={categoryBadgeClasses[moduleData.category] ?? "bg-muted text-muted-foreground"}>
                {categoryLabels[moduleData.category] ?? moduleData.category}
              </Badge>
              <Badge variant="outline">{moduleData.points_available} pts</Badge>
            </div>
            <div>
              <CardTitle className="text-2xl">{moduleData.title}</CardTitle>
              <p className="text-sm text-muted-foreground">{moduleData.description ?? "Sin descripción"}</p>
            </div>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <div className="flex items-center justify-between text-sm">
                <span>Progreso general</span>
                <span className="font-semibold">{Math.round(progressPercent)}%</span>
              </div>
              <Progress value={progressPercent} />
            </div>
            <div className="grid gap-3 text-sm md:grid-cols-3">
              <div>
                <p className="text-xs text-muted-foreground">Estado</p>
                <p className="font-medium">
                  {progressRecord?.status === "completed"
                    ? "Completado"
                    : viewedLessons > 0
                    ? "En progreso"
                    : "No iniciado"}
                </p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Intentos</p>
                <p className="font-medium">{progressRecord?.attempts_count ?? 0}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Calificación final</p>
                <p className="font-medium">{progressRecord?.final_score ?? "—"}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Lecciones</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {lessons.length === 0 ? (
              <p className="text-sm text-muted-foreground">No hay lecciones activas.</p>
            ) : (
              lessons.map((lesson) => {
                const contentBlocks = Array.isArray(lesson.content_blocks)
                  ? lesson.content_blocks
                  : [];
                return (
                  <Card key={lesson.id} className="border-dashed">
                    <CardHeader className="flex flex-row items-start justify-between gap-4">
                      <div>
                        <CardTitle className="text-base">{lesson.title}</CardTitle>
                        <p className="text-xs text-muted-foreground">
                          {viewedLessonIds.has(lesson.id) ? "Vista" : "Pendiente"}
                        </p>
                      </div>
                      {viewedLessonIds.has(lesson.id) ? (
                        <Badge className="bg-emerald-100 text-emerald-700">
                          <CheckCircle2 className="mr-1 h-3 w-3" />
                          Vista
                        </Badge>
                      ) : (
                        <Button size="sm" onClick={() => markLessonViewed(lesson.id)}>
                          Marcar como vista
                        </Button>
                      )}
                    </CardHeader>
                    <CardContent className="space-y-3 text-sm">
                      {contentBlocks.length === 0 ? (
                        <p className="text-muted-foreground">Sin contenido configurado.</p>
                      ) : (
                        contentBlocks.map((block: any, index: number) => {
                          if (block.type === "text") {
                            return <p key={`block-${lesson.id}-${index}`}>{block.value}</p>;
                          }
                          return (
                            <a
                              key={`block-${lesson.id}-${index}`}
                              href={block.value}
                              target="_blank"
                              rel="noreferrer"
                              className="text-primary underline"
                            >
                              {block.type.toUpperCase()} - {block.value}
                            </a>
                          );
                        })
                      )}
                    </CardContent>
                  </Card>
                );
              })
            )}
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Cuestionario final</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {!quizUnlocked ? (
              <div className="flex items-center gap-2 rounded-lg border border-dashed bg-muted/30 p-4 text-sm text-muted-foreground">
                <Lock className="h-4 w-4" />
                Bloqueado: completa todas las lecciones para desbloquearlo.
              </div>
            ) : quizQuestions.length === 0 ? (
              <p className="text-sm text-muted-foreground">No hay preguntas configuradas.</p>
            ) : (
              <div className="space-y-6">
                {quizQuestions.map((question, index) => (
                  <div key={`quiz-${index}`} className="space-y-3">
                    <div className="flex items-center justify-between">
                      <p className="font-semibold">{question.question}</p>
                      <Badge variant="outline">{question.points} pts</Badge>
                    </div>
                    <RadioGroup
                      value={answers[index]?.toString() ?? ""}
                      onValueChange={(value) =>
                        setAnswers((prev) => ({
                          ...prev,
                          [index]: Number(value),
                        }))
                      }
                      className="gap-3"
                    >
                      {question.options.map((option, optionIndex) => (
                        <label
                          key={`quiz-${index}-option-${optionIndex}`}
                          className="flex items-center gap-2 rounded-md border border-muted px-3 py-2 text-sm"
                        >
                          <RadioGroupItem value={optionIndex.toString()} />
                          {option}
                        </label>
                      ))}
                    </RadioGroup>
                  </div>
                ))}

                {(progressRecord?.attempts_count ?? 0) >= MAX_MODULE_ATTEMPTS ? (
                  <p className="text-sm text-muted-foreground">
                    Alcanzaste el máximo de {MAX_MODULE_ATTEMPTS} intentos permitidos.
                  </p>
                ) : null}
                <Button
                  onClick={submitQuiz}
                  disabled={savingQuiz || (progressRecord?.attempts_count ?? 0) >= MAX_MODULE_ATTEMPTS}
                >
                  {savingQuiz ? "Enviando..." : "Enviar cuestionario"}
                </Button>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </SGIDashboardLayout>
  );
}