"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { BookOpen, GraduationCap, LineChart, Plus, Star } from "lucide-react";
import { type User } from "@supabase/supabase-js";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { Tabs, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "admin2.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

const categoryOptions = [
  { value: "all", label: "Todas" },
  { value: "environment", label: "Medio Ambiente" },
  { value: "quality", label: "Calidad" },
  { value: "safety", label: "Seguridad" },
  { value: "general", label: "General" },
];

const categoryLabels: Record<string, string> = {
  environment: "Medio Ambiente",
  quality: "Calidad",
  safety: "Seguridad",
  general: "General",
};

const categoryBadgeClasses: Record<string, string> = {
  environment: "bg-emerald-100 text-emerald-700",
  quality: "bg-sky-100 text-sky-700",
  safety: "bg-rose-100 text-rose-700",
  general: "bg-slate-100 text-slate-700",
};

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  return new Date(value).toLocaleDateString("es-MX");
};

const formatPercent = (value: number) => `${Math.round(value)}%`;

const formatScore = (value?: number | null) => (value == null ? "—" : `${Math.round(value)}`);

const formatPoints = (value?: number | null) => (value == null ? 0 : value);

export default function AwarenessPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [modules, setModules] = useState<any[]>([]);
  const [lessons, setLessons] = useState<any[]>([]);
  const [progress, setProgress] = useState<any[]>([]);
  const [lessonViews, setLessonViews] = useState<any[]>([]);
  const [categoryFilter, setCategoryFilter] = useState("all");

  const canManage = allowedManagers.has(user?.email ?? "");

  const fetchData = useCallback(async () => {
    setLoading(true);
    const { data: { session } } = await supabase.auth.getSession();

    if (!session?.user) {
      router.push("/");
      return;
    }

    setUser(session.user);
    const canManageModules = allowedManagers.has(session.user.email ?? "");

    let moduleQuery = supabase
      .from("awareness_modules")
      .select("*")
      .order("order_index", { ascending: true, nullsFirst: true });

    if (!canManageModules) {
      moduleQuery = moduleQuery.eq("is_active", true);
    }

    const moduleResponse = await moduleQuery;
    if (moduleResponse.error) {
      toast.error("No se pudieron cargar los módulos.");
      console.error(moduleResponse.error);
      setLoading(false);
      return;
    }

    const modulesData = moduleResponse.data ?? [];
    setModules(modulesData);

    const moduleIds = modulesData.map((module) => module.id);

    const [lessonResponse, progressResponse] = await Promise.all([
      moduleIds.length
        ? supabase
            .from("awareness_lessons")
            .select("*")
            .in("module_id", moduleIds)
            .eq("is_active", true)
            .order("order_index", { ascending: true })
        : Promise.resolve({ data: [], error: null }),
      supabase
        .from("awareness_user_module_progress")
        .select("*")
        .eq("user_id", session.user.id),
    ]);

    if (lessonResponse.error) {
      toast.error("No se pudieron cargar las lecciones.");
      console.error(lessonResponse.error);
    }

    if (progressResponse.error) {
      toast.error("No se pudo cargar el progreso.");
      console.error(progressResponse.error);
    }

    const lessonData = lessonResponse.data ?? [];
    setLessons(lessonData);
    setProgress(progressResponse.data ?? []);

    const lessonIds = lessonData.map((lesson) => lesson.id);
    if (lessonIds.length) {
      const viewsResponse = await supabase
        .from("awareness_user_lesson_views")
        .select("lesson_id, viewed_at")
        .eq("user_id", session.user.id)
        .in("lesson_id", lessonIds);

      if (viewsResponse.error) {
        toast.error("No se pudieron cargar las vistas de lecciones.");
        console.error(viewsResponse.error);
      }

      setLessonViews(viewsResponse.data ?? []);
    } else {
      setLessonViews([]);
    }

    setLoading(false);
  }, [router]);

  useEffect(() => {
    void fetchData();
  }, [fetchData]);

  const lessonCountByModule = useMemo(() => {
    return lessons.reduce<Record<string, number>>((acc, lesson) => {
      acc[lesson.module_id] = (acc[lesson.module_id] ?? 0) + 1;
      return acc;
    }, {});
  }, [lessons]);

  const viewedCountByModule = useMemo(() => {
    const lessonLookup = lessons.reduce<Record<string, string>>((acc, lesson) => {
      acc[lesson.id] = lesson.module_id;
      return acc;
    }, {});

    return lessonViews.reduce<Record<string, number>>((acc, view) => {
      const moduleId = lessonLookup[view.lesson_id];
      if (!moduleId) return acc;
      acc[moduleId] = (acc[moduleId] ?? 0) + 1;
      return acc;
    }, {});
  }, [lessonViews, lessons]);

  const progressByModule = useMemo(() => {
    return progress.reduce<Record<string, any>>((acc, record) => {
      acc[record.module_id] = record;
      return acc;
    }, {});
  }, [progress]);

  const modulesWithStatus = useMemo(() => {
    return modules.map((module) => {
      const totalLessons = lessonCountByModule[module.id] ?? 0;
      const viewedLessons = viewedCountByModule[module.id] ?? 0;
      const percent = totalLessons > 0 ? (viewedLessons / totalLessons) * 100 : 0;
      const progressRecord = progressByModule[module.id];

      const status = progressRecord?.status === "completed"
        ? "Completado"
        : viewedLessons > 0
        ? "En progreso"
        : "No iniciado";

      return {
        ...module,
        totalLessons,
        viewedLessons,
        progressPercent: percent,
        status,
        progressRecord,
      };
    });
  }, [modules, lessonCountByModule, viewedCountByModule, progressByModule]);

  const filteredModules = useMemo(() => {
    if (categoryFilter === "all") return modulesWithStatus;
    return modulesWithStatus.filter((module) => module.category === categoryFilter);
  }, [modulesWithStatus, categoryFilter]);

  const kpis = useMemo(() => {
    const availableModules = modulesWithStatus.length;
    const completedModules = modulesWithStatus.filter((module) => module.status === "Completado");

    const avgProgress =
      availableModules === 0
        ? 0
        : modulesWithStatus.reduce((total, module) => total + module.progressPercent, 0) /
          availableModules;

    const weightedScoreNumerator = completedModules.reduce((total, module) => {
      const points = module.points_available ?? 0;
      const score = module.progressRecord?.final_score ?? 0;
      return total + score * points;
    }, 0);

    const weightedScoreDenominator = completedModules.reduce((total, module) => {
      return total + (module.points_available ?? 0);
    }, 0);

    const globalScore = weightedScoreDenominator ? weightedScoreNumerator / weightedScoreDenominator : 0;

    const pointsObtained = modulesWithStatus.reduce((total, module) => {
      return total + (module.progressRecord?.points_obtained ?? 0);
    }, 0);

    const pointsAvailable = modulesWithStatus.reduce((total, module) => {
      return total + (module.points_available ?? 0);
    }, 0);

    return {
      availableModules,
      completedModules: completedModules.length,
      avgProgress,
      globalScore,
      pointsObtained,
      pointsAvailable,
    };
  }, [modulesWithStatus]);

  return (
    <SGIDashboardLayout>
      <div className="space-y-8">
        <div className="flex flex-wrap items-center justify-between gap-4">
          <div>
            <p className="text-sm font-semibold uppercase tracking-[0.2em] text-muted-foreground">
              Toma de Conciencia · Capacitación
            </p>
            <h1 className="text-2xl font-bold text-gray-900">Dashboard de capacitación</h1>
            <p className="text-sm text-muted-foreground">
              Revisa tus módulos, completa lecciones, desbloquea el cuestionario final y gana puntos.
            </p>
          </div>
          <div className="flex flex-wrap gap-3">
            <Button variant="outline" onClick={() => router.push("/meiwa-core/meiwa-sgi/support/awareness/grades")}>
              Ir a calificaciones
            </Button>
            {canManage && (
              <Button onClick={() => router.push("/meiwa-core/meiwa-sgi/support/awareness/modules/new")}>
                <Plus className="mr-2 h-4 w-4" />
                Crear módulo
              </Button>
            )}
          </div>
        </div>

        <div className="grid gap-4 lg:grid-cols-3">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Módulos disponibles</CardTitle>
              <BookOpen className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{kpis.availableModules}</p>
              <p className="text-xs text-muted-foreground">Activos para capacitación</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Módulos completados</CardTitle>
              <GraduationCap className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{kpis.completedModules}</p>
              <p className="text-xs text-muted-foreground">Con calificación final</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Avance total</CardTitle>
              <LineChart className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent className="space-y-2">
              <p className="text-2xl font-semibold">{formatPercent(kpis.avgProgress)}</p>
              <Progress value={kpis.avgProgress} />
            </CardContent>
          </Card>
        </div>

        <div className="grid gap-4 lg:grid-cols-2">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Calificación global</CardTitle>
              <Star className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent className="flex items-end justify-between">
              <p className="text-2xl font-semibold">{formatScore(kpis.globalScore)}</p>
              <span className="text-xs text-muted-foreground">Promedio ponderado</span>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle className="text-sm">Puntos obtenidos</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">
                {kpis.pointsObtained} / {kpis.pointsAvailable}
              </p>
              <p className="text-xs text-muted-foreground">Disponibles vs. obtenidos</p>
            </CardContent>
          </Card>
        </div>

        <Tabs value={categoryFilter} onValueChange={setCategoryFilter} className="w-full">
          <TabsList className="flex flex-wrap gap-2">
            {categoryOptions.map((option) => (
              <TabsTrigger key={option.value} value={option.value}>
                {option.label}
              </TabsTrigger>
            ))}
          </TabsList>
        </Tabs>

        {loading ? (
          <Card>
            <CardContent className="py-8 text-center text-sm text-muted-foreground">
              Cargando módulos...
            </CardContent>
          </Card>
        ) : filteredModules.length === 0 ? (
          <Card>
            <CardContent className="py-8 text-center text-sm text-muted-foreground">
              No hay módulos disponibles en esta categoría.
            </CardContent>
          </Card>
        ) : (
          <div className="grid gap-6 xl:grid-cols-2">
            {filteredModules.map((module) => (
              <Card key={module.id} className="flex h-full flex-col">
                <CardHeader className="space-y-3">
                  <div className="flex flex-wrap items-center justify-between gap-3">
                    <Badge className={categoryBadgeClasses[module.category] ?? "bg-muted text-muted-foreground"}>
                      {categoryLabels[module.category] ?? module.category}
                    </Badge>
                    {!module.is_active && (
                      <Badge variant="outline" className="text-xs">
                        Inactivo
                      </Badge>
                    )}
                  </div>
                  <div>
                    <CardTitle className="text-lg">{module.title}</CardTitle>
                    <p className="text-sm text-muted-foreground">{module.description ?? "Sin descripción"}</p>
                  </div>
                </CardHeader>
                <CardContent className="flex flex-1 flex-col gap-4">
                  <div className="space-y-2">
                    <div className="flex items-center justify-between text-sm">
                      <span>
                        Progreso: {module.viewedLessons}/{module.totalLessons} lecciones
                      </span>
                      <span className="font-semibold">{formatPercent(module.progressPercent)}</span>
                    </div>
                    <Progress value={module.progressPercent} />
                  </div>

                  <div className="grid gap-3 text-sm md:grid-cols-2">
                    <div>
                      <p className="text-xs text-muted-foreground">Estado</p>
                      <p className="font-medium">{module.status}</p>
                    </div>
                    <div>
                      <p className="text-xs text-muted-foreground">Calificación final</p>
                      <p className="font-medium">{formatScore(module.progressRecord?.final_score)}</p>
                    </div>
                    <div>
                      <p className="text-xs text-muted-foreground">Puntos</p>
                      <p className="font-medium">
                        {formatPoints(module.progressRecord?.points_obtained)} / {module.points_available}
                      </p>
                    </div>
                    <div>
                      <p className="text-xs text-muted-foreground">Intentos</p>
                      <p className="font-medium">{module.progressRecord?.attempts_count ?? 0}</p>
                    </div>
                    <div>
                      <p className="text-xs text-muted-foreground">Fecha inicio</p>
                      <p className="font-medium">{formatDate(module.progressRecord?.start_date)}</p>
                    </div>
                    <div>
                      <p className="text-xs text-muted-foreground">Fecha término</p>
                      <p className="font-medium">{formatDate(module.progressRecord?.end_date)}</p>
                    </div>
                  </div>

                  <div className="mt-auto flex flex-wrap gap-2">
                    <Button
                      onClick={() =>
                        router.push(`/meiwa-core/meiwa-sgi/support/awareness/modules/view?moduleId=${module.id}`)
                      }
                    >
                      {module.status === "No iniciado"
                        ? "Iniciar"
                        : module.status === "En progreso"
                        ? "Continuar"
                        : "Ver"}
                    </Button>
                    {canManage && (
                      <Button
                        variant="outline"
                        onClick={() =>
                          router.push(`/meiwa-core/meiwa-sgi/support/awareness/modules/edit?moduleId=${module.id}`)
                        }
                      >
                        Editar
                      </Button>
                    )}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>
    </SGIDashboardLayout>
  );
}