"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { type User } from "@supabase/supabase-js";
import { Download, Pencil, Plus, RefreshCcw, Trash2 } from "lucide-react";
import {
  PolarAngleAxis,
  PolarGrid,
  PolarRadiusAxis,
  Radar,
  RadarChart,
  ResponsiveContainer,
  Tooltip,
} from "recharts";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "recursoshumanos@meiwa.com.mx",
]);

const dimensionFields = [
  { key: "env_score", label: "Resultado de Medio Ambiente" },
  { key: "quality_score", label: "Resultado de Calidad" },
  { key: "safety_score", label: "Resultado de Seguridad" },
  { key: "s5_score", label: "Resultado de 5S" },
  { key: "incidents_score", label: "Resultado de Incidencias" },
  { key: "productivity_score", label: "Resultado de Productividad" },
  { key: "general_eval_score", label: "Resultado de Evaluación General" },
  { key: "kpi_training_score", label: "Resultado de KPI / Capacitaciones" },
] as const;

const sortOptions = [
  { value: "overall_score", label: "Score general" },
  ...dimensionFields.map((dimension) => ({
    value: dimension.key,
    label: dimension.label,
  })),
];

type DimensionKey = (typeof dimensionFields)[number]["key"];

type CompetenceResult = {
  id: string;
  user_id: string;
  department: string | null;
  position: string | null;
  env_score: number | null;
  quality_score: number | null;
  safety_score: number | null;
  s5_score: number | null;
  incidents_score: number | null;
  productivity_score: number | null;
  general_eval_score: number | null;
  kpi_training_score: number | null;
  overall_score: number | null;
  period: string | null;
  evaluation_date: string | null;
  updated_at: string | null;
};

type ProfileRecord = {
  id: string;
  full_name: string | null;
  email: string | null;
};

type CompetenceFormState = {
  user_id: string;
  department: string;
  position: string;
  period: string;
  evaluation_date: string;
  env_score: string;
  quality_score: string;
  safety_score: string;
  s5_score: string;
  incidents_score: string;
  productivity_score: string;
  general_eval_score: string;
  kpi_training_score: string;
};

const buildEmptyFormState = (userId = ""): CompetenceFormState => ({
  user_id: userId,
  department: "",
  position: "",
  period: "",
  evaluation_date: "",
  env_score: "",
  quality_score: "",
  safety_score: "",
  s5_score: "",
  incidents_score: "",
  productivity_score: "",
  general_eval_score: "",
  kpi_training_score: "",
});

const formatScore = (value: number) => `${Math.round(value)}%`;

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  return new Date(value).toLocaleDateString("es-MX");
};

const parseScoreInput = (value: string) => {
  const parsed = Number(value);
  return Number.isNaN(parsed) ? null : parsed;
};

const getDimensionValue = (record: CompetenceResult, key: DimensionKey) =>
  Number(record[key] ?? 0);

const computeOverallScore = (record: CompetenceResult) => {
  if (record.overall_score != null) return record.overall_score;
  const total = dimensionFields.reduce((sum, dimension) => {
    return sum + getDimensionValue(record, dimension.key);
  }, 0);
  return total / dimensionFields.length;
};

export default function CompetencePage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [profiles, setProfiles] = useState<ProfileRecord[]>([]);
  const [results, setResults] = useState<CompetenceResult[]>([]);
  const [activeUserId, setActiveUserId] = useState<string>("");
  const [viewMode, setViewMode] = useState<"individual" | "department">("individual");
  const [activeDepartment, setActiveDepartment] = useState<string>("");
  const [userFilter, setUserFilter] = useState("all");
  const [departmentFilter, setDepartmentFilter] = useState("all");
  const [positionFilter, setPositionFilter] = useState("all");
  const [searchText, setSearchText] = useState("");
  const [sortField, setSortField] = useState<string>("overall_score");
  const [sortDirection, setSortDirection] = useState<"asc" | "desc">("desc");
  const [selectedRecordId, setSelectedRecordId] = useState<string | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingRecord, setEditingRecord] = useState<CompetenceResult | null>(null);
  const [formState, setFormState] = useState<CompetenceFormState>(buildEmptyFormState());
  const [deleteTarget, setDeleteTarget] = useState<CompetenceResult | null>(null);

  const canManage = allowedManagers.has(user?.email ?? "");

  useEffect(() => {
    const fetchSession = async () => {
      setLoading(true);
      const {
        data: { session },
      } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/");
        return;
      }

      setUser(session.user);

      if (allowedManagers.has(session.user.email ?? "")) {
        const { data, error } = await supabase
          .from("profiles")
          .select("id, full_name, email")
          .order("email", { ascending: true });

        if (error) {
          toast.error("No se pudieron cargar los usuarios.");
          console.error(error);
        }

        setProfiles(data ?? []);
      } else {
        const { data } = await supabase
          .from("profiles")
          .select("id, full_name, email")
          .eq("id", session.user.id)
          .maybeSingle();

        if (data) setProfiles([data]);
      }

      setActiveUserId(session.user.id);
      setLoading(false);
    };

    void fetchSession();
  }, [router]);

  const fetchResults = useCallback(async () => {
    if (!user) return;

    setLoading(true);
    let query = supabase.from("competence_results").select("*");

    if (!canManage) {
      query = query.eq("user_id", user.id);
    }

    const { data, error } = await query.order("updated_at", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar los resultados.");
      console.error(error);
      setLoading(false);
      return;
    }

    setResults((data ?? []) as CompetenceResult[]);
    setLoading(false);
  }, [user, canManage]);

  useEffect(() => {
    void fetchResults();
  }, [fetchResults]);

  useEffect(() => {
    if (!canManage) return;
    if (results.length === 0) return;
    if (!activeUserId) {
      setActiveUserId(results[0]?.user_id ?? "");
      return;
    }
    const hasActive = results.some((result) => result.user_id === activeUserId);
    if (!hasActive) {
      setActiveUserId(results[0]?.user_id ?? "");
    }
  }, [results, activeUserId, canManage]);

  const openCreateDialog = () => {
    const fallbackUserId = activeUserId || profiles[0]?.id || "";
    setEditingRecord(null);
    setFormState(buildEmptyFormState(fallbackUserId));
    setDialogOpen(true);
  };

  const openEditDialog = (record: CompetenceResult) => {
    setEditingRecord(record);
    setFormState({
      user_id: record.user_id,
      department: record.department ?? "",
      position: record.position ?? "",
      period: record.period ?? "",
      evaluation_date: record.evaluation_date ?? "",
      env_score: record.env_score?.toString() ?? "",
      quality_score: record.quality_score?.toString() ?? "",
      safety_score: record.safety_score?.toString() ?? "",
      s5_score: record.s5_score?.toString() ?? "",
      incidents_score: record.incidents_score?.toString() ?? "",
      productivity_score: record.productivity_score?.toString() ?? "",
      general_eval_score: record.general_eval_score?.toString() ?? "",
      kpi_training_score: record.kpi_training_score?.toString() ?? "",
    });
    setDialogOpen(true);
  };

  const selectedRecord = useMemo(() => {
    if (!selectedRecordId) return null;
    return results.find((record) => record.id === selectedRecordId) ?? null;
  }, [results, selectedRecordId]);

  const handleFormChange = (field: keyof CompetenceFormState, value: string) => {
    setFormState((prev) => ({ ...prev, [field]: value }));
  };

  const validateScores = () => {
    const scoreFields: Array<keyof CompetenceFormState> = [
      "env_score",
      "quality_score",
      "safety_score",
      "s5_score",
      "incidents_score",
      "productivity_score",
      "general_eval_score",
      "kpi_training_score",
    ];

    for (const field of scoreFields) {
      const value = parseScoreInput(formState[field]);
      if (value == null || value < 0 || value > 100) {
        toast.error("Todos los resultados deben estar entre 0 y 100.");
        return null;
      }
    }

    return scoreFields.map((field) => parseScoreInput(formState[field]) ?? 0);
  };

  const handleSave = async () => {
    if (!formState.user_id) {
      toast.error("Selecciona un usuario.");
      return;
    }

    const scores = validateScores();
    if (!scores) return;

    if (!editingRecord && formState.period) {
      const duplicate = results.some(
        (record) =>
          record.user_id === formState.user_id &&
          record.period === formState.period
      );
      if (duplicate) {
        toast.error("Ya existe un registro para este usuario y periodo.");
        return;
      }
    }

    const [env_score, quality_score, safety_score, s5_score, incidents_score, productivity_score, general_eval_score, kpi_training_score] =
      scores;
    const overall_score =
      (env_score +
        quality_score +
        safety_score +
        s5_score +
        incidents_score +
        productivity_score +
        general_eval_score +
        kpi_training_score) /
      8;

    const payload = {
      user_id: formState.user_id,
      department: formState.department || null,
      position: formState.position || null,
      period: formState.period || null,
      evaluation_date: formState.evaluation_date || null,
      env_score,
      quality_score,
      safety_score,
      s5_score,
      incidents_score,
      productivity_score,
      general_eval_score,
      kpi_training_score,
      overall_score,
    };

    setLoading(true);
    if (editingRecord) {
      const { error } = await supabase
        .from("competence_results")
        .update(payload)
        .eq("id", editingRecord.id);

      if (error) {
        toast.error("No se pudo actualizar el registro.");
        console.error(error);
        setLoading(false);
        return;
      }
      toast.success("Registro actualizado.");
    } else {
      const { error } = await supabase.from("competence_results").insert(payload);
      if (error) {
        toast.error("No se pudo crear el registro.");
        console.error(error);
        setLoading(false);
        return;
      }
      toast.success("Registro creado.");
    }

    setActiveUserId(formState.user_id);
    setDialogOpen(false);
    setEditingRecord(null);
    setSelectedRecordId(null);
    await fetchResults();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;
    const { error } = await supabase
      .from("competence_results")
      .delete()
      .eq("id", deleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el registro.");
      console.error(error);
      return;
    }

    toast.success("Registro eliminado.");
    setDeleteTarget(null);
    setSelectedRecordId(null);
    await fetchResults();
  };

  const handleExportCsv = () => {
    if (sortedResults.length === 0) {
      toast.error("No hay datos para exportar.");
      return;
    }
    const headers = [
      "Usuario",
      "Departamento",
      "Puesto",
      ...dimensionFields.map((dimension) => dimension.label),
      "Score general",
      "Periodo",
      "Fecha evaluación",
    ];
    const rows = sortedResults.map((record) => {
      const profile = profileLookup[record.user_id];
      return [
        profile?.full_name || profile?.email || record.user_id,
        record.department ?? "",
        record.position ?? "",
        ...dimensionFields.map((dimension) => getDimensionValue(record, dimension.key)),
        computeOverallScore(record).toFixed(2),
        record.period ?? "",
        record.evaluation_date ?? "",
      ];
    });
    const csv = [headers, ...rows]
      .map((row) => row.map((cell) => `"${String(cell).replace(/\"/g, '""')}"`).join(","))
      .join("\n");
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.download = "competence-results.csv";
    link.click();
    URL.revokeObjectURL(url);
  };

  const profileLookup = useMemo(() => {
    return profiles.reduce<Record<string, ProfileRecord>>((acc, profile) => {
      acc[profile.id] = profile;
      return acc;
    }, {});
  }, [profiles]);

  const departmentOptions = useMemo(() => {
    const values = new Set(
      results.map((result) => result.department?.trim()).filter((value): value is string => !!value)
    );
    return Array.from(values).sort((a, b) => a.localeCompare(b));
  }, [results]);

  const positionOptions = useMemo(() => {
    const values = new Set(
      results.map((result) => result.position?.trim()).filter((value): value is string => !!value)
    );
    return Array.from(values).sort((a, b) => a.localeCompare(b));
  }, [results]);

  const filteredResults = useMemo(() => {
    const search = searchText.trim().toLowerCase();
    return results.filter((record) => {
      if (canManage && userFilter !== "all" && record.user_id !== userFilter) {
        return false;
      }

      if (departmentFilter !== "all" && record.department !== departmentFilter) {
        return false;
      }

      if (positionFilter !== "all" && record.position !== positionFilter) {
        return false;
      }

      if (!search) return true;
      const profile = profileLookup[record.user_id];
      const userLabel = `${profile?.full_name ?? ""} ${profile?.email ?? ""}`.toLowerCase();
      const department = record.department?.toLowerCase() ?? "";
      const position = record.position?.toLowerCase() ?? "";
      return (
        userLabel.includes(search) ||
        department.includes(search) ||
        position.includes(search)
      );
    });
  }, [
    results,
    canManage,
    userFilter,
    departmentFilter,
    positionFilter,
    searchText,
    profileLookup,
  ]);

  const sortedResults = useMemo(() => {
    const sorted = [...filteredResults];
    sorted.sort((a, b) => {
      const aValue =
        sortField === "overall_score"
          ? computeOverallScore(a)
          : getDimensionValue(a, sortField as DimensionKey);
      const bValue =
        sortField === "overall_score"
          ? computeOverallScore(b)
          : getDimensionValue(b, sortField as DimensionKey);
      if (aValue === bValue) return 0;
      return sortDirection === "asc" ? aValue - bValue : bValue - aValue;
    });
    return sorted;
  }, [filteredResults, sortField, sortDirection]);

  const departmentSummaries = useMemo(() => {
    const summaryMap = new Map<
      string,
      {
        department: string;
        count: number;
        totals: Record<DimensionKey, number>;
        overallTotal: number;
      }
    >();

    filteredResults.forEach((record) => {
      const department = record.department?.trim() || "Sin departamento";
      const existing = summaryMap.get(department);
      const totals: Record<DimensionKey, number> =
        existing?.totals ??
        dimensionFields.reduce((acc, dimension) => {
          acc[dimension.key] = 0;
          return acc;
        }, {} as Record<DimensionKey, number>);
      const overallValue = computeOverallScore(record);

      dimensionFields.forEach((dimension) => {
        totals[dimension.key] += getDimensionValue(record, dimension.key);
      });

      if (existing) {
        summaryMap.set(department, {
          department,
          count: existing.count + 1,
          totals,
          overallTotal: existing.overallTotal + overallValue,
        });
      } else {
        summaryMap.set(department, {
          department,
          count: 1,
          totals,
          overallTotal: overallValue,
        });
      }
    });

    return Array.from(summaryMap.values())
      .map((summary) => {
        const averageScores = dimensionFields.reduce((acc, dimension) => {
          acc[dimension.key] = summary.totals[dimension.key] / summary.count;
          return acc;
        }, {} as Record<DimensionKey, number>);
        return {
          department: summary.department,
          count: summary.count,
          averageScores,
          overallAverage: summary.overallTotal / summary.count,
        };
      })
      .sort((a, b) => a.department.localeCompare(b.department));
  }, [filteredResults]);

  useEffect(() => {
    if (viewMode !== "department") return;
    if (departmentSummaries.length === 0) {
      setActiveDepartment("");
      return;
    }
    if (!activeDepartment) {
      setActiveDepartment(departmentSummaries[0]?.department ?? "");
      return;
    }
    const hasActive = departmentSummaries.some(
      (summary) => summary.department === activeDepartment
    );
    if (!hasActive) {
      setActiveDepartment(departmentSummaries[0]?.department ?? "");
    }
  }, [departmentSummaries, activeDepartment, viewMode]);

  const activeRecord = useMemo(() => {
    if (!activeUserId) return null;
    return results.find((record) => record.user_id === activeUserId) ?? null;
  }, [results, activeUserId]);

  const activeProfile = useMemo(() => {
    if (!activeRecord) return null;
    return profileLookup[activeRecord.user_id];
  }, [activeRecord, profileLookup]);

  const radarData = useMemo(() => {
    if (!activeRecord) return [];
    return dimensionFields.map((dimension) => ({
      subject: dimension.label,
      value: getDimensionValue(activeRecord, dimension.key),
    }));
  }, [activeRecord]);

  const activeDepartmentSummary = useMemo(() => {
    if (!activeDepartment) return null;
    return (
      departmentSummaries.find((summary) => summary.department === activeDepartment) ??
      null
    );
  }, [departmentSummaries, activeDepartment]);

  const departmentRadarData = useMemo(() => {
    if (!activeDepartmentSummary) return [];
    return dimensionFields.map((dimension) => ({
      subject: dimension.label,
      value: activeDepartmentSummary.averageScores[dimension.key],
    }));
  }, [activeDepartmentSummary]);

  const overallScore = activeRecord ? computeOverallScore(activeRecord) : 0;
  const departmentOverallScore = activeDepartmentSummary?.overallAverage ?? 0;

  const formOverallScore = useMemo(() => {
    const scores = [
      parseScoreInput(formState.env_score),
      parseScoreInput(formState.quality_score),
      parseScoreInput(formState.safety_score),
      parseScoreInput(formState.s5_score),
      parseScoreInput(formState.incidents_score),
      parseScoreInput(formState.productivity_score),
      parseScoreInput(formState.general_eval_score),
      parseScoreInput(formState.kpi_training_score),
    ].filter((value): value is number => value != null);

    if (scores.length !== 8) return 0;
    return scores.reduce((sum, value) => sum + value, 0) / scores.length;
  }, [formState]);

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-slate-50/60 pb-16">
        <section className="rounded-3xl border border-slate-200 bg-white px-6 py-8 shadow-sm sm:px-10">
          <div className="flex flex-wrap items-center justify-between gap-4">
            <div>
              <p className="text-xs font-semibold uppercase tracking-[0.4em] text-emerald-500">
                Soporte · SGI
              </p>
              <h1 className="mt-2 text-3xl font-bold text-slate-900">Competencia</h1>
              <p className="mt-2 max-w-2xl text-sm text-slate-600 sm:text-base">
                Matriz de competencia y desempeño
              </p>
            </div>
            {canManage ? (
              <Badge className="bg-emerald-100 text-emerald-700">Manager</Badge>
            ) : (
              <Badge className="bg-slate-100 text-slate-700">Usuario</Badge>
            )}
          </div>
        </section>

        {canManage && (
          <section className="flex flex-wrap items-center justify-between gap-3 rounded-3xl border border-slate-200 bg-white p-4 shadow-sm">
            <div className="flex flex-wrap items-center gap-2">
              <Button onClick={openCreateDialog} className="gap-2">
                <Plus className="h-4 w-4" />
                Nuevo registro
              </Button>
              <Button
                variant="outline"
                className="gap-2"
                onClick={() => selectedRecord && openEditDialog(selectedRecord)}
                disabled={!selectedRecord}
              >
                <Pencil className="h-4 w-4" />
                Editar
              </Button>
              <Button
                variant="destructive"
                className="gap-2"
                onClick={() => selectedRecord && setDeleteTarget(selectedRecord)}
                disabled={!selectedRecord}
              >
                <Trash2 className="h-4 w-4" />
                Eliminar
              </Button>
            </div>
            <div className="flex flex-wrap items-center gap-2">
              <Button variant="outline" onClick={handleExportCsv} className="gap-2">
                <Download className="h-4 w-4" />
                Exportar CSV
              </Button>
              <Button variant="ghost" onClick={fetchResults} className="gap-2">
                <RefreshCcw className="h-4 w-4" />
                Recargar
              </Button>
            </div>
          </section>
        )}

        <section className="grid gap-4 rounded-3xl border border-slate-200 bg-white p-6 shadow-sm lg:grid-cols-3">
          <div className="space-y-2">
            <p className="text-sm font-semibold text-slate-700">Vista</p>
            <Select
              value={viewMode}
              onValueChange={(value) => setViewMode(value as "individual" | "department")}
            >
              <SelectTrigger>
                <SelectValue placeholder="Selecciona una vista" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="individual">Individual</SelectItem>
                <SelectItem value="department">Departamento</SelectItem>
              </SelectContent>
            </Select>
          </div>

          {viewMode === "individual" ? (
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Usuario activo</p>
              <Select value={activeUserId} onValueChange={setActiveUserId}>
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona un usuario" />
                </SelectTrigger>
                <SelectContent>
                  {profiles.map((profile) => (
                    <SelectItem key={profile.id} value={profile.id}>
                      {profile.full_name || profile.email || profile.id}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          ) : (
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Departamento activo</p>
              <Select value={activeDepartment} onValueChange={setActiveDepartment}>
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona un departamento" />
                </SelectTrigger>
                <SelectContent>
                  {departmentSummaries.map((summary) => (
                    <SelectItem key={summary.department} value={summary.department}>
                      {summary.department}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          )}

          {canManage && (
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Búsqueda rápida</p>
              <Input
                value={searchText}
                onChange={(event) => setSearchText(event.target.value)}
                placeholder="Busca por usuario, departamento o puesto"
              />
            </div>
          )}
        </section>

        <section className="grid gap-6 lg:grid-cols-[2fr,1fr]">
          <Card className="border-slate-200 shadow-sm">
            <CardHeader>
              <CardTitle>Radar de desempeño</CardTitle>
            </CardHeader>
            <CardContent className="h-[320px]">
              {viewMode === "individual" ? (
                activeRecord ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <RadarChart data={radarData}>
                      <PolarGrid />
                      <PolarAngleAxis dataKey="subject" tick={{ fontSize: 11 }} />
                      <PolarRadiusAxis angle={30} domain={[0, 100]} />
                      <Radar
                        name="Resultados"
                        dataKey="value"
                        stroke="#10b981"
                        fill="#34d399"
                        fillOpacity={0.4}
                      />
                      <Tooltip formatter={(value) => `${value}`} />
                    </RadarChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="flex h-full items-center justify-center text-sm text-slate-500">
                    No hay resultados disponibles para mostrar.
                  </div>
                )
              ) : activeDepartmentSummary ? (
                <ResponsiveContainer width="100%" height="100%">
                  <RadarChart data={departmentRadarData}>
                    <PolarGrid />
                    <PolarAngleAxis dataKey="subject" tick={{ fontSize: 11 }} />
                    <PolarRadiusAxis angle={30} domain={[0, 100]} />
                    <Radar
                      name="Resultados"
                      dataKey="value"
                      stroke="#10b981"
                      fill="#34d399"
                      fillOpacity={0.4}
                    />
                    <Tooltip formatter={(value) => `${value}`} />
                  </RadarChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex h-full items-center justify-center text-sm text-slate-500">
                  No hay resultados disponibles para mostrar.
                </div>
              )}
            </CardContent>
          </Card>

          <div className="grid gap-6">
            <Card className="border-slate-200 shadow-sm">
              <CardHeader>
                <CardTitle>Score general</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-4xl font-bold text-emerald-600">
                  {formatScore(viewMode === "individual" ? overallScore : departmentOverallScore)}
                </div>
                <p className="mt-2 text-sm text-slate-500">
                  Promedio de las 8 dimensiones evaluadas.
                </p>
              </CardContent>
            </Card>

            {viewMode === "individual" ? (
              <Card className="border-slate-200 shadow-sm">
                <CardHeader>
                  <CardTitle>Usuario activo</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3 text-sm text-slate-600">
                  <div>
                    <p className="text-xs font-semibold uppercase text-slate-400">
                      Usuario
                    </p>
                    <p className="font-medium text-slate-900">
                      {activeProfile?.full_name || activeProfile?.email || "Sin usuario"}
                    </p>
                    {activeProfile?.email && (
                      <p className="text-xs text-slate-500">{activeProfile.email}</p>
                    )}
                  </div>
                  <div className="grid gap-2 sm:grid-cols-2">
                    <div>
                      <p className="text-xs font-semibold uppercase text-slate-400">
                        Departamento
                      </p>
                      <p className="font-medium text-slate-900">
                        {activeRecord?.department || "Sin departamento"}
                      </p>
                    </div>
                    <div>
                      <p className="text-xs font-semibold uppercase text-slate-400">Puesto</p>
                      <p className="font-medium text-slate-900">
                        {activeRecord?.position || "Sin puesto"}
                      </p>
                    </div>
                    <div>
                      <p className="text-xs font-semibold uppercase text-slate-400">
                        Periodo
                      </p>
                      <p className="font-medium text-slate-900">
                        {activeRecord?.period || formatDate(activeRecord?.evaluation_date)}
                      </p>
                    </div>
                    <div>
                      <p className="text-xs font-semibold uppercase text-slate-400">
                        Actualizado
                      </p>
                      <p className="font-medium text-slate-900">
                        {formatDate(activeRecord?.updated_at)}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card className="border-slate-200 shadow-sm">
                <CardHeader>
                  <CardTitle>Departamento activo</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3 text-sm text-slate-600">
                  <div>
                    <p className="text-xs font-semibold uppercase text-slate-400">
                      Departamento
                    </p>
                    <p className="font-medium text-slate-900">
                      {activeDepartmentSummary?.department || "Sin departamento"}
                    </p>
                  </div>
                  <div className="grid gap-2 sm:grid-cols-2">
                    <div>
                      <p className="text-xs font-semibold uppercase text-slate-400">
                        Colaboradores
                      </p>
                      <p className="font-medium text-slate-900">
                        {activeDepartmentSummary?.count ?? 0}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        </section>

        {canManage && viewMode === "individual" && (
          <section className="grid gap-4 rounded-3xl border border-slate-200 bg-white p-6 shadow-sm">
            <div className="grid gap-4 lg:grid-cols-3">
              <div className="space-y-2">
                <p className="text-sm font-semibold text-slate-700">Usuario</p>
                <Select value={userFilter} onValueChange={setUserFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todos los usuarios" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {profiles.map((profile) => (
                      <SelectItem key={profile.id} value={profile.id}>
                        {profile.full_name || profile.email || profile.id}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <p className="text-sm font-semibold text-slate-700">Departamento</p>
                <Select value={departmentFilter} onValueChange={setDepartmentFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todos los departamentos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {departmentOptions.map((department) => (
                      <SelectItem key={department} value={department}>
                        {department}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <p className="text-sm font-semibold text-slate-700">Puesto</p>
                <Select value={positionFilter} onValueChange={setPositionFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todos los puestos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {positionOptions.map((position) => (
                      <SelectItem key={position} value={position}>
                        {position}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="grid gap-4 lg:grid-cols-3">
              <div className="space-y-2">
                <p className="text-sm font-semibold text-slate-700">Ordenamiento</p>
                <div className="grid gap-2 sm:grid-cols-2">
                  <Select value={sortField} onValueChange={setSortField}>
                    <SelectTrigger>
                      <SelectValue placeholder="Campo" />
                    </SelectTrigger>
                    <SelectContent>
                      {sortOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Select
                    value={sortDirection}
                    onValueChange={(value) => setSortDirection(value as "asc" | "desc")}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Dirección" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="desc">Descendente</SelectItem>
                      <SelectItem value="asc">Ascendente</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </div>
          </section>
        )}

        {viewMode === "individual" ? (
          <section className="rounded-3xl border border-slate-200 bg-white p-6 shadow-sm">
            <div className="flex flex-wrap items-center justify-between gap-2">
              <div>
                <h2 className="text-lg font-semibold text-slate-900">Matriz principal</h2>
                <p className="text-sm text-slate-500">
                  {canManage
                    ? "Consulta los resultados por colaborador y dimensión."
                    : "Tus resultados de competencia y desempeño."}
                </p>
              </div>
              <Badge className="bg-slate-100 text-slate-700">
                {loading ? "Cargando" : `${sortedResults.length} registros`}
              </Badge>
            </div>

            <div className="mt-6 overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Usuario</TableHead>
                    <TableHead>Departamento</TableHead>
                    <TableHead>Puesto</TableHead>
                    {dimensionFields.map((dimension) => (
                      <TableHead key={dimension.key}>{dimension.label}</TableHead>
                    ))}
                    <TableHead>Score general</TableHead>
                    <TableHead>Periodo</TableHead>
                    <TableHead>Actualizado</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {sortedResults.length === 0 ? (
                    <TableRow>
                      <TableCell
                        colSpan={dimensionFields.length + 6}
                        className="text-center"
                      >
                        No se encontraron resultados para los filtros actuales.
                      </TableCell>
                    </TableRow>
                  ) : (
                    sortedResults.map((record) => {
                      const profile = profileLookup[record.user_id];
                      const overall = computeOverallScore(record);
                      const isSelected = selectedRecordId === record.id;
                      return (
                        <TableRow
                          key={record.id}
                          onClick={() => {
                            if (!canManage) return;
                            setSelectedRecordId(record.id);
                            setActiveUserId(record.user_id);
                          }}
                          className={
                            canManage
                              ? `cursor-pointer ${isSelected ? "bg-emerald-50" : ""}`
                              : undefined
                          }
                        >
                          <TableCell className="font-medium text-slate-900">
                            {profile?.full_name || profile?.email || record.user_id}
                          </TableCell>
                          <TableCell>{record.department || "Sin departamento"}</TableCell>
                          <TableCell>{record.position || "Sin puesto"}</TableCell>
                          {dimensionFields.map((dimension) => (
                            <TableCell key={dimension.key}>
                              {formatScore(getDimensionValue(record, dimension.key))}
                            </TableCell>
                          ))}
                          <TableCell className="font-semibold text-emerald-600">
                            {formatScore(overall)}
                          </TableCell>
                          <TableCell>
                            {record.period || formatDate(record.evaluation_date)}
                          </TableCell>
                          <TableCell>{formatDate(record.updated_at)}</TableCell>
                        </TableRow>
                      );
                    })
                  )}
                </TableBody>
              </Table>
            </div>
          </section>
        ) : (
          <section className="rounded-3xl border border-slate-200 bg-white p-6 shadow-sm">
            <div className="flex flex-wrap items-center justify-between gap-2">
              <div>
                <h2 className="text-lg font-semibold text-slate-900">
                  Resultados por departamento
                </h2>
                <p className="text-sm text-slate-500">
                  Promedios generales por departamento según los filtros actuales.
                </p>
              </div>
              <Badge className="bg-slate-100 text-slate-700">
                {departmentSummaries.length} departamentos
              </Badge>
            </div>

            <div className="mt-6 overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Departamento</TableHead>
                    <TableHead>Colaboradores</TableHead>
                    {dimensionFields.map((dimension) => (
                      <TableHead key={dimension.key}>{dimension.label}</TableHead>
                    ))}
                    <TableHead>Score general</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {departmentSummaries.length === 0 ? (
                    <TableRow>
                      <TableCell
                        colSpan={dimensionFields.length + 3}
                        className="text-center"
                      >
                        No hay datos disponibles para estos filtros.
                      </TableCell>
                    </TableRow>
                  ) : (
                    departmentSummaries.map((summary) => (
                      <TableRow key={summary.department}>
                        <TableCell className="font-medium text-slate-900">
                          {summary.department}
                        </TableCell>
                        <TableCell>{summary.count}</TableCell>
                        {dimensionFields.map((dimension) => (
                          <TableCell key={dimension.key}>
                            {formatScore(summary.averageScores[dimension.key])}
                          </TableCell>
                        ))}
                        <TableCell className="font-semibold text-emerald-600">
                          {formatScore(summary.overallAverage)}
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            </div>
          </section>
        )}
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>{editingRecord ? "Editar registro" : "Nuevo registro"}</DialogTitle>
          </DialogHeader>

          <div className="grid gap-4 lg:grid-cols-2">
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Usuario</p>
              <Select
                value={formState.user_id}
                onValueChange={(value) => handleFormChange("user_id", value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona un usuario" />
                </SelectTrigger>
                <SelectContent>
                  {profiles.map((profile) => (
                    <SelectItem key={profile.id} value={profile.id}>
                      {profile.full_name || profile.email || profile.id}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Departamento</p>
              <Input
                value={formState.department}
                onChange={(event) => handleFormChange("department", event.target.value)}
                placeholder="Departamento"
              />
            </div>
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Puesto</p>
              <Input
                value={formState.position}
                onChange={(event) => handleFormChange("position", event.target.value)}
                placeholder="Puesto"
              />
            </div>
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Periodo</p>
              <Input
                value={formState.period}
                onChange={(event) => handleFormChange("period", event.target.value)}
                placeholder="Ej. 2024-Q1"
              />
            </div>
            <div className="space-y-2">
              <p className="text-sm font-semibold text-slate-700">Fecha de evaluación</p>
              <Input
                type="date"
                value={formState.evaluation_date}
                onChange={(event) => handleFormChange("evaluation_date", event.target.value)}
              />
            </div>
            <Card className="border-slate-200 bg-slate-50">
              <CardHeader>
                <CardTitle>Score general</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-3xl font-bold text-emerald-600">
                  {formatScore(formOverallScore)}
                </p>
                <p className="mt-1 text-xs text-slate-500">
                  Se calcula automáticamente con el promedio simple.
                </p>
              </CardContent>
            </Card>
          </div>

          <div className="mt-4 grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
            {dimensionFields.map((dimension) => (
              <div key={dimension.key} className="space-y-2">
                <p className="text-xs font-semibold uppercase text-slate-400">
                  {dimension.label}
                </p>
                <Input
                  type="number"
                  min="0"
                  max="100"
                  value={formState[dimension.key]}
                  onChange={(event) =>
                    handleFormChange(dimension.key, event.target.value)
                  }
                />
              </div>
            ))}
          </div>

          <DialogFooter className="mt-6">
            <Button variant="outline" onClick={() => setDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSave} disabled={loading}>
              {editingRecord ? "Guardar cambios" : "Crear registro"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <AlertDialog open={Boolean(deleteTarget)} onOpenChange={() => setDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar este registro?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction className="bg-rose-600 hover:bg-rose-700" onClick={handleDelete}>
              Eliminar
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}