"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import type { User } from "@supabase/supabase-js";
import {
  Calendar,
  Eye,
  FilePlus,
  Filter,
  Pencil,
  Plus,
  Search,
  Trash2,
} from "lucide-react";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";

const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "recursoshumanos@meiwa.com.mx"]);

const docTypeOptions = [
  { value: "IT", label: "Instrucción de Trabajo" },
  { value: "ST", label: "Estándar" },
  { value: "DO", label: "Documento Oficial" },
  { value: "RT", label: "Reporte" },
  { value: "FO", label: "Formato" },
  { value: "PO", label: "Política" },
  { value: "EV", label: "Evaluación" },
  { value: "RG", label: "Registro" },
];

const departmentOptions = [
  { value: "DS", label: "Diseño" },
  { value: "PE", label: "Producción de Ingeniería" },
  { value: "P1", label: "Producción 1" },
  { value: "IN", label: "Inspección" },
  { value: "P2", label: "Producción 2 – Ensamble" },
  { value: "QC", label: "Control de Calidad" },
  { value: "PC", label: "Control de Producción" },
  { value: "MT", label: "Mantenimiento" },
  { value: "RH", label: "Recursos Humanos" },
  { value: "FN", label: "Finanzas" },
  { value: "SH", label: "Seguridad e Higiene" },
  { value: "CM", label: "Comercial" },
  { value: "SI", label: "Sistemas" },
  { value: "SG", label: "Sistemas de Gestión" },
];

const classificationOptions = [
  { value: "SGA", label: "Sistemas de Gestión Ambiental" },
  { value: "SGC", label: "Sistemas de Gestión de Calidad" },
  { value: "SGS", label: "Sistemas de Gestión de Seguridad" },
];

const statusLabels: Record<string, string> = {
  Planeado: "Planeado",
  "En Proceso": "En proceso",
  Completo: "Completo",
  Retrasado: "Retrasado",
  "Sin objetivo": "Sin objetivo",
};

const statusBadgeClasses: Record<string, string> = {
  Planeado: "bg-sky-100 text-sky-700",
  "En Proceso": "bg-amber-100 text-amber-700",
  Completo: "bg-emerald-100 text-emerald-700",
  Retrasado: "bg-rose-100 text-rose-700",
  "Sin objetivo": "bg-slate-100 text-slate-600",
};

type DocumentControlRecord = {
  id: string;
  category: string | null;
  concept: string;
  doc_type: string;
  department: string;
  classification: string;
  sequence_by_type: number;
  last_revision_number: number | null;
  doc_code: string;
  doc_name: string;
  responsible: string | null;
  doc_link: string | null;
  target_date: string | null;
  start_date: string | null;
  issue_date: string | null;
  status: string | null;
  last_review_date: string | null;
  last_change_summary: string | null;
  created_at: string;
  updated_at: string | null;
  created_by: string | null;
  updated_by: string | null;
};

type DocumentRevision = {
  id: string;
  document_id: string;
  revision_number: number;
  revision_date: string;
  change_summary: string;
  created_at: string;
  created_by: string | null;
};

type DocumentFormState = {
  category: string;
  concept: string;
  doc_type: string;
  department: string;
  classification: string;
  sequence_by_type: number | null;
  last_revision_number: number;
  responsible: string;
  doc_link: string;
  target_date: string;
  start_date: string;
  issue_date: string;
  last_review_date: string;
  last_change_summary: string;
};

const emptyDocumentForm: DocumentFormState = {
  category: "",
  concept: "",
  doc_type: "",
  department: "",
  classification: "",
  sequence_by_type: null,
  last_revision_number: 0,
  responsible: "",
  doc_link: "",
  target_date: "",
  start_date: "",
  issue_date: "",
  last_review_date: "",
  last_change_summary: "",
};

const formatDate = (value?: string | null) => {
  if (!value) return "—";
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return "—";
  return parsed.toLocaleDateString("es-MX");
};

const normalizeDateInput = (value?: string | null) => (value ? value.slice(0, 10) : "");

const padNumber = (value: number, length: number) => value.toString().padStart(length, "0");

const buildDocCode = (
  docType: string,
  classification: string,
  department: string,
  sequence: number,
  revision: number,
) => `${docType}-${classification}-${department}-${padNumber(sequence, 3)}-${padNumber(revision, 2)}`;

const buildDocName = (docCode: string, concept: string) => `${docCode} - ${concept}`.trim();

const computeStatus = (targetDate?: string | null, startDate?: string | null, issueDate?: string | null) => {
  if (!targetDate) return "Sin objetivo";
  const today = new Date();
  const target = new Date(`${targetDate}T23:59:59`);

  if (!issueDate && today > target) return "Retrasado";
  if (targetDate && !startDate) return "Planeado";
  if (targetDate && startDate && !issueDate) return "En Proceso";
  if (targetDate && startDate && issueDate) return "Completo";
  return "Sin objetivo";
};

export default function DocumentControlPage() {
  const [user, setUser] = useState<User | null>(null);
  const [canManage, setCanManage] = useState(false);
  const [documents, setDocuments] = useState<DocumentControlRecord[]>([]);
  const [loadingDocuments, setLoadingDocuments] = useState(false);
  const [searchValue, setSearchValue] = useState("");
  const [typeFilter, setTypeFilter] = useState("all");
  const [departmentFilter, setDepartmentFilter] = useState("all");
  const [classificationFilter, setClassificationFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [categoryFilter, setCategoryFilter] = useState("all");
  const [documentDialogOpen, setDocumentDialogOpen] = useState(false);
  const [detailDialogOpen, setDetailDialogOpen] = useState(false);
  const [editingDocument, setEditingDocument] = useState<DocumentControlRecord | null>(null);
  const [documentForm, setDocumentForm] = useState<DocumentFormState>(emptyDocumentForm);
  const [documentDeleteTarget, setDocumentDeleteTarget] = useState<DocumentControlRecord | null>(null);
  const [revisionDialogOpen, setRevisionDialogOpen] = useState(false);
  const [revisionForm, setRevisionForm] = useState({
    revision_date: normalizeDateInput(new Date().toISOString()),
    change_summary: "",
  });
  const [revisions, setRevisions] = useState<DocumentRevision[]>([]);
  const [loadingRevisions, setLoadingRevisions] = useState(false);
  const [sequenceLoading, setSequenceLoading] = useState(false);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      setCanManage(allowedManagers.has(currentUser?.email ?? ""));
    };

    loadSession();

    const { data: subscription } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
      setCanManage(allowedManagers.has(session?.user?.email ?? ""));
    });

    return () => {
      subscription.subscription.unsubscribe();
    };
  }, []);

  const loadDocuments = useCallback(async () => {
    setLoadingDocuments(true);
    const { data, error } = await supabase
      .from("document_control")
      .select("*")
      .order("doc_code", { ascending: true });

    if (error) {
      toast.error("No se pudo cargar la matriz documental.");
      setLoadingDocuments(false);
      return;
    }

    setDocuments(data ?? []);
    setLoadingDocuments(false);
  }, []);

  const loadRevisions = useCallback(async (documentId: string) => {
    setLoadingRevisions(true);
    const { data, error } = await supabase
      .from("document_control_revisions")
      .select("*")
      .eq("document_id", documentId)
      .order("revision_number", { ascending: false });

    if (error) {
      toast.error("No se pudo cargar el historial de revisiones.");
      setLoadingRevisions(false);
      return;
    }

    setRevisions(data ?? []);
    setLoadingRevisions(false);
  }, []);

  useEffect(() => {
    if (user) {
      loadDocuments();
    }
  }, [user, loadDocuments]);

  const categoryOptions = useMemo(() => {
    const categories = new Set<string>();
    documents.forEach((doc) => {
      if (doc.category) categories.add(doc.category);
    });
    return Array.from(categories).sort((a, b) => a.localeCompare(b));
  }, [documents]);

  const responsibleOptions = useMemo(() => {
    const responsibles = new Set<string>();
    documents.forEach((doc) => {
      if (doc.responsible) responsibles.add(doc.responsible);
    });
    return Array.from(responsibles).sort((a, b) => a.localeCompare(b));
  }, [documents]);

  const statusStats = useMemo(() => {
    const total = documents.length;
    return Object.keys(statusLabels).map((statusKey) => {
      const count = documents.filter(
        (doc) => computeStatus(doc.target_date, doc.start_date, doc.issue_date) === statusKey,
      ).length;
      const percent = total ? Math.round((count / total) * 100) : 0;
      return { statusKey, count, percent };
    });
  }, [documents]);

  const filteredDocuments = useMemo(() => {
    const search = searchValue.trim().toLowerCase();
    return documents.filter((doc) => {
      const status = computeStatus(doc.target_date, doc.start_date, doc.issue_date);
      if (typeFilter !== "all" && doc.doc_type !== typeFilter) return false;
      if (departmentFilter !== "all" && doc.department !== departmentFilter) return false;
      if (classificationFilter !== "all" && doc.classification !== classificationFilter) return false;
      if (statusFilter !== "all" && status !== statusFilter) return false;
      if (categoryFilter !== "all" && (doc.category ?? "") !== categoryFilter) return false;
      if (!search) return true;
      return (
        doc.doc_code.toLowerCase().includes(search) ||
        doc.doc_name.toLowerCase().includes(search) ||
        doc.concept.toLowerCase().includes(search) ||
        (doc.responsible ?? "").toLowerCase().includes(search)
      );
    });
  }, [
    documents,
    searchValue,
    typeFilter,
    departmentFilter,
    classificationFilter,
    statusFilter,
    categoryFilter,
  ]);

  const getNextSequence = useCallback(async (docType: string) => {
    if (!docType) return null;
    setSequenceLoading(true);
    const { data, error } = await supabase
      .from("document_control")
      .select("sequence_by_type")
      .eq("doc_type", docType)
      .order("sequence_by_type", { ascending: false })
      .limit(1)
      .maybeSingle();

    setSequenceLoading(false);

    if (error) {
      toast.error("No se pudo obtener el consecutivo del documento.");
      return null;
    }

    return (data?.sequence_by_type ?? 0) + 1;
  }, []);

  const openCreateDialog = () => {
    setEditingDocument(null);
    setDocumentForm(emptyDocumentForm);
    setDocumentDialogOpen(true);
  };

  const openDetailDialog = (doc: DocumentControlRecord) => {
    setEditingDocument(doc);
    setDocumentForm({
      category: doc.category ?? "",
      concept: doc.concept,
      doc_type: doc.doc_type,
      department: doc.department,
      classification: doc.classification,
      sequence_by_type: doc.sequence_by_type,
      last_revision_number: doc.last_revision_number ?? 0,
      responsible: doc.responsible ?? "",
      doc_link: doc.doc_link ?? "",
      target_date: normalizeDateInput(doc.target_date),
      start_date: normalizeDateInput(doc.start_date),
      issue_date: normalizeDateInput(doc.issue_date),
      last_review_date: normalizeDateInput(doc.last_review_date),
      last_change_summary: doc.last_change_summary ?? "",
    });
    setDetailDialogOpen(true);
    loadRevisions(doc.id);
  };

  const closeDetailDialog = () => {
    setDetailDialogOpen(false);
    setEditingDocument(null);
    setRevisions([]);
    setRevisionForm({
      revision_date: normalizeDateInput(new Date().toISOString()),
      change_summary: "",
    });
  };

  const handleDocTypeChange = async (value: string) => {
    setDocumentForm((prev) => ({ ...prev, doc_type: value }));

    if (!editingDocument) {
      const nextSequence = await getNextSequence(value);
      if (nextSequence) {
        setDocumentForm((prev) => ({ ...prev, sequence_by_type: nextSequence }));
      }
    }
  };

  const generatedDocCode = useMemo(() => {
    if (!documentForm.doc_type || !documentForm.classification || !documentForm.department) return "";
    const sequence = documentForm.sequence_by_type ?? 0;
    return buildDocCode(
      documentForm.doc_type,
      documentForm.classification,
      documentForm.department,
      sequence,
      documentForm.last_revision_number,
    );
  }, [documentForm]);

  const generatedDocName = useMemo(() => {
    if (!generatedDocCode || !documentForm.concept) return "";
    return buildDocName(generatedDocCode, documentForm.concept.trim());
  }, [generatedDocCode, documentForm.concept]);

  const saveDocument = async () => {
    if (!user) return;

    if (!editingDocument && !canManage) {
      toast.error("No tienes permisos para crear documentos.");
      return;
    }

    if (canManage) {
      if (
        !documentForm.category.trim() ||
        !documentForm.concept.trim() ||
        !documentForm.doc_type ||
        !documentForm.department ||
        !documentForm.classification
      ) {
        toast.error("Categoría, concepto, tipo, departamento y clasificación son obligatorios.");
        return;
      }
    }

    let sequence = documentForm.sequence_by_type;

    if (!editingDocument) {
      sequence = sequence ?? (await getNextSequence(documentForm.doc_type));
      if (!sequence) {
        toast.error("No se pudo generar el consecutivo del documento.");
        return;
      }
    } else if (canManage && editingDocument.doc_type !== documentForm.doc_type) {
      sequence = await getNextSequence(documentForm.doc_type);
      if (!sequence) {
        toast.error("No se pudo reasignar el consecutivo del documento.");
        return;
      }
    }

    if (sequence === null) {
      toast.error("No se pudo determinar el consecutivo del documento.");
      return;
    }

    const resolvedSequence = sequence;
    const baseTargetDate = canManage
      ? documentForm.target_date || null
      : editingDocument?.target_date ?? null;
    const baseStartDate = documentForm.start_date || null;
    const baseIssueDate = documentForm.issue_date || null;

    const status = computeStatus(baseTargetDate, baseStartDate, baseIssueDate);

    if (canManage) {
      const docCode = buildDocCode(
        documentForm.doc_type,
        documentForm.classification,
        documentForm.department,
        resolvedSequence,
        documentForm.last_revision_number,
      );
      const docName = buildDocName(docCode, documentForm.concept.trim());

      const payload = {
        category: documentForm.category.trim() || null,
        concept: documentForm.concept.trim(),
        doc_type: documentForm.doc_type,
        department: documentForm.department,
        classification: documentForm.classification,
        sequence_by_type: resolvedSequence,
        last_revision_number: documentForm.last_revision_number,
        doc_code: docCode,
        doc_name: docName,
        responsible: documentForm.responsible.trim() || null,
        doc_link: documentForm.doc_link.trim() || null,
        target_date: documentForm.target_date || null,
        start_date: baseStartDate,
        issue_date: baseIssueDate,
        status,
        last_review_date: documentForm.last_review_date || null,
        last_change_summary: documentForm.last_change_summary.trim() || null,
        updated_at: new Date().toISOString(),
        updated_by: user.email ?? null,
      };

      const response = editingDocument
        ? await supabase.from("document_control").update(payload).eq("id", editingDocument.id).select().single()
        : await supabase
            .from("document_control")
            .insert({
              ...payload,
              created_by: user.email ?? null,
            })
            .select()
            .single();

      if (response.error || !response.data) {
        toast.error("No se pudo guardar el documento.");
        return;
      }

      toast.success(editingDocument ? "Documento actualizado." : "Documento creado.");
      setDocumentDialogOpen(false);
      if (detailDialogOpen) {
        setEditingDocument(response.data);
        setDocumentForm({
          category: response.data.category ?? "",
          concept: response.data.concept,
          doc_type: response.data.doc_type,
          department: response.data.department,
          classification: response.data.classification,
          sequence_by_type: response.data.sequence_by_type,
          last_revision_number: response.data.last_revision_number ?? 0,
          responsible: response.data.responsible ?? "",
          doc_link: response.data.doc_link ?? "",
          target_date: normalizeDateInput(response.data.target_date),
          start_date: normalizeDateInput(response.data.start_date),
          issue_date: normalizeDateInput(response.data.issue_date),
          last_review_date: normalizeDateInput(response.data.last_review_date),
          last_change_summary: response.data.last_change_summary ?? "",
        });
      }
      await loadDocuments();
      return;
    }

    if (!editingDocument) return;

    const response = await supabase
      .from("document_control")
      .update({
        start_date: baseStartDate,
        issue_date: baseIssueDate,
        status,
        updated_at: new Date().toISOString(),
        updated_by: user.email ?? null,
      })
      .eq("id", editingDocument.id)
      .select()
      .single();

    if (response.error || !response.data) {
      toast.error("No se pudo guardar las fechas.");
      return;
    }

    toast.success("Fechas actualizadas.");
    setEditingDocument(response.data);
    await loadDocuments();
  };

  const confirmDeleteDocument = async () => {
    if (!documentDeleteTarget) return;

    const { error } = await supabase.from("document_control").delete().eq("id", documentDeleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el documento.");
      return;
    }

    toast.success("Documento eliminado.");
    setDocumentDeleteTarget(null);
    await loadDocuments();
  };

  const createRevision = async () => {
    if (!editingDocument || !user) return;
    if (!canManage) {
      toast.error("No tienes permisos para crear revisiones.");
      return;
    }

    if (!revisionForm.revision_date || !revisionForm.change_summary.trim()) {
      toast.error("Fecha y resumen del cambio son obligatorios.");
      return;
    }

    const nextRevision = (editingDocument.last_revision_number ?? 0) + 1;

    const revisionPayload = {
      document_id: editingDocument.id,
      revision_number: nextRevision,
      revision_date: revisionForm.revision_date,
      change_summary: revisionForm.change_summary.trim(),
      created_by: user.email ?? null,
    };

    const { error: revisionError } = await supabase.from("document_control_revisions").insert(revisionPayload);

    if (revisionError) {
      toast.error("No se pudo registrar la revisión.");
      return;
    }

    const updatedCode = buildDocCode(
      editingDocument.doc_type,
      editingDocument.classification,
      editingDocument.department,
      editingDocument.sequence_by_type,
      nextRevision,
    );
    const updatedName = buildDocName(updatedCode, editingDocument.concept);

    const { data, error: documentError } = await supabase
      .from("document_control")
      .update({
        last_revision_number: nextRevision,
        last_review_date: revisionForm.revision_date,
        last_change_summary: revisionForm.change_summary.trim(),
        doc_code: updatedCode,
        doc_name: updatedName,
        updated_at: new Date().toISOString(),
        updated_by: user.email ?? null,
      })
      .eq("id", editingDocument.id)
      .select()
      .single();

    if (documentError || !data) {
      toast.error("No se pudo actualizar el documento con la revisión.");
      return;
    }

    toast.success("Revisión creada y documento actualizado.");
    setEditingDocument(data);
    setDocumentForm((prev) => ({
      ...prev,
      last_revision_number: data.last_revision_number ?? nextRevision,
      last_review_date: normalizeDateInput(data.last_review_date),
      last_change_summary: data.last_change_summary ?? "",
    }));
    setRevisionDialogOpen(false);
    setRevisionForm({
      revision_date: normalizeDateInput(new Date().toISOString()),
      change_summary: "",
    });
    await loadDocuments();
    await loadRevisions(editingDocument.id);
  };

  const documentStatus = editingDocument
    ? computeStatus(editingDocument.target_date, editingDocument.start_date, editingDocument.issue_date)
    : null;

  return (
    <SGIDashboardLayout>
      <div className="flex flex-col gap-6">
        <div className="flex flex-wrap items-start justify-between gap-4">
          <div>
            <h1 className="text-2xl font-semibold text-gray-900">Control Documental</h1>
            <p className="text-sm text-muted-foreground">
              Administra la matriz documental con control de versiones, vigencia y trazabilidad.
            </p>
          </div>
          {canManage && (
            <Button onClick={openCreateDialog}>
              <Plus className="mr-2 h-4 w-4" />
              Crear documento
            </Button>
          )}
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Estado de documentos</CardTitle>
          </CardHeader>
          <CardContent className="grid gap-3 sm:grid-cols-2 lg:grid-cols-5">
            {statusStats.map((stat) => (
              <div key={stat.statusKey} className="rounded-lg border border-slate-100 bg-white p-3">
                <div className="text-xs text-muted-foreground">{statusLabels[stat.statusKey]}</div>
                <div className="mt-1 text-lg font-semibold">{stat.count}</div>
                <div className="text-xs text-muted-foreground">{stat.percent}% del total</div>
              </div>
            ))}
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div>
              <CardTitle className="text-lg">Matriz de documentos</CardTitle>
              <p className="text-sm text-muted-foreground">
                Filtra por tipo, departamento, clasificación, estado o categoría.
              </p>
            </div>
            <div className="flex w-full flex-col gap-2 sm:w-auto sm:flex-row sm:items-center">
              <div className="relative w-full sm:w-72">
                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                  value={searchValue}
                  onChange={(event) => setSearchValue(event.target.value)}
                  placeholder="Buscar por código, nombre, concepto o responsable"
                  className="pl-9"
                />
              </div>
              <Badge variant="outline" className="self-start">
                {filteredDocuments.length} documentos visibles
              </Badge>
            </div>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-5">
              <Select value={categoryFilter} onValueChange={setCategoryFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Categoría" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas las categorías</SelectItem>
                  {categoryOptions.map((category) => (
                    <SelectItem key={category} value={category}>
                      {category}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={typeFilter} onValueChange={setTypeFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Tipo de documento" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los tipos</SelectItem>
                  {docTypeOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={departmentFilter} onValueChange={setDepartmentFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Departamento" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los departamentos</SelectItem>
                  {departmentOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={classificationFilter} onValueChange={setClassificationFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Clasificación" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas las clasificaciones</SelectItem>
                  {classificationOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Estado" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los estados</SelectItem>
                  {Object.keys(statusLabels).map((statusKey) => (
                    <SelectItem key={statusKey} value={statusKey}>
                      {statusLabels[statusKey]}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-center gap-2 text-sm text-muted-foreground">
              <Filter className="h-4 w-4" />
              Filtros activos por datos clave de la matriz documental.
            </div>

            <div className="overflow-hidden rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Código</TableHead>
                    <TableHead>Nombre</TableHead>
                    <TableHead>Concepto</TableHead>
                    <TableHead>Tipo</TableHead>
                    <TableHead>Departamento</TableHead>
                    <TableHead>Clasificación</TableHead>
                    <TableHead>Responsable</TableHead>
                    <TableHead>Documento vigente</TableHead>
                    <TableHead>Fecha objetivo</TableHead>
                    <TableHead>Estado</TableHead>
                    <TableHead>Última revisión</TableHead>
                    <TableHead className="text-right">Acciones</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {loadingDocuments ? (
                    <TableRow>
                      <TableCell colSpan={12} className="text-center text-sm text-muted-foreground">
                        Cargando documentos...
                      </TableCell>
                    </TableRow>
                  ) : filteredDocuments.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={12} className="text-center text-sm text-muted-foreground">
                        No hay documentos registrados.
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredDocuments.map((doc) => {
                      const status = computeStatus(doc.target_date, doc.start_date, doc.issue_date);
                      return (
                        <TableRow key={doc.id}>
                          <TableCell className="font-medium">{doc.doc_code}</TableCell>
                          <TableCell>{doc.doc_name}</TableCell>
                          <TableCell>{doc.concept}</TableCell>
                          <TableCell>{doc.doc_type}</TableCell>
                          <TableCell>
                            {departmentOptions.find((option) => option.value === doc.department)?.label ?? doc.department}
                          </TableCell>
                          <TableCell>
                            {classificationOptions.find((option) => option.value === doc.classification)?.label ??
                              doc.classification}
                          </TableCell>
                          <TableCell>{doc.responsible || "—"}</TableCell>
                          <TableCell>
                            {doc.doc_link ? (
                              <a
                                href={doc.doc_link}
                                target="_blank"
                                rel="noreferrer"
                                className="text-sm text-sky-600 hover:underline"
                              >
                                Ver documento
                              </a>
                            ) : (
                              "—"
                            )}
                          </TableCell>
                          <TableCell>{formatDate(doc.target_date)}</TableCell>
                          <TableCell>
                            <Badge className={statusBadgeClasses[status] ?? "bg-slate-100 text-slate-600"}>
                              {statusLabels[status] ?? status}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <div className="text-xs text-muted-foreground">
                              {doc.last_revision_number ?? 0}
                              <span className="ml-2">{formatDate(doc.last_review_date)}</span>
                            </div>
                          </TableCell>
                          <TableCell className="text-right">
                            <div className="flex justify-end gap-2">
                              <Button size="icon" variant="ghost" onClick={() => openDetailDialog(doc)}>
                                <Eye className="h-4 w-4" />
                              </Button>
                              {canManage && (
                                <Button size="icon" variant="ghost" onClick={() => openDetailDialog(doc)}>
                                  <Pencil className="h-4 w-4" />
                                </Button>
                              )}
                              {canManage && (
                                <Button size="icon" variant="ghost" onClick={() => setDocumentDeleteTarget(doc)}>
                                  <Trash2 className="h-4 w-4 text-rose-500" />
                                </Button>
                              )}
                            </div>
                          </TableCell>
                        </TableRow>
                      );
                    })
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      </div>

      <Dialog open={documentDialogOpen} onOpenChange={setDocumentDialogOpen}>
        <DialogContent className="max-h-[85vh] max-w-3xl overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{editingDocument ? "Editar documento" : "Crear documento"}</DialogTitle>
            <DialogDescription>
              Registra la información base. El código y nombre se generan automáticamente.
            </DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 md:grid-cols-3">
            <div className="space-y-2">
              <Label>Código generado</Label>
              <Input value={generatedDocCode || "Se genera al guardar"} readOnly />
              {sequenceLoading && (
                <p className="text-xs text-muted-foreground">Calculando consecutivo...</p>
              )}
            </div>
            <div className="space-y-2">
              <Label>Nombre generado</Label>
              <Input value={generatedDocName || "Se genera al guardar"} readOnly />
            </div>
            <div className="space-y-2">
              <Label>Tipo de documento</Label>
              <Select value={documentForm.doc_type} onValueChange={handleDocTypeChange}>
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar" />
                </SelectTrigger>
                <SelectContent>
                  {docTypeOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Categoría</Label>
              <Input
                value={documentForm.category}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, category: event.target.value }))}
                placeholder="Ej. Ambiental"
              />
            </div>
            <div className="space-y-2">
              <Label>Concepto</Label>
              <Input
                value={documentForm.concept}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, concept: event.target.value }))}
                placeholder="Descripción breve del documento"
              />
            </div>
            <div className="space-y-2">
              <Label>Departamento</Label>
              <Select
                value={documentForm.department}
                onValueChange={(value) => setDocumentForm((prev) => ({ ...prev, department: value }))}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar" />
                </SelectTrigger>
                <SelectContent>
                  {departmentOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Clasificación</Label>
              <Select
                value={documentForm.classification}
                onValueChange={(value) => setDocumentForm((prev) => ({ ...prev, classification: value }))}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar" />
                </SelectTrigger>
                <SelectContent>
                  {classificationOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Responsable (solo managers)</Label>
              <Input
                value={documentForm.responsible}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, responsible: event.target.value }))}
                placeholder="Nombre del responsable"
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label>Link documento vigente</Label>
              <Input
                value={documentForm.doc_link}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, doc_link: event.target.value }))}
                placeholder="https://..."
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha objetivo (solo managers)</Label>
              <Input
                type="date"
                value={documentForm.target_date}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, target_date: event.target.value }))}
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha de inicio</Label>
              <Input
                type="date"
                value={documentForm.start_date}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, start_date: event.target.value }))}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha de emisión</Label>
              <Input
                type="date"
                value={documentForm.issue_date}
                onChange={(event) => setDocumentForm((prev) => ({ ...prev, issue_date: event.target.value }))}
              />
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setDocumentDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={saveDocument}>
              <FilePlus className="mr-2 h-4 w-4" />
              {editingDocument ? "Guardar cambios" : "Crear documento"}
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={detailDialogOpen} onOpenChange={(open) => (!open ? closeDetailDialog() : null)}>
        <DialogContent className="max-h-[85vh] max-w-5xl overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Detalle del documento</DialogTitle>
            <DialogDescription>
              Consulta los datos generales, fechas, estado automático y trazabilidad de revisiones.
            </DialogDescription>
          </DialogHeader>
          {editingDocument && (
            <div className="grid gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Datos generales</CardTitle>
                </CardHeader>
                <CardContent className="grid gap-4 md:grid-cols-3">
                  <div className="space-y-2">
                    <Label>Código de documento</Label>
                    <Input value={editingDocument.doc_code} readOnly />
                  </div>
                  <div className="space-y-2">
                    <Label>Nombre del documento</Label>
                    <Input value={editingDocument.doc_name} readOnly />
                  </div>
                  <div className="space-y-2">
                    <Label>Tipo de documento</Label>
                    <Select
                      value={documentForm.doc_type}
                      onValueChange={(value) => setDocumentForm((prev) => ({ ...prev, doc_type: value }))}
                      disabled={!canManage}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Seleccionar" />
                      </SelectTrigger>
                      <SelectContent>
                        {docTypeOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label>Categoría</Label>
                    <Input
                      value={documentForm.category}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, category: event.target.value }))}
                      disabled={!canManage}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Concepto</Label>
                    <Input
                      value={documentForm.concept}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, concept: event.target.value }))}
                      disabled={!canManage}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Departamento</Label>
                    <Select
                      value={documentForm.department}
                      onValueChange={(value) => setDocumentForm((prev) => ({ ...prev, department: value }))}
                      disabled={!canManage}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Seleccionar" />
                      </SelectTrigger>
                      <SelectContent>
                        {departmentOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label>Clasificación</Label>
                    <Select
                      value={documentForm.classification}
                      onValueChange={(value) => setDocumentForm((prev) => ({ ...prev, classification: value }))}
                      disabled={!canManage}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Seleccionar" />
                      </SelectTrigger>
                      <SelectContent>
                        {classificationOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label>Responsable</Label>
                    <Input
                      value={documentForm.responsible}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, responsible: event.target.value }))}
                      disabled={!canManage}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Link documento vigente</Label>
                    <Input
                      value={documentForm.doc_link}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, doc_link: event.target.value }))}
                      disabled={!canManage}
                    />
                  </div>
                </CardContent>
              </Card>
              <Card>        
                <CardHeader>
                  <CardTitle className="text-base">Fechas y estado</CardTitle>
                </CardHeader>
                <CardContent className="grid gap-4 sm:grid-cols-2 lg:grid-cols-7">
                  <div className="space-y-2">
                    <Label>Fecha objetivo</Label>
                    <Input
                      type="date"
                      value={documentForm.target_date}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, target_date: event.target.value }))}
                      disabled={!canManage}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Fecha de inicio</Label>
                    <Input
                      type="date"
                      value={documentForm.start_date}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, start_date: event.target.value }))}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Fecha de emisión</Label>
                    <Input
                      type="date"
                      value={documentForm.issue_date}
                      onChange={(event) => setDocumentForm((prev) => ({ ...prev, issue_date: event.target.value }))}
                    />
                  </div>
                  <div className="rounded-lg border border-dashed border-slate-200 bg-slate-50 p-3">
                    <div className="flex items-center gap-2 text-xs text-muted-foreground">
                      <Calendar className="h-4 w-4" />
                      Estado calculado
                    </div>
                    <Badge className={`mt-2 ${statusBadgeClasses[documentStatus ?? ""] ?? "bg-slate-100"}`}>
                      {documentStatus ? statusLabels[documentStatus] : "Sin objetivo"}
                    </Badge>
                  </div>
                  <div className="space-y-2">
                    <Label>Última revisión</Label>
                    <Input value={formatDate(editingDocument.last_review_date)} readOnly />
                  </div>
                  <div className="space-y-2 lg:col-span-2">
                    <Label>Último cambio</Label>
                    <Textarea value={editingDocument.last_change_summary ?? "Sin cambios registrados"} readOnly />
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-col gap-2 lg:flex-row lg:items-center lg:justify-between">
                  <div>
                    <CardTitle className="text-base">Historial de revisiones</CardTitle>
                    <p className="text-xs text-muted-foreground">
                      Registro de cambios, revisión vigente y trazabilidad.
                    </p>
                  </div>
                  {canManage && (
                    <Button variant="outline" onClick={() => setRevisionDialogOpen(true)}>
                      <FilePlus className="mr-2 h-4 w-4" />
                      Crear revisión
                    </Button>
                  )}
                </CardHeader>
                <CardContent className="space-y-3">
                  <div className="overflow-hidden rounded-lg border">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Revisión</TableHead>
                          <TableHead>Fecha</TableHead>
                          <TableHead>Vigente</TableHead>
                          <TableHead>Último cambio</TableHead>
                          <TableHead>Creado por</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {loadingRevisions ? (
                          <TableRow>
                            <TableCell colSpan={5} className="text-center text-sm text-muted-foreground">
                              Cargando revisiones...
                            </TableCell>
                          </TableRow>
                        ) : revisions.length === 0 ? (
                          <TableRow>
                            <TableCell colSpan={5} className="text-center text-sm text-muted-foreground">
                              No hay revisiones registradas.
                            </TableCell>
                          </TableRow>
                        ) : (
                          revisions.map((revision) => (
                            <TableRow key={revision.id}>
                              <TableCell className="font-medium">{padNumber(revision.revision_number, 2)}</TableCell>
                              <TableCell>{formatDate(revision.revision_date)}</TableCell>
                              <TableCell>
                                {revision.revision_number === (editingDocument.last_revision_number ?? 0) ? (
                                  <Badge variant="outline">Vigente</Badge>
                                ) : (
                                  "—"
                                )}
                              </TableCell>
                              <TableCell>{revision.change_summary}</TableCell>
                              <TableCell>{revision.created_by ?? "—"}</TableCell>
                            </TableRow>
                          ))
                        )}
                      </TableBody>
                    </Table>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}
          <div className="flex flex-wrap justify-end gap-2">
            <Button variant="outline" onClick={closeDetailDialog}>
              Cerrar
            </Button>
            {editingDocument && (
              <Button onClick={saveDocument}>{canManage ? "Guardar cambios" : "Guardar fechas"}</Button>
            )}
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={revisionDialogOpen} onOpenChange={setRevisionDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>Crear revisión</DialogTitle>
            <DialogDescription>Registra la revisión vigente y el cambio aplicado.</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>Fecha de última revisión</Label>
              <Input
                type="date"
                value={revisionForm.revision_date}
                onChange={(event) =>
                  setRevisionForm((prev) => ({ ...prev, revision_date: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2">
              <Label>Último cambio realizado</Label>
              <Textarea
                value={revisionForm.change_summary}
                onChange={(event) =>
                  setRevisionForm((prev) => ({ ...prev, change_summary: event.target.value }))
                }
                placeholder="Describe el cambio realizado en esta revisión"
              />
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setRevisionDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={createRevision}>
              <FilePlus className="mr-2 h-4 w-4" />
              Guardar revisión
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      <AlertDialog open={Boolean(documentDeleteTarget)} onOpenChange={() => setDocumentDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar este documento?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDeleteDocument}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}