"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Checkbox } from "@/components/ui/checkbox";
import { FileText, Filter, Pencil, Plus, Search, Trash2, Upload } from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);

type ResourceRecord = {
  id: string;
  category: string | null;
  resource_type: string;
  resource_name: string;
  management_system_use: string;
  process_activity: string | null;
  responsible: string | null;
  evidence_url: string | null;
  evidence_path: string | null;
  status: string;
  is_critical: boolean | null;
  linked_to_legal: boolean | null;
  created_at: string;
  updated_at: string | null;
  created_by: string | null;
  updated_by: string | null;
};

type ProcessRecord = {
  id: string;
  process_name: string;
  is_active: boolean | null;
};

const resourceTypeOptions = [
  { value: "human", label: "Humanos" },
  { value: "financial", label: "Financieros" },
  { value: "technical", label: "Materiales / Técnicos" },
  { value: "organizational", label: "Organizacionales" },
];

const statusOptions = [
  { value: "current", label: "Vigente" },
  { value: "improving", label: "En mejora" },
  { value: "implementing", label: "En implementación" },
  { value: "to_update", label: "Por actualizar" },
];

const statusBadgeClasses: Record<string, string> = {
  current: "bg-emerald-100 text-emerald-700",
  improving: "bg-amber-100 text-amber-700",
  implementing: "bg-sky-100 text-sky-700",
  to_update: "bg-rose-100 text-rose-700",
};

const emptyResourceForm = {
  category: "",
  resource_type: "",
  resource_name: "",
  management_system_use: "",
  process_activity: "",
  responsible: "",
  evidence_url: "",
  status: "",
  is_critical: false,
  linked_to_legal: false,
};

const emptyProcessForm = {
  process_name: "",
  is_active: true,
};

const formatPercent = (value: number) => `${Math.round(value)}%`;

const getPublicEvidenceUrl = (path: string) => {
  const { data } = supabase.storage.from("sga-resources-evidence").getPublicUrl(path);
  return data.publicUrl;
};

export default function ResourcesPage() {
  const [user, setUser] = useState<User | null>(null);
  const [canManage, setCanManage] = useState(false);
  const [resources, setResources] = useState<ResourceRecord[]>([]);
  const [processes, setProcesses] = useState<ProcessRecord[]>([]);
  const [loadingResources, setLoadingResources] = useState(false);
  const [loadingProcesses, setLoadingProcesses] = useState(false);
  const [searchValue, setSearchValue] = useState("");
  const [typeFilter, setTypeFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [categoryFilter, setCategoryFilter] = useState("all");
  const [processFilter, setProcessFilter] = useState("all");
  const [responsibleFilter, setResponsibleFilter] = useState("all");
  const [criticalFilter, setCriticalFilter] = useState("all");
  const [legalFilter, setLegalFilter] = useState("all");
  const [resourceDialogOpen, setResourceDialogOpen] = useState(false);
  const [editingResource, setEditingResource] = useState<ResourceRecord | null>(null);
  const [resourceForm, setResourceForm] = useState(emptyResourceForm);
  const [resourceFile, setResourceFile] = useState<File | null>(null);
  const [resourceDeleteTarget, setResourceDeleteTarget] = useState<ResourceRecord | null>(null);
  const [processDialogOpen, setProcessDialogOpen] = useState(false);
  const [processForm, setProcessForm] = useState(emptyProcessForm);
  const [editingProcess, setEditingProcess] = useState<ProcessRecord | null>(null);
  const [processDeleteTarget, setProcessDeleteTarget] = useState<ProcessRecord | null>(null);
  const [processTotalOverride, setProcessTotalOverride] = useState("");

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      setCanManage(allowedManagers.has(currentUser?.email ?? ""));
    };

    loadSession();

    const { data: subscription } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
      setCanManage(allowedManagers.has(session?.user?.email ?? ""));
    });

    return () => {
      subscription.subscription.unsubscribe();
    };
  }, []);

  useEffect(() => {
    if (typeof window === "undefined") return;
    const storedValue = window.localStorage.getItem("sgm_process_total");
    if (storedValue) {
      setProcessTotalOverride(storedValue);
    }
  }, []);

  useEffect(() => {
    if (typeof window === "undefined") return;
    window.localStorage.setItem("sgm_process_total", processTotalOverride);
  }, [processTotalOverride]);

  const loadResources = useCallback(async () => {
    setLoadingResources(true);
    const { data, error } = await supabase
      .from("sgm_resources")
      .select("*")
      .order("resource_type", { ascending: true })
      .order("resource_name", { ascending: true });

    if (error) {
      toast.error("No se pudo cargar la matriz de recursos.");
      setLoadingResources(false);
      return;
    }

    setResources(data ?? []);
    setLoadingResources(false);
  }, []);

  const loadProcesses = useCallback(async () => {
    setLoadingProcesses(true);
    const { data, error } = await supabase
      .from("sgm_processes")
      .select("*")
      .order("process_name", { ascending: true });

    if (error) {
      toast.error("No se pudieron cargar los procesos del SGA.");
      setLoadingProcesses(false);
      return;
    }

    setProcesses(data ?? []);
    setLoadingProcesses(false);
  }, []);

  useEffect(() => {
    if (user) {
      loadResources();
      loadProcesses();
    }
  }, [user, loadResources, loadProcesses]);

  const processOptions = useMemo(() => {
    const names = new Set<string>();
    processes.forEach((process) => {
      if (process.process_name) names.add(process.process_name);
    });
    resources.forEach((resource) => {
      if (resource.process_activity) names.add(resource.process_activity);
    });
    return Array.from(names).sort((a, b) => a.localeCompare(b));
  }, [processes, resources]);

  const responsibleOptions = useMemo(() => {
    const names = new Set<string>();
    resources.forEach((resource) => {
      if (resource.responsible) names.add(resource.responsible);
    });
    return Array.from(names).sort((a, b) => a.localeCompare(b));
  }, [resources]);

  const categoryOptions = useMemo(() => {
    const categories = new Set<string>();
    resources.forEach((resource) => {
      if (resource.category) categories.add(resource.category);
    });
    return Array.from(categories).sort((a, b) => a.localeCompare(b));
  }, [resources]);

  const filteredResources = useMemo(() => {
    const search = searchValue.trim().toLowerCase();
    return resources.filter((resource) => {
      if (categoryFilter !== "all" && (resource.category ?? "") !== categoryFilter) return false;
      if (typeFilter !== "all" && resource.resource_type !== typeFilter) return false;
      if (statusFilter !== "all" && resource.status !== statusFilter) return false;
      if (processFilter !== "all" && (resource.process_activity ?? "") !== processFilter) return false;
      if (responsibleFilter !== "all" && (resource.responsible ?? "") !== responsibleFilter) return false;
      if (criticalFilter !== "all") {
        const isCritical = Boolean(resource.is_critical);
        if (criticalFilter === "yes" && !isCritical) return false;
        if (criticalFilter === "no" && isCritical) return false;
      }
      if (legalFilter !== "all") {
        const isLinked = Boolean(resource.linked_to_legal);
        if (legalFilter === "yes" && !isLinked) return false;
        if (legalFilter === "no" && isLinked) return false;
      }
      if (!search) return true;
      return (
        resource.resource_name.toLowerCase().includes(search) ||
        resource.management_system_use.toLowerCase().includes(search) ||
        (resource.category ?? "").toLowerCase().includes(search) ||
        (resource.process_activity ?? "").toLowerCase().includes(search) ||
        (resource.responsible ?? "").toLowerCase().includes(search)
      );
    });
  }, [
    resources,
    categoryFilter,
    typeFilter,
    statusFilter,
    processFilter,
    responsibleFilter,
    criticalFilter,
    legalFilter,
    searchValue,
  ]);

  const totalResources = resources.length;
  const typeStats = resourceTypeOptions.map((option) => {
    const count = resources.filter((resource) => resource.resource_type === option.value).length;
    const percent = totalResources ? (count / totalResources) * 100 : 0;
    return { ...option, count, percent };
  });
  const statusStats = statusOptions.map((option) => {
    const count = resources.filter((resource) => resource.status === option.value).length;
    const percent = totalResources ? (count / totalResources) * 100 : 0;
    return { ...option, count, percent };
  });
  const activeProcesses = processes.filter((process) => process.is_active ?? true);
  const totalProcesses = activeProcesses.length || Number.parseInt(processTotalOverride, 10) || 0;
  const processesWithResources = new Set(
    resources.map((resource) => resource.process_activity).filter(Boolean) as string[],
  ).size;
  const coveragePercent = totalProcesses ? (processesWithResources / totalProcesses) * 100 : 0;
  const criticalNotCurrent = resources.filter(
    (resource) => resource.is_critical && resource.status !== "current",
  ).length;
  const implementingLegal = resources.filter(
    (resource) => resource.status === "implementing" && resource.linked_to_legal,
  ).length;

  const openResourceDialog = (resource?: ResourceRecord) => {
    if (resource) {
      setEditingResource(resource);
      setResourceForm({
        category: resource.category ?? "",
        resource_type: resource.resource_type,
        resource_name: resource.resource_name,
        management_system_use: resource.management_system_use,
        process_activity: resource.process_activity ?? "",
        responsible: resource.responsible ?? "",
        evidence_url: resource.evidence_url ?? "",
        status: resource.status,
        is_critical: Boolean(resource.is_critical),
        linked_to_legal: Boolean(resource.linked_to_legal),
      });
    } else {
      setEditingResource(null);
      setResourceForm(emptyResourceForm);
    }
    setResourceFile(null);
    setResourceDialogOpen(true);
  };

  const uploadEvidence = async (resourceId: string, file: File) => {
    const filePath = `${resourceId}/${Date.now()}-${file.name}`;
    const { error } = await supabase.storage
      .from("sga-resources-evidence")
      .upload(filePath, file, { upsert: true });

    if (error) {
      toast.error("No se pudo subir la evidencia.");
      return null;
    }
    return filePath;
  };

  const saveResource = async () => {
    if (!canManage) {
      toast.error("No tienes permisos para gestionar recursos.");
      return;
    }

    if (
      !resourceForm.resource_type ||
      !resourceForm.resource_name ||
      !resourceForm.management_system_use ||
      !resourceForm.status
    ) {
      toast.error("Tipo, recurso, uso en el sistema de gestión y estado son obligatorios.");
      return;
    }

    const payload = {
      category: resourceForm.category.trim() || null,
      resource_type: resourceForm.resource_type,
      resource_name: resourceForm.resource_name.trim(),
      management_system_use: resourceForm.management_system_use.trim(),
      process_activity: resourceForm.process_activity.trim() || null,
      responsible: resourceForm.responsible.trim() || null,
      evidence_url: resourceForm.evidence_url.trim() || null,
      status: resourceForm.status,
      is_critical: resourceForm.is_critical,
      linked_to_legal: resourceForm.linked_to_legal,
      updated_at: new Date().toISOString(),
      updated_by: user?.email ?? null,
    };

    const response = editingResource
      ? await supabase.from("sgm_resources").update(payload).eq("id", editingResource.id).select().single()
      : await supabase
          .from("sgm_resources")
          .insert({ ...payload, created_by: user?.email ?? null })
          .select()
          .single();

    if (response.error || !response.data) {
      toast.error("No se pudo guardar el recurso.");
      return;
    }

    if (resourceFile) {
      const filePath = await uploadEvidence(response.data.id, resourceFile);
      if (filePath) {
        const { error: updateError } = await supabase
          .from("sgm_resources")
          .update({ evidence_path: filePath, updated_at: new Date().toISOString(), updated_by: user?.email ?? null })
          .eq("id", response.data.id);

        if (updateError) {
          toast.error("No se pudo registrar la evidencia.");
        }
      }
    }

    toast.success(editingResource ? "Recurso actualizado." : "Recurso agregado.");
    setResourceDialogOpen(false);
    await loadResources();
  };

  const confirmDeleteResource = async () => {
    if (!resourceDeleteTarget) return;
    const { error } = await supabase.from("sgm_resources").delete().eq("id", resourceDeleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el recurso.");
      return;
    }

    toast.success("Recurso eliminado.");
    setResourceDeleteTarget(null);
    await loadResources();
  };

  const openProcessDialog = (process?: ProcessRecord) => {
    if (process) {
      setEditingProcess(process);
      setProcessForm({
        process_name: process.process_name,
        is_active: process.is_active ?? true,
      });
    } else {
      setEditingProcess(null);
      setProcessForm(emptyProcessForm);
    }
    setProcessDialogOpen(true);
  };

  const saveProcess = async () => {
    if (!canManage) {
      toast.error("No tienes permisos para gestionar procesos.");
      return;
    }

    if (!processForm.process_name.trim()) {
      toast.error("El nombre del proceso es obligatorio.");
      return;
    }

    const payload = {
      process_name: processForm.process_name.trim(),
      is_active: processForm.is_active,
    };

    const response = editingProcess
      ? await supabase.from("sgm_processes").update(payload).eq("id", editingProcess.id)
      : await supabase.from("sgm_processes").insert(payload);

    if (response.error) {
      toast.error("No se pudo guardar el proceso.");
      return;
    }

    toast.success(editingProcess ? "Proceso actualizado." : "Proceso agregado.");
    setProcessDialogOpen(false);
    await loadProcesses();
  };

  const confirmDeleteProcess = async () => {
    if (!processDeleteTarget) return;

    const { error } = await supabase.from("sgm_processes").delete().eq("id", processDeleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el proceso.");
      return;
    }

    toast.success("Proceso eliminado.");
    setProcessDeleteTarget(null);
    await loadProcesses();
  };

  return (
    <SGIDashboardLayout>
      <div className="flex flex-col gap-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Matriz de Recursos de Sistemas de Gestión</h1>
          <p className="text-sm text-muted-foreground">
            Consolida recursos, responsables, evidencias y estados para los sistemas de gestión.
          </p>
        </div>
        <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
          <Card>
            <CardHeader>
              <CardTitle className="text-base font-semibold">Distribución por tipo</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm">
              {typeStats.map((stat) => (
                <div key={stat.value} className="flex items-center justify-between">
                  <span className="text-muted-foreground">{stat.label}</span>
                  <span className="font-semibold">{formatPercent(stat.percent)}</span>
                </div>
              ))}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="text-base font-semibold">Estado de recursos</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm">
              {statusStats.map((stat) => (
                <div key={stat.value} className="flex items-center justify-between">
                  <span className="text-muted-foreground">{stat.label}</span>
                  <span className="font-semibold">{formatPercent(stat.percent)}</span>
                </div>
              ))}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="text-base font-semibold">Cobertura de procesos</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm">
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Procesos con recursos</span>
                <span className="font-semibold">
                  {processesWithResources} / {totalProcesses}
                </span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Cobertura</span>
                <span className="font-semibold">{formatPercent(coveragePercent)}</span>
              </div>
              {processes.length === 0 && canManage && (
                <div className="space-y-2">
                  <Label htmlFor="process-total" className="text-xs text-muted-foreground">
                    Total de procesos (configurable)
                  </Label>
                  <Input
                    id="process-total"
                    type="number"
                    min="0"
                    value={processTotalOverride}
                    onChange={(event) => setProcessTotalOverride(event.target.value)}
                    placeholder="Ej. 12"
                  />
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="text-base font-semibold">Recursos críticos</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm">
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Clave no vigente</span>
                <span className="font-semibold">{criticalNotCurrent}</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">En implementación + legal</span>
                <span className="font-semibold">{implementingLegal}</span>
              </div>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div>
              <CardTitle className="text-lg font-semibold">Matriz de Recursos</CardTitle>
              <p className="text-sm text-muted-foreground">
                Administra los recursos del sistema de gestión, evidencia y responsables asignados.
              </p>
            </div>
            <div className="flex flex-wrap gap-2">
              {canManage && (
                <>
                  <Button onClick={() => openProcessDialog()} variant="outline">
                    Gestionar procesos
                  </Button>
                  <Button onClick={() => openResourceDialog()}>
                    <Plus className="mr-2 h-4 w-4" />
                    Agregar recurso
                  </Button>
                </>
              )}
            </div>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-3 lg:grid-cols-[2fr_repeat(6,1fr)]">
              <div className="relative">
                <Search className="pointer-events-none absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                <Input
                  className="pl-9"
                  placeholder="Buscar recurso, uso, proceso o responsable"
                  value={searchValue}
                  onChange={(event) => setSearchValue(event.target.value)}
                />
              </div>
              <Select value={categoryFilter} onValueChange={setCategoryFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Categoría" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas las categorías</SelectItem>
                  {categoryOptions.map((category) => (
                    <SelectItem key={category} value={category}>
                      {category}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={typeFilter} onValueChange={setTypeFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Tipo" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los tipos</SelectItem>
                  {resourceTypeOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Estado" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los estados</SelectItem>
                  {statusOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={processFilter} onValueChange={setProcessFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Proceso" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los procesos</SelectItem>
                  {processOptions.map((process) => (
                    <SelectItem key={process} value={process}>
                      {process}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={responsibleFilter} onValueChange={setResponsibleFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Responsable" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los responsables</SelectItem>
                  {responsibleOptions.map((responsible) => (
                    <SelectItem key={responsible} value={responsible}>
                      {responsible}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <div className="flex items-center gap-2 text-sm text-muted-foreground">
                <Filter className="h-4 w-4" />
                Filtros avanzados
              </div>
            </div>

            <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-4">
              <Select value={criticalFilter} onValueChange={setCriticalFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Crítico" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Críticos (todos)</SelectItem>
                  <SelectItem value="yes">Solo críticos</SelectItem>
                  <SelectItem value="no">No críticos</SelectItem>
                </SelectContent>
              </Select>
              <Select value={legalFilter} onValueChange={setLegalFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Legal" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Legal (todos)</SelectItem>
                  <SelectItem value="yes">Ligados a legal</SelectItem>
                  <SelectItem value="no">No ligados</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="overflow-hidden rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Categoría</TableHead>
                    <TableHead>Tipo</TableHead>
                    <TableHead>Recurso</TableHead>
                    <TableHead>Uso en el sistema de gestión</TableHead>
                    <TableHead>Proceso / Actividad</TableHead>
                    <TableHead>Responsable</TableHead>
                    <TableHead>Evidencia</TableHead>
                    <TableHead>Estado</TableHead>
                    <TableHead>Criticidad</TableHead>
                    {canManage && <TableHead className="text-right">Acciones</TableHead>}
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {loadingResources ? (
                    <TableRow>
                      <TableCell colSpan={canManage ? 10 : 9} className="text-center text-sm text-muted-foreground">
                        Cargando recursos...
                      </TableCell>
                    </TableRow>
                  ) : filteredResources.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={canManage ? 10 : 9} className="text-center text-sm text-muted-foreground">
                        No hay recursos registrados.
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredResources.map((resource) => {
                      const evidenceLink =
                        resource.evidence_url || (resource.evidence_path ? getPublicEvidenceUrl(resource.evidence_path) : "");
                      return (
                        <TableRow key={resource.id}>
                          <TableCell>{resource.category || "—"}</TableCell>
                          <TableCell>
                            {resourceTypeOptions.find((option) => option.value === resource.resource_type)?.label ??
                              resource.resource_type}
                          </TableCell>
                          <TableCell className="font-medium">{resource.resource_name}</TableCell>
                          <TableCell>{resource.management_system_use}</TableCell>
                          <TableCell>{resource.process_activity || "—"}</TableCell>
                          <TableCell>{resource.responsible || "—"}</TableCell>
                          <TableCell>
                            {evidenceLink ? (
                              <a
                                href={evidenceLink}
                                target="_blank"
                                rel="noreferrer"
                                className="inline-flex items-center gap-2 text-sm text-sky-600 hover:underline"
                              >
                                <FileText className="h-4 w-4" />
                                Ver evidencia
                              </a>
                            ) : (
                              "—"
                            )}
                          </TableCell>
                          <TableCell>
                            <Badge className={statusBadgeClasses[resource.status] ?? "bg-slate-100 text-slate-700"}>
                              {statusOptions.find((option) => option.value === resource.status)?.label ??
                                resource.status}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <div className="flex flex-col text-xs text-muted-foreground">
                              <span>{resource.is_critical ? "Crítico" : "No crítico"}</span>
                              <span>{resource.linked_to_legal ? "Ligado a legal" : "Sin requisito legal"}</span>
                            </div>
                          </TableCell>
                          {canManage && (
                            <TableCell className="text-right">
                              <div className="flex justify-end gap-2">
                                <Button size="icon" variant="ghost" onClick={() => openResourceDialog(resource)}>
                                  <Pencil className="h-4 w-4" />
                                </Button>
                                <Button
                                  size="icon"
                                  variant="ghost"
                                  onClick={() => setResourceDeleteTarget(resource)}
                                >
                                  <Trash2 className="h-4 w-4 text-rose-500" />
                                </Button>
                              </div>
                            </TableCell>
                          )}
                        </TableRow>
                      );
                    })
                  )}
                </TableBody>
              </Table>
            </div>

            <div className="flex flex-wrap items-center gap-3 text-xs text-muted-foreground">
              <Badge variant="outline">{filteredResources.length} recursos visibles</Badge>
              <Badge variant="outline">{totalResources} recursos totales</Badge>
              {loadingProcesses && <span>Cargando procesos...</span>}
            </div>
          </CardContent>
        </Card>
      </div>

      <Dialog open={resourceDialogOpen} onOpenChange={setResourceDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>{editingResource ? "Editar recurso" : "Agregar recurso"}</DialogTitle>
          </DialogHeader>
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label>Categoría</Label>
              <Input
                value={resourceForm.category}
                onChange={(event) => setResourceForm((prev) => ({ ...prev, category: event.target.value }))}
                placeholder="Ej. Ambiental, Calidad, Seguridad"
              />
            </div>
            <div className="space-y-2">
              <Label>Tipo de recurso</Label>
              <Select
                value={resourceForm.resource_type}
                onValueChange={(value) => setResourceForm((prev) => ({ ...prev, resource_type: value }))}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar" />
                </SelectTrigger>
                <SelectContent>
                  {resourceTypeOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Recurso</Label>
              <Input
                value={resourceForm.resource_name}
                onChange={(event) => setResourceForm((prev) => ({ ...prev, resource_name: event.target.value }))}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Uso en el sistema de gestión</Label>
              <Textarea
                value={resourceForm.management_system_use}
                onChange={(event) => setResourceForm((prev) => ({ ...prev, management_system_use: event.target.value }))}
              />
            </div>
            <div className="space-y-2">
              <Label>Proceso / Actividad</Label>
              <Input
                value={resourceForm.process_activity}
                onChange={(event) => setResourceForm((prev) => ({ ...prev, process_activity: event.target.value }))}
                placeholder="Proceso o actividad del SGA"
              />
            </div>
            <div className="space-y-2">
              <Label>Responsable</Label>
              <Input
                value={resourceForm.responsible}
                onChange={(event) => setResourceForm((prev) => ({ ...prev, responsible: event.target.value }))}
              />
            </div>
            <div className="space-y-2">
              <Label>Evidencia (link)</Label>
              <Input
                value={resourceForm.evidence_url}
                onChange={(event) => setResourceForm((prev) => ({ ...prev, evidence_url: event.target.value }))}
                placeholder="https://..."
              />
            </div>
            <div className="space-y-2">
              <Label>Estado</Label>
              <Select
                value={resourceForm.status}
                onValueChange={(value) => setResourceForm((prev) => ({ ...prev, status: value }))}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar" />
                </SelectTrigger>
                <SelectContent>
                  {statusOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Evidencia (archivo)</Label>
              <Input type="file" onChange={(event) => setResourceFile(event.target.files?.[0] ?? null)} />
            </div>
            <div className="space-y-2 md:col-span-2">
              <div className="flex flex-wrap gap-6">
                <Label className="flex items-center gap-2 text-sm font-normal">
                  <Checkbox
                    checked={resourceForm.is_critical}
                    onCheckedChange={(checked) =>
                      setResourceForm((prev) => ({ ...prev, is_critical: checked === true }))
                    }
                  />
                  Recurso crítico / clave
                </Label>
                <Label className="flex items-center gap-2 text-sm font-normal">
                  <Checkbox
                    checked={resourceForm.linked_to_legal}
                    onCheckedChange={(checked) =>
                      setResourceForm((prev) => ({ ...prev, linked_to_legal: checked === true }))
                    }
                  />
                  Ligado a requisito legal
                </Label>
              </div>
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setResourceDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={saveResource}>
              <Upload className="mr-2 h-4 w-4" />
              Guardar
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={processDialogOpen} onOpenChange={setProcessDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Procesos del SGA</DialogTitle>
          </DialogHeader>
          <div className="grid gap-4 md:grid-cols-[2fr,1fr]">
            <div className="space-y-2">
              <Label>Nombre del proceso</Label>
              <Input
                value={processForm.process_name}
                onChange={(event) => setProcessForm((prev) => ({ ...prev, process_name: event.target.value }))}
              />
            </div>
            <div className="flex items-end gap-2 pb-1">
              <Label className="flex items-center gap-2 text-sm font-normal">
                <Checkbox
                  checked={processForm.is_active}
                  onCheckedChange={(checked) =>
                    setProcessForm((prev) => ({ ...prev, is_active: checked === true }))
                  }
                />
                Activo
              </Label>
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setProcessDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={saveProcess}>
              <Plus className="mr-2 h-4 w-4" />
              {editingProcess ? "Actualizar proceso" : "Agregar proceso"}
            </Button>
          </div>

          <div className="rounded-lg border">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Proceso</TableHead>
                  <TableHead>Estado</TableHead>
                  <TableHead className="text-right">Acciones</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {loadingProcesses ? (
                  <TableRow>
                    <TableCell colSpan={3} className="text-center text-sm text-muted-foreground">
                      Cargando procesos...
                    </TableCell>
                  </TableRow>
                ) : processes.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={3} className="text-center text-sm text-muted-foreground">
                      No hay procesos registrados.
                    </TableCell>
                  </TableRow>
                ) : (
                  processes.map((process) => (
                    <TableRow key={process.id}>
                      <TableCell className="font-medium">{process.process_name}</TableCell>
                      <TableCell>
                        <Badge variant="outline">{process.is_active ?? true ? "Activo" : "Inactivo"}</Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-2">
                          <Button size="icon" variant="ghost" onClick={() => openProcessDialog(process)}>
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button size="icon" variant="ghost" onClick={() => setProcessDeleteTarget(process)}>
                            <Trash2 className="h-4 w-4 text-rose-500" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>
        </DialogContent>
      </Dialog>

      <AlertDialog open={Boolean(resourceDeleteTarget)} onOpenChange={() => setResourceDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar este recurso?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDeleteResource}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <AlertDialog open={Boolean(processDeleteTarget)} onOpenChange={() => setProcessDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar este proceso?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDeleteProcess}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}