"use client";

import { useEffect, useState } from 'react';
import { supabase, type Database } from '@/lib/supabase';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { CheckCircle, Clock, AlertCircle, TrendingUp } from 'lucide-react';

type PersonalTask = Database['public']['Tables']['tasks']['Row'];
type TeamTask = Database['public']['Tables']['team_tasks']['Row'];

interface Stats {
  total: number;
  completed: number;
  pending: number;
  overdue: number;
  completionRate: number;  
}

export function StatsCards() {
  const [stats, setStats] = useState<Stats>({
    total: 0,
    completed: 0,
    pending: 0,
    overdue: 0,
    completionRate: 0
  });

  useEffect(() => {
    const fetchStats = async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const [personalTasks, teamTasks] = await Promise.all([
        supabase
          .from('tasks')
          .select('*')
          .eq('user_id', user.id),
        supabase
          .from('team_tasks')
          .select('*')
          .eq('member_id', user.id)
      ]);

      if (personalTasks.error) {
        console.warn('Error fetching personal tasks', personalTasks.error);
      }

      if (teamTasks.error) {
        console.warn('Error fetching team tasks', teamTasks.error);
      }

      const mappedPersonal = (personalTasks.data || []).map((task: PersonalTask) => ({
        completed: task.completed,
        due_date: task.due_date
      }));

      const mappedTeam = (teamTasks.data || []).map((task: TeamTask) => ({
        completed: task.status === 'completed',
        due_date: task.due_date
      }));

      const combined = [...mappedPersonal, ...mappedTeam];

      const now = new Date();
      const total = combined.length;
      const completed = combined.filter((task) => task.completed).length;
      const pending = combined.filter((task) => !task.completed).length;
      const overdue = combined.filter(
        (task) =>
          !task.completed &&
          task.due_date &&
          new Date(task.due_date) < now
      ).length;

      const completionRate = total ? Math.round((completed / total) * 100) : 0;

      setStats({ total, completed, pending, overdue, completionRate });

    };

    fetchStats();
  }, []);

  const cards = [
    {
      title: 'Total de Tareas',
      value: stats.total,
      icon: CheckCircle,
      color: 'text-blue-600',
      bgColor: 'bg-blue-50',
      trend: stats.total ? `${stats.completionRate}% completadas` : 'Sin tareas aún'
    },
    {
      title: 'Completadas',
      value: stats.completed,
      icon: CheckCircle,
      color: 'text-green-600',
      bgColor: 'bg-green-50',
      trend: stats.total ? `${stats.completionRate}% de avance` : 'Agrega tu primera tarea'
    },
    {
      title: 'Pendientes',
      value: stats.pending,
      icon: Clock,
      color: 'text-yellow-600',
      bgColor: 'bg-yellow-50',
      trend: `${Math.max(stats.pending - stats.overdue, 0)} a tiempo`
    },
    {
      title: 'Atrasadas',
      value: stats.overdue,
      icon: AlertCircle,
      color: 'text-red-600',
      bgColor: 'bg-red-50',
      trend: stats.overdue ? 'Prioriza estas tareas' : 'Todo al día'
    }
  ];

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
      {cards.map((card, index) => (
        <Card key={index} className="border-0 shadow-sm">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium text-gray-600">
              {card.title}
            </CardTitle>
            <div className={`p-2 rounded-full ${card.bgColor}`}>
              <card.icon className={`h-4 w-4 ${card.color}`} />
            </div>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-gray-900">{card.value}</div>
            <div className="flex items-center gap-1 text-sm text-gray-500 mt-1">
              <TrendingUp className="h-3 w-3" />
              <span>{card.trend}</span>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}