"use client";

import { useState } from 'react';
import { supabase, type Database } from '@/lib/supabase';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { 
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue 
} from '@/components/ui/select';
import { 
  Dialog, 
  DialogContent, 
  DialogHeader,
  DialogDescription,  
  DialogTitle
} from '@/components/ui/dialog';
import { toast } from 'sonner';

type TaskInsert = Database['public']['Tables']['tasks']['Insert'];

interface TaskFormProps {
  onClose: () => void;
  onSuccess: () => void;
}

const initialFormData = {
  title: '',
  description: '',
  priority: 'medium' as 'low' | 'medium' | 'high',
  due_date: '',
  link_url: '',  
  recurrence_pattern: 'none' as TaskInsert['recurrence_pattern'] | 'none',
  recurrence_interval: '1'
};

export function TaskForm({ onClose, onSuccess }: TaskFormProps) {
  const [formData, setFormData] = useState(initialFormData);
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      toast.error('Usuario no autenticado');
      setLoading(false);
      return;
    }

    if (formData.recurrence_pattern !== 'none' && !formData.due_date) {
      toast.error('Agrega una fecha de vencimiento para las tareas recurrentes');
      setLoading(false);
      return;
    }

    const recurrencePattern = formData.recurrence_pattern === 'none'
      ? null
      : formData.recurrence_pattern;
    const recurrenceInterval = recurrencePattern
      ? Math.max(1, Number(formData.recurrence_interval) || 1)
      : null;

    const { error } = await supabase
      .from('tasks')
      .insert([{
        title: formData.title,
        description: formData.description || null,
        priority: formData.priority,
        due_date: formData.due_date || null,
        link_url: formData.link_url || null,        
        user_id: user.id,
        completed: false,
        recurrence_pattern: recurrencePattern,
        recurrence_interval: recurrenceInterval,
        parent_task_id: null
      }]);

    if (error) {
      toast.error('Error al crear la tarea');
      console.error('Error creating task:', error);
    } else {
      toast.success('Tarea creada correctamente');
      setFormData(initialFormData);
      onSuccess();
    }

    setLoading(false);
  };

  const handleInputChange = (field: string, value: string) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleRecurrencePatternChange = (value: TaskInsert['recurrence_pattern'] | 'none') => {
    setFormData(prev => ({
      ...prev,
      recurrence_pattern: value,
      recurrence_interval: value === 'none' ? '1' : prev.recurrence_interval || '1'
    }));
  };

  return (
    <Dialog open onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[425px]">
        <DialogHeader>
          <DialogTitle>Nueva Tarea</DialogTitle>
          <DialogDescription>
            Ingresa los detalles para crear una nueva tarea y asigna su prioridad.
          </DialogDescription>          
        </DialogHeader>
        
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="title">Título *</Label>
            <Input
              id="title"
              value={formData.title}
              onChange={(e) => handleInputChange('title', e.target.value)}
              placeholder="¿Qué necesitas hacer?"
              required
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="description">Descripción</Label>
            <Textarea
              id="description"
              value={formData.description}
              onChange={(e) => handleInputChange('description', e.target.value)} 
              placeholder="Detalles adicionales (opcional)"
              rows={3}
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="priority">Prioridad</Label>
              <Select 
                value={formData.priority} 
                onValueChange={(value) => handleInputChange('priority', value)}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="low">Baja</SelectItem>
                  <SelectItem value="medium">Media</SelectItem>
                  <SelectItem value="high">Alta</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="link_url">Enlace (opcional)</Label>
            <Input
              id="link_url"
              type="url"
              placeholder="https://..."
              value={formData.link_url}
              onChange={(e) => handleInputChange('link_url', e.target.value)}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="due_date">Fecha de vencimiento</Label>
            <Input
              id="due_date"
              type="date"
              value={formData.due_date}
              onChange={(e) => handleInputChange('due_date', e.target.value)}
              min={new Date().toISOString().split('T')[0]}
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="recurrence_pattern">Recurrencia</Label>
              <Select
                value={formData.recurrence_pattern || 'none'}
                onValueChange={(value) =>
                  handleRecurrencePatternChange(value as TaskInsert['recurrence_pattern'] | 'none')
                }
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona una recurrencia" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="none">Sin recurrencia</SelectItem>
                  <SelectItem value="daily">Diaria</SelectItem>
                  <SelectItem value="weekly">Semanal</SelectItem>
                  <SelectItem value="monthly">Mensual</SelectItem>
                  <SelectItem value="yearly">Anual</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="recurrence_interval">Intervalo</Label>
              <Input
                id="recurrence_interval"
                type="number"
                min={1}
                disabled={formData.recurrence_pattern === 'none'}
                value={formData.recurrence_interval}
                onChange={(e) => handleInputChange('recurrence_interval', e.target.value)}
              />
            </div>
          </div>

          <div className="flex justify-end gap-2 pt-4">
            <Button type="button" variant="outline" onClick={onClose}>
              Cancelar
            </Button>
            <Button type="submit" disabled={loading || !formData.title}>
              {loading ? 'Creando...' : 'Crear Tarea'}
            </Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  );
}