"use client";

import {
  addDays,
  differenceInCalendarDays,
  format,
  isValid,
  parseISO,
  startOfDay,
} from "date-fns";
import { Badge } from "@/components/ui/badge";
import { ScrollArea, ScrollBar } from "@/components/ui/scroll-area";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { cn } from "@/lib/utils";
import { UnifiedTask } from "./types";

type TimelineTask = {
  id: string;
  title: string;
  start: Date;
  end: Date;
  status: UnifiedTask["status"] | "pending";
  priority: UnifiedTask["priority"];
  source: UnifiedTask["source"];
  team_name?: string | null;
  due_date?: string | null;
};

const statusStyles: Record<
  NonNullable<TimelineTask["status"]>,
  { bar: string; text: string; accent: string }
> = {
  pending: {
    bar: "bg-amber-100 border border-amber-200",
    text: "text-amber-800",
    accent: "bg-amber-500",
  },
  in_progress: {
    bar: "bg-blue-100 border border-blue-200",
    text: "text-blue-800",
    accent: "bg-blue-500",
  },
  completed: {
    bar: "bg-emerald-100 border border-emerald-200",
    text: "text-emerald-800",
    accent: "bg-emerald-500",
  },
};

const priorityBadge: Record<TimelineTask["priority"], string> = {
  low: "bg-emerald-100 text-emerald-700 border border-emerald-200",
  medium: "bg-amber-100 text-amber-700 border border-amber-200",
  high: "bg-rose-100 text-rose-700 border border-rose-200",
};

const parseDateSafe = (value?: string | null): Date | null => {
  if (!value) return null;
  const parsed = parseISO(value);
  return isValid(parsed) ? parsed : null;
};

export function TaskGantt({
  tasks,
  loading,
}: {
  tasks: UnifiedTask[];
  loading?: boolean;
}) {
  const timelineTasks: TimelineTask[] = tasks.map((task) => {
    const startDate = startOfDay(parseDateSafe(task.created_at) ?? new Date());
    const dueDate = parseDateSafe(task.due_date);
    const safeDueDate = dueDate && dueDate < startDate ? startDate : dueDate;
    const endDate = startOfDay(safeDueDate ?? addDays(startDate, 1));

    return {
      id: task.id,
      title: task.title,
      start: startDate,
      end: endDate,
      status: task.status ?? (task.completed ? "completed" : "pending"),
      priority: task.priority,
      source: task.source,
      team_name: task.team_name,
      due_date: task.due_date,
    };
  });

  if (loading) {
    return (
      <div className="p-6 text-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto" />
        <p className="mt-2 text-gray-600">Cargando cronograma...</p>
      </div>
    );
  }

  if (timelineTasks.length === 0) {
    return (
      <div className="p-10 text-center space-y-2">
        <p className="text-lg font-semibold text-gray-800">Aún no hay tareas para mostrar</p>
        <p className="text-gray-600">Crea una tarea para ver cómo aparece en el cronograma.</p>
      </div>
    );
  }

  const minStart = startOfDay(
    timelineTasks.reduce(
      (min, task) => (task.start < min ? task.start : min),
      timelineTasks[0]?.start ?? startOfDay(new Date())
    )
  );

  const maxEnd = startOfDay(
    timelineTasks.reduce(
      (max, task) => (task.end > max ? task.end : max),
      timelineTasks[0]?.end ?? startOfDay(new Date())
    )
  );

  const totalRangeDays = Math.max(differenceInCalendarDays(maxEnd, minStart) + 1, 7);
  const adjustedEnd = addDays(minStart, totalRangeDays - 1);

  const totalDays = differenceInCalendarDays(adjustedEnd, minStart) + 1;
  const dayColumns = Array.from({ length: totalDays }, (_, index) => addDays(minStart, index));
  const timelineMinWidth = `${totalDays * 64}px`;
  const timelineGridStyle = {
    gridTemplateColumns: `repeat(${totalDays}, minmax(64px, 1fr))`,
    minWidth: timelineMinWidth,
  };
  const rowGridStyle = {
    gridTemplateColumns: `300px minmax(${timelineMinWidth}, 1fr)`,
  };

  return (
    <div className="space-y-4">
      <div className="flex flex-wrap items-center gap-3 text-sm text-gray-700">
        <div className="flex items-center gap-2">
          <span className="h-2 w-2 rounded-full bg-emerald-500" />
          <span>Completada</span>
        </div>
        <div className="flex items-center gap-2">
          <span className="h-2 w-2 rounded-full bg-blue-500" />
          <span>En progreso</span>
        </div>
        <div className="flex items-center gap-2">
          <span className="h-2 w-2 rounded-full bg-amber-500" />
          <span>Pendiente</span>
        </div>
        <div className="flex items-center gap-2">
          <span className="h-2 w-2 rounded-full bg-slate-400" />
          <span>Personal</span>
        </div>
        <div className="flex items-center gap-2">
          <span className="h-2 w-2 rounded-full bg-purple-500" />
          <span>Equipo</span>
        </div>
      </div>

      <div className="rounded-lg border bg-white shadow-sm">
        <ScrollArea className="h-[70vh] min-h-[560px] w-full">
          <div className="min-w-max">
            <div className="grid border-b bg-slate-50 sticky top-0 z-10" style={rowGridStyle}>
              <div className="p-3 text-sm font-semibold text-gray-800 border-r">Tarea</div>
              <div className="overflow-hidden">
                <div className="grid text-xs text-gray-600" style={timelineGridStyle}>
                  {dayColumns.map((day, index) => (
                    <div
                      key={index}
                      className="px-3 py-2 border-l border-slate-200 first:border-l-0 text-center"
                    >
                      <div className="font-medium">{format(day, "dd MMM")}</div>
                      <div className="text-[11px] text-gray-500">{format(day, "EEE")}</div>
                    </div>
                  ))}
                </div>
              </div>
            </div>

            <div className="divide-y">
              {timelineTasks.map((task) => {
                const offsetDays = Math.max(0, differenceInCalendarDays(task.start, minStart));
                const duration = Math.max(1, differenceInCalendarDays(task.end, task.start) + 1);
                const leftPercent = (offsetDays / totalDays) * 100;
                const widthPercent = (duration / totalDays) * 100;
                const statusStyle = statusStyles[task.status ?? "pending"];

                return (
                  <div key={task.id} className="grid" style={rowGridStyle}>
                    <div className="p-4 border-r space-y-2">
                      <div className="flex items-center gap-2">
                        <span
                          className={cn(
                            "h-2 w-2 rounded-full",
                            task.source === "team" ? "bg-purple-500" : "bg-slate-400"
                          )}
                        />
                        <p className="font-medium text-gray-900 truncate">{task.title}</p>
                      </div>
                      <div className="flex flex-wrap items-center gap-2 text-xs text-gray-600">
                        <Badge variant="outline" className={priorityBadge[task.priority]}>
                          Prioridad: {task.priority === "high" ? "Alta" : task.priority === "medium" ? "Media" : "Baja"}
                        </Badge>
                        <Badge variant="outline" className={statusStyle.bar + " " + statusStyle.text}>
                          {task.status === "completed"
                            ? "Completada"
                            : task.status === "in_progress"
                            ? "En progreso"
                            : "Pendiente"}
                        </Badge>
                        {task.source === "team" && task.team_name ? (
                          <Badge variant="outline" className="bg-purple-50 text-purple-700 border border-purple-200">
                            Equipo: {task.team_name}
                          </Badge>
                        ) : null}
                        {task.due_date ? (
                          <span className="text-xs text-gray-500">
                            Vence: {format(parseISO(task.due_date), "dd MMM")}
                          </span>
                        ) : null}
                      </div>                        
                    </div>
                    <div className="relative h-16 overflow-hidden">
                      <div
                        className="absolute inset-0 grid"
                        style={timelineGridStyle}
                      >
                        {dayColumns.map((_, index) => (
                          <div
                            key={index}
                            className={cn(
                              "border-l border-slate-100 first:border-l-0",
                              index % 7 === 0 ? "bg-slate-50/60" : "bg-white"
                            )}
                          />
                        ))}
                      </div>
                      <div className="absolute inset-0">
                        <TooltipProvider>
                          <Tooltip delayDuration={100}>
                            <TooltipTrigger asChild>
                              <div
                                className={cn(
                                  "absolute top-3 h-10 rounded-md shadow-sm flex items-center px-3 gap-2",
                                  statusStyle.bar
                                )}
                                style={{
                                  left: `${leftPercent}%`,
                                  width: `${widthPercent}%`,
                                  minWidth: "6%",
                                }}
                              >
                                <span className={cn("h-2 w-2 rounded-full", statusStyle.accent)} />
                                <span className={cn("text-xs font-medium truncate", statusStyle.text)}>
                                  {task.title}
                                </span>
                              </div>
                            </TooltipTrigger>
                            <TooltipContent className="text-xs max-w-xs">
                              <div className="font-semibold text-gray-800">{task.title}</div>
                              <div className="text-gray-600">
                                Inicio: {format(task.start, "dd MMM yyyy")}
                              </div>
                              <div className="text-gray-600">
                                Fin estimado: {format(task.end, "dd MMM yyyy")}
                              </div>
                              <div className="text-gray-600 capitalize">Estado: {task.status ?? "pendiente"}</div>
                            </TooltipContent>
                          </Tooltip>
                        </TooltipProvider>
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
          <ScrollBar orientation="vertical" />
          <ScrollBar orientation="horizontal" />     
        </ScrollArea>
      </div>
    </div>
  );
}