"use client";

import { FormEvent, useMemo, useState } from "react";
import { supabase, type Database } from "@/lib/supabase";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { MoreHorizontal, Edit, Trash2, Calendar, Flag, Link2 } from "lucide-react";
import { toast } from "sonner";
import { format } from "date-fns";
import { es } from "date-fns/locale";
import { calculateNextDueDate } from "@/lib/recurrence";
import { UnifiedTask } from "./types";

type PersonalTaskRow = Database["public"]["Tables"]["tasks"]["Row"];

type RecurrenceOption =
  | NonNullable<Database["public"]["Tables"]["tasks"]["Row"]["recurrence_pattern"]>
  | "none";

const priorityColors = {
  low: "bg-green-100 text-green-800",
  medium: "bg-yellow-100 text-yellow-800",
  high: "bg-red-100 text-red-800",
};

const priorityLabels = {
  low: "Baja",
  medium: "Media",
  high: "Alta",
};

const recurrenceLabels: Record<NonNullable<Database["public"]["Tables"]["tasks"]["Row"]["recurrence_pattern"]>, string> =
  {
    daily: "Diaria",
    weekly: "Semanal",
    monthly: "Mensual",
    yearly: "Anual",
  };

interface TaskItemProps {
  task: UnifiedTask;
  onUpdate: (taskId: string, updates: Partial<UnifiedTask>) => void;
  onDelete: (taskId: string) => void;
  onCreate: (task: UnifiedTask) => void;
}

export function TaskItem({ task, onUpdate, onDelete, onCreate }: TaskItemProps) {
  const [loading, setLoading] = useState(false);
  const [isEditing, setIsEditing] = useState(false);
  const [editData, setEditData] = useState({
    title: task.title,
    description: task.description ?? "",
    priority: task.priority,
    due_date: task.due_date ?? "",
    recurrence_pattern: (task.recurrence_pattern ?? "none") as RecurrenceOption,
    recurrence_interval: String(task.recurrence_interval ?? 1),
    link_url: task.link_url ?? "",
  });

  const isTeamTask = task.source === "team";

  const mapPersonalTaskToUnified = (
    personalTask: PersonalTaskRow,
    ownerId?: string | null
  ): UnifiedTask => ({
    id: personalTask.id,
    title: personalTask.title,
    description: personalTask.description,
    priority: personalTask.priority,
    link_url: (personalTask as any).link_url ?? null,
    due_date: personalTask.due_date,
    recurrence_pattern: personalTask.recurrence_pattern,
    recurrence_interval: personalTask.recurrence_interval,
    created_at: personalTask.created_at,
    source: "personal",
    completed: personalTask.completed,
    status: personalTask.completed ? "completed" : "pending",
    team_name: null,
    assigned_by_name: null,
    assigned_by_id: ownerId ?? null,
    assignee_name: null,
    owner_id: ownerId ?? null,
    parent_task_id: personalTask.parent_task_id,
    canEdit: true,
    canDelete: true,
  });

  const handleToggleComplete = async (checked: boolean) => {
    if (isTeamTask && !task.canEdit) {
      toast.error("Solo el creador puede actualizar esta tarea de equipo");
      return;
    }

    const nextCompleted = Boolean(checked);    
    setLoading(true);

    if (isTeamTask) {
      const { data, error } = await supabase
        .from("team_tasks")
        .update({
          status: nextCompleted ? "completed" : "pending",
          link_url: task.link_url,
        })
        .eq("id", task.id)
        .select("id, status")
        .maybeSingle();

      if (error || !data) {
        toast.error("Error al actualizar la tarea");
        setLoading(false);
        return;
      }

      onUpdate(task.id, {
        status: data.status,
        completed: data.status === "completed",
      });
      toast.success(nextCompleted ? "Tarea completada" : "Tarea marcada como pendiente");
      setLoading(false);
      return;
    }
    
    const { data, error } = await supabase
      .from("tasks")
      .update({
        completed: nextCompleted,
        updated_at: new Date().toISOString(),
      })
      .eq("id", task.id)
      .select("id, completed, recurrence_pattern, recurrence_interval, due_date, parent_task_id, title, description, priority, link_url, user_id")
      .maybeSingle();

    if (error || !data) {
      toast.error("Error al actualizar la tarea");
      setLoading(false);
      return;
    }

    onUpdate(task.id, {
      completed: data.completed,
      status: data.completed ? "completed" : "pending",
    });
    toast.success(nextCompleted ? "Tarea completada" : "Tarea marcada como pendiente");

    if (nextCompleted && !task.completed && task.recurrence_pattern) {
      const nextDueDate = calculateNextDueDate(
        data.due_date,
        data.recurrence_pattern as any,
        data.recurrence_interval || undefined
      );

      if (nextDueDate) {
        const parentId = data.parent_task_id || data.id;
        const { data: existingNext } = await supabase
          .from("tasks")
          .select("id")
          .eq("parent_task_id", parentId)
          .eq("due_date", nextDueDate)
          .limit(1);

        if (!existingNext || existingNext.length === 0) {
          const { data: nextTask, error: insertError } = await supabase
            .from("tasks")
            .insert([
              {
                title: data.title,
                description: data.description,
                priority: data.priority,
                due_date: nextDueDate,
                user_id: data.user_id ?? "",
                completed: false,
                recurrence_pattern: data.recurrence_pattern,
                recurrence_interval: data.recurrence_interval,
                parent_task_id: parentId,
                link_url: data.link_url,              
              },
            ])
            .select("*")
            .single();

          if (insertError) {
            toast.error("No se pudo programar la siguiente recurrencia");
          } else if (nextTask) {
            onCreate(mapPersonalTaskToUnified(nextTask as PersonalTaskRow, task.owner_id));
            toast.success("Nueva recurrencia creada");
          }
        }
      }
    }
    
    setLoading(false);
  };

  const handleDelete = async () => {
    if (!confirm("¿Estás seguro de que quieres eliminar esta tarea?")) return;

    if (isTeamTask && !task.canDelete) {
      toast.error("Solo el creador puede eliminar esta tarea de equipo");
      return;
    }

    setLoading(true);

    const { error } = isTeamTask
      ? await supabase.from("team_tasks").delete().eq("id", task.id)
      : await supabase.from("tasks").delete().eq("id", task.id);

    if (error) {
      toast.error("Error al eliminar la tarea");
    } else {
      onDelete(task.id);
      toast.success("Tarea eliminada correctamente");
    }
    
    setLoading(false);
  };

  const isOverdue = task.due_date && new Date(task.due_date) < new Date() && !task.completed;
  const statusLabel = useMemo(() => {
    if (!task.status) return task.completed ? "Completada" : "Pendiente";
    const mapping: Record<"pending" | "in_progress" | "completed", string> = {
      pending: "Pendiente",
      in_progress: "En progreso",
      completed: "Completada",
    };
    return mapping[task.status];
  }, [task.completed, task.status]);

  const handleEditSubmit = async (event: FormEvent) => {
    event.preventDefault();
    if (isTeamTask && !task.canEdit) {
      toast.error("Solo el creador puede editar esta tarea de equipo");
      return;
    }

    setLoading(true);

    if (!isTeamTask && editData.recurrence_pattern !== "none" && !editData.due_date) {
      toast.error("Agrega una fecha de vencimiento para las tareas recurrentes");
      setLoading(false);
      return;
    }

    const recurrencePattern = editData.recurrence_pattern === "none" ? null : editData.recurrence_pattern;
    const recurrenceInterval = recurrencePattern ? Math.max(1, Number(editData.recurrence_interval) || 1) : null;

    if (isTeamTask) {
      const { error } = await supabase
        .from("team_tasks")
        .update({
          title: editData.title,
          description: editData.description || null,
          priority: editData.priority as any,
          due_date: editData.due_date || null,
          link_url: editData.link_url || null,
          status: task.status,
        })
        .eq("id", task.id);

      if (error) {
        toast.error("Error al actualizar la tarea");
        setLoading(false);
        return;
      }

      onUpdate(task.id, {
        title: editData.title,
        description: editData.description || null,
        priority: editData.priority as any,
        due_date: editData.due_date || null,
        link_url: editData.link_url || null,
      });
      toast.success("Tarea actualizada");
      setIsEditing(false);
      setLoading(false);
      return;
    }

    const { error } = await supabase
      .from("tasks")
      .update({
        title: editData.title,
        description: editData.description || null,
        priority: editData.priority as any,
        due_date: editData.due_date || null,
        recurrence_pattern: recurrencePattern,
        recurrence_interval: recurrenceInterval,
        link_url: editData.link_url || null,
        updated_at: new Date().toISOString(),
      })
      .eq("id", task.id);

    if (error) {
      toast.error("Error al actualizar la tarea");
      setLoading(false);
      return;
    }

  onUpdate(task.id, {
    title: editData.title,
    description: editData.description || null,
    priority: editData.priority as any,
    due_date: editData.due_date || null,
    recurrence_pattern: recurrencePattern ?? undefined,
    recurrence_interval: recurrenceInterval ?? undefined,
    link_url: editData.link_url || null,
  });
    toast.success("Tarea actualizada");
    setIsEditing(false);
    setLoading(false);
  };
  
  return (
    <div
      className={`p-4 rounded-lg border transition-all hover:shadow-md ${
        task.completed ? "bg-gray-50 border-gray-200" : isOverdue ? "bg-red-50 border-red-200" : "bg-white border-gray-200"
      }`}
    >
      <div className="flex items-start gap-3">
        <Checkbox
          checked={task.completed}
          onCheckedChange={handleToggleComplete}
          disabled={loading || (isTeamTask && !task.canEdit)}
          className="mt-1"
        />
        
        <div className="flex-1 min-w-0">
          <div className="flex items-start justify-between gap-2">
            <div className="flex-1 min-w-0">
              <h3 className={`font-medium ${task.completed ? "text-gray-500 line-through" : "text-gray-900"}`}>
                {task.title}
              </h3>
              
              {task.description && (
                <p className={`text-sm mt-1 ${task.completed ? "text-gray-400" : "text-gray-600"}`}>{task.description}</p>
              )}
              

              {task.link_url && (
                <a
                  href={task.link_url}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="inline-flex items-center gap-1 text-sm text-blue-700 hover:underline mt-1"
                >
                  <Link2 className="h-3 w-3" />
                  Abrir enlace
                </a>
              )}

              <div className="flex items-center gap-2 mt-2 flex-wrap">
                <Badge variant="secondary" className={priorityColors[task.priority]}>
                  <Flag className="h-3 w-3 mr-1" />
                  {priorityLabels[task.priority]}
                </Badge>
                
                {task.due_date && (
                  <Badge variant="outline" className={isOverdue ? "text-red-600 border-red-300" : ""}>
                    <Calendar className="h-3 w-3 mr-1" />
                    {format(new Date(task.due_date), "dd MMM yyyy", { locale: es })}
                  </Badge>
                )}

                {task.recurrence_pattern && (
                  <Badge variant="outline" className="gap-1">
                    Recurrente: {recurrenceLabels[task.recurrence_pattern]}
                    {task.recurrence_interval && task.recurrence_interval > 1 ? ` (${task.recurrence_interval}x)` : ""}
                  </Badge>
                )}

                {isTeamTask && task.team_name && <Badge variant="outline">Equipo: {task.team_name}</Badge>}

                {task.assigned_by_name && <Badge variant="outline">Creada por: {task.assigned_by_name}</Badge>}

                {task.assignee_name && task.source === "team" && (
                  <Badge variant="outline">Asignada a: {task.assignee_name}</Badge>
                )}

                <Badge
                  variant="outline"
                  className={
                    task.completed
                      ? "text-green-700 border-green-200"
                      : task.status === "in_progress"
                      ? "text-blue-700 border-blue-200"
                      : "text-amber-700 border-amber-200"
                  }
                >
                  {statusLabel}
                </Badge>
              </div>
            </div>
            
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                  <MoreHorizontal className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem
                  onClick={() => (task.canEdit ? setIsEditing(true) : toast.error("Solo el creador puede editar esta tarea"))}
                  className={!task.canEdit ? "opacity-60 cursor-not-allowed" : ""}
                >
                  <Edit className="h-4 w-4 mr-2" />
                  Editar
                </DropdownMenuItem>
                <DropdownMenuItem
                  onClick={task.canDelete ? handleDelete : undefined}
                  className={`text-red-600 focus:text-red-600 ${!task.canDelete ? "opacity-60 cursor-not-allowed" : ""}`}
                >
                  <Trash2 className="h-4 w-4 mr-2" />
                  Eliminar
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>
      </div>

      <Dialog open={isEditing} onOpenChange={setIsEditing}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>Editar tarea</DialogTitle>
            <DialogDescription>
              Actualiza los detalles de la tarea para mantener la información al día.
            </DialogDescription>            
          </DialogHeader>
          <form className="space-y-4" onSubmit={handleEditSubmit}>
            <div className="space-y-2">
              <Label htmlFor="edit-title">Título</Label>
              <Input
                id="edit-title"
                value={editData.title}
                onChange={(event) => setEditData((prev) => ({ ...prev, title: event.target.value }))}
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-description">Descripción</Label>
              <Textarea
                id="edit-description"
                value={editData.description}
                onChange={(event) => setEditData((prev) => ({ ...prev, description: event.target.value }))}
                rows={3}
              />
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Prioridad</Label>
                <Select
                  value={editData.priority}
                  onValueChange={(value: "low" | "medium" | "high") =>
                    setEditData((prev) => ({ ...prev, priority: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="low">Baja</SelectItem>
                    <SelectItem value="medium">Media</SelectItem>
                    <SelectItem value="high">Alta</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="space-y-2">
              <Label>Enlace</Label>
              <Input
                value={editData.link_url}
                onChange={(event) => setEditData((prev) => ({ ...prev, link_url: event.target.value }))}
                placeholder="https://..."
                type="url"
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha de vencimiento</Label>
              <Input
                type="date"
                value={editData.due_date}
                onChange={(event) => setEditData((prev) => ({ ...prev, due_date: event.target.value }))}
              />
            </div>
            {!isTeamTask && (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Recurrencia</Label>
                  <Select
                    value={editData.recurrence_pattern}
                    onValueChange={(value: RecurrenceOption) =>
                      setEditData((prev) => ({
                        ...prev,
                        recurrence_pattern: value,
                        recurrence_interval: value === "none" ? "1" : prev.recurrence_interval,
                      }))
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="none">Sin recurrencia</SelectItem>
                      <SelectItem value="daily">Diaria</SelectItem>
                      <SelectItem value="weekly">Semanal</SelectItem>
                      <SelectItem value="monthly">Mensual</SelectItem>
                      <SelectItem value="yearly">Anual</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>Intervalo</Label>
                  <Input
                    type="number"
                    min={1}
                    disabled={editData.recurrence_pattern === "none"}
                    value={editData.recurrence_interval}
                    onChange={(event) => setEditData((prev) => ({ ...prev, recurrence_interval: event.target.value }))}
                  />
                </div>
              </div>
            )}
            <div className="flex justify-end gap-2 pt-2">
              <Button type="button" variant="outline" onClick={() => setIsEditing(false)}>
                Cancelar
              </Button>
              <Button type="submit" disabled={loading}>
                Guardar cambios
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
}