"use client";

import { useCallback, useEffect, useState } from "react";
import { supabase, type Database } from "@/lib/supabase";
import { UnifiedTask } from "./types";

type TaskRow = Database["public"]["Tables"]["tasks"]["Row"];

type TeamTaskRow = Database["public"]["Tables"]["team_tasks"]["Row"] & {
  assignee?: { full_name?: string | null; email?: string | null } | { full_name?: string | null; email?: string | null }[] | null;
  assigner?: { full_name?: string | null; email?: string | null } | { full_name?: string | null; email?: string | null }[] | null;
  team?: { name?: string | null } | { name?: string | null }[] | null;
};

const mapPersonalTaskToUnified = (task: TaskRow): UnifiedTask => ({
  id: task.id,
  title: task.title,
  description: task.description,
  priority: task.priority,
  link_url: (task as any).link_url ?? null,
  due_date: task.due_date,
  recurrence_pattern: task.recurrence_pattern,
  recurrence_interval: task.recurrence_interval,
  created_at: task.created_at,
  source: "personal",
  completed: task.completed,
  status: task.completed ? "completed" : "pending",
  team_name: null,
  assigned_by_name: null,
  assigned_by_id: task.user_id,
  assignee_name: null,
  owner_id: task.user_id,
  parent_task_id: task.parent_task_id,
  canEdit: true,
  canDelete: true,
});

const mapTeamTaskToUnified = (task: TeamTaskRow, userId: string): UnifiedTask => {
  const assignee = Array.isArray(task.assignee) ? task.assignee[0] ?? null : task.assignee ?? null;
  const assigner = Array.isArray(task.assigner) ? task.assigner[0] ?? null : task.assigner ?? null;
  const team = Array.isArray(task.team) ? task.team[0] ?? null : task.team ?? null;

  return {
    id: task.id,
    title: task.title,
    description: task.description,
    priority: task.priority,
    link_url: (task as any).link_url ?? null,
    due_date: task.due_date,
    recurrence_pattern: task.recurrence_pattern,
    recurrence_interval: task.recurrence_interval,
    created_at: task.created_at,
    source: "team",
    status: task.status ?? "pending",
    completed: (task.status ?? "pending") === "completed",
    team_name: team?.name ?? null,
    assigned_by_name: assigner?.full_name ?? assigner?.email ?? null,
    assigned_by_id: (task as TeamTaskRow).assigned_by,
    assignee_name: assignee?.full_name ?? assignee?.email ?? null,
    owner_id: (task as TeamTaskRow).member_id,
    parent_task_id: null,
    canEdit: (task as TeamTaskRow).assigned_by === userId || (task as TeamTaskRow).member_id === userId,
    canDelete: (task as TeamTaskRow).assigned_by === userId,
  };
};

export function useUnifiedTasks(refreshKey?: number, options?: { skip?: boolean }) {
  const shouldSkip = options?.skip ?? false;
  const [tasks, setTasks] = useState<UnifiedTask[]>([]);
  const [loading, setLoading] = useState(!shouldSkip);

  const fetchTasks = useCallback(async () => {
    if (shouldSkip) {
      return;
    }

    try {
      setLoading(true);
      const {
        data: { user },
      } = await supabase.auth.getUser();

      if (!user) {
        setTasks([]);
        setLoading(false);
        return;
      }

      const [personalTasks, teamTasks] = await Promise.all([
        supabase.from("tasks").select("*").eq("user_id", user.id).order("created_at", { ascending: false }),
        supabase
          .from("team_tasks")
          .select(
            `
            id,
            title,
            description,
            status,
            priority,
            link_url,
            due_date,
            recurrence_pattern,
            recurrence_interval,
            recurrence_end_date,
            member_id,
            assigned_by,
            team_id,
            created_at,
            assignee:member_id(full_name, email),
            assigner:assigned_by(full_name, email),
            team:team_id(name)
          `
          )
          .eq("member_id", user.id)
          .order("created_at", { ascending: false }),
      ]);

      if (personalTasks.error) {
        console.warn("Error fetching personal tasks", personalTasks.error);
      }
      if (teamTasks.error) {
        console.warn("Error fetching team tasks", teamTasks.error);
      }

      const personal = (personalTasks.data || []).map(mapPersonalTaskToUnified);
      const team = (teamTasks.data || []).map((task) => mapTeamTaskToUnified(task as TeamTaskRow, user.id));

      const combined = [...personal, ...team].sort(
        (a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime()
      );

      setTasks(combined);
    } catch (error) {
      console.error("Error fetching tasks", error);
    } finally {
      setLoading(false);
    }
  }, [shouldSkip]);

  useEffect(() => {
    if (shouldSkip) {
      setLoading(false);
      return;
    }

    fetchTasks();
  }, [fetchTasks, refreshKey, shouldSkip]);

  return { tasks, setTasks, loading, refetch: fetchTasks };
}

export default useUnifiedTasks;